package org.planx.msd.list;

import java.util.AbstractList;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.RandomAccess;
import org.planx.msd.*;
import org.planx.msd.util.*;
import org.planx.util.Association;
import org.planx.util.Pair;

/**
 * A <code>Discriminator</code> capable of discriminating a multiset
 * of <code>List</code>s of objects. A <code>Discriminator</code> for the
 * type contained within the lists must be provided in the constructor.
 * This discriminator checks if all lists support random access, and uses
 * a <code>SizeRandomAccessListDiscriminator</code> to perform discrimination.
 * If some lists do not support random access, they are wrapped in
 * <code>IteratorList</code>s that allow the algorithm to run in
 * <i>O(N)</i> time by utilizing the order in which elements are examined.
 * <p>
 * <b>Note that this implementation is not synchronized.</b> If multiple
 * threads access an instance of this class concurrently, it must be
 * synchronized externally.
 *
 * @author Thomas Ambus
 */
public class ListDiscriminator<T> extends AbstractDiscriminator<List<T>> {
    private Discriminator<List<T>> disc;

    /**
     * Creates a new <code>ListDiscriminator</code> able to discriminate
     * a multiset of <code>List</code> objects. Each <code>List</code>
     * must contain objects of type <code>T</code>, where the argument
     * discriminator <code>d</code> is able
     * to discriminate objects of type <code>T</code>.
     * The instance will reuse the memory of the specified
     * <code>Memory</code>.
     */
    public ListDiscriminator(Discriminator<T> d, Memory memory) {
        disc = new SizeRandomAccessListDiscriminator<T>(d, memory);
    }

    public <U,S> Collection<List<S>> discriminate(List<? extends U> values,
                                        Extractor<U,? extends List<T>,S> e) {
        boolean isRandomAccess = true;
        for (U elm : values) {
            if (!(e.getLabel(elm) instanceof RandomAccess)) {
                isRandomAccess = false;
                break;
            }
        }
        if (isRandomAccess) return disc.discriminate(values, e);

        // Some lists are not RandomAccess, wrap all in IteratorList

        List<Pair<List<T>,S>> input = new ArrayList<Pair<List<T>,S>>(values.size());
        for (U elm : values) {
            List<T> list = new IteratorList<T>(e.getLabel(elm));
            input.add(new Association<List<T>,S>(list, e.getValue(elm)));
        }
        Extractor<Pair<List<T>,S>,List<T>,S> ext = Discriminators.pairExtractor();
        return disc.discriminate(input, ext);
    }

    /**
     * Allows the algorithm to run in O(N) time even for lists that do not support
     * random access. This relies on the fact that the algorithm first discriminates
     * according to element 0, then element 1, etc. In other words, iteration order.
     * When calling the <code>get</code> method of an IteratorList, an internal
     * <code>Iterator</code> is first checked to see if it can fetch the element in
     * guaranteed constant time (i.e. the element is the next element).
     */
    private static class IteratorList<E> extends AbstractList<E> {
        private List<E> list;
        private Iterator<E> iterator = null;
        private int itIndex = -1;
        private E current = null;

        IteratorList(List<E> list) {
            this.list = list;
        }
        public List<E> getList() {
            return list;
        }
        public int size() {
            return list.size();
        }
        public E get(int index) {
            if (index == 0) {
                iterator = list.iterator();
                itIndex = 0;
                current = iterator.next();
                return current;
            }
            if (index == itIndex+1 && iterator.hasNext()) {
                itIndex++;
                current = iterator.next();
                return current;
            }
            if (index == itIndex) return current;
            return list.get(index);
        }
    }
}
