package org.planx.msd.list;

import java.util.AbstractList;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import org.planx.msd.*;
import org.planx.msd.number.IntExtractor;
import org.planx.msd.number.IntegerDiscriminator;
import org.planx.msd.util.*;
import org.planx.util.Array;
import org.planx.util.Association;
import org.planx.util.Pair;
import org.planx.util.Pairs;

/**
 * A <code>Discriminator</code> capable of discriminating a multiset
 * of <code>List</code>s of objects. The discriminator is implemented
 * using iterators stepping through the lists, thus achieving O(N)
 * performing for non-random access lists.
 * <p>
 * <b>Note that this implementation is not synchronized.</b> If multiple
 * threads access an instance of this class concurrently, it must be
 * synchronized externally.
 *
 * @author Thomas Ambus
 */
public class IterationListDiscriminator<T> extends AbstractDiscriminator<List<T>> {
    private static final int INIT_CAPACITY = 10;
    private Discriminator<T> disc;
    private IntegerDiscriminator intDisc;

    /**
     * Creates a new <code>IterationListDiscriminator</code> able to
     * discriminate a multiset of <code>List</code> objects. Each
     * <code>List</code> must contain objects of type <code>T</code>,
     * where the argument discriminator <code>d</code> is able
     * to discriminate objects of type <code>T</code>.
     * The instance will reuse the memory of the specified <code>Memory</code>.
     */
    public IterationListDiscriminator(Discriminator<T> d, Memory memory) {
        disc = d;
        intDisc = new IntegerDiscriminator(memory);
    }

    public <U,S> Collection<List<S>> discriminate(List<? extends U> values,
                                  final Extractor<U,? extends List<T>,S> e) {
        if (values.isEmpty()) return Collections.emptyList();

        Collection<List<S>> result = new ArrayList<List<S>>();
        Collection<List<Pair<CurrentIterator<T>,S>>> partition;
        Extractor<Pair<CurrentIterator<T>,S>,T,Pair<CurrentIterator<T>,S>> ext =
            new CurrentExtractor<T,S>();

        List[] work = new List[INIT_CAPACITY];
        int work_capacity = INIT_CAPACITY;
        int work_head = 0;

        // Perform initial partitioning based on list size

        IntExtractor<U,Pair<CurrentIterator<T>,S>> intExt =
            new IntExtractor<U,Pair<CurrentIterator<T>,S>>() {
                public int getLabel(U elm) {
                    return e.getLabel(elm).size();
                }
                public Pair<CurrentIterator<T>,S> getValue(U elm) {
                    List<T> list = e.getLabel(elm);
                    CurrentIterator<T> it = new CurrentIterator<T>(
                        list.iterator());
                    return new Association<CurrentIterator<T>,S>(it,
                        e.getValue(elm));
                }
            };
        partition = intDisc.discriminate(values, intExt);

        // Add all size-equivalent blocks to unfinished work

        for (List<Pair<CurrentIterator<T>,S>> block : partition) {
            // If the block only contains one element it cannot be
            // partitioned any further
            if (block.size() > 1) {
                if (work_head+1 >= work_capacity) {
                    work = Array.ensureCapacity(
                        work, work_head, work_capacity+1);
                    work_capacity = work.length;
                }
                work[work_head] = block;
                work_head++;
            } else {
                result.add(Pairs.retainSecond(block));
            }
        }

        // Refine until there are no unfinished blocks

        while (work_head > 0) {
            if (work_head < work_capacity) work[work_head] = null;
            work_head--;

            // Subpartition current unfinished block
            // it is guaranteed that all lists in the block have
            // the same size

            List<Pair<CurrentIterator<T>,S>> block = work[work_head];
            int blockSize = block.size();

            if (block.get(0).getFirst().hasNext()) {

                // End not reached, partition further
                // Step to next element in all iterators

                for (Pair<CurrentIterator<T>,S> pair : block) {
                    pair.getFirst().next();
                }
                partition = disc.discriminate(block, ext);

                // Add blocks in partition to result or unfinished

                for (List<Pair<CurrentIterator<T>,S>> subBlock : partition) {

                    // If the block only contains one element it
                    // cannot be partitioned any further

                    if (subBlock.size() > 1) {
                        if (work_head+1 >= work_capacity) {
                            work = Array.ensureCapacity(
                                work, work_head, work_capacity+1);
                            work_capacity = work.length;
                        }
                        work[work_head] = subBlock;
                        work_head++;
                    } else {
                        result.add(Pairs.retainSecond(subBlock));
                    }
                }
            } else {
                // End of all lists in block reached, i.e. they are
                // equivalent
                result.add(Pairs.retainSecond(block));
            }
        }
        return result;
    }

    private static class CurrentIterator<E> implements Iterator<E> {
        Iterator<E> it;
        E current = null;

        public CurrentIterator(Iterator<E> it) {
            this.it = it;
        }
        public boolean hasNext() {
            return it.hasNext();
        }
        public E next() {
            current = it.next();
            return current;
        }
        public E current() {
            return current;
        }
        public void remove() {
            throw new UnsupportedOperationException();
        }
    }

    private static class CurrentExtractor<E,F> implements
                        Extractor<Pair<CurrentIterator<E>,F>,E,
                                    Pair<CurrentIterator<E>,F>> {
        public E getLabel(Pair<CurrentIterator<E>,F> elm) {
            return elm.getFirst().current;
        }
        public Pair<CurrentIterator<E>,F> getValue(
                        Pair<CurrentIterator<E>,F> elm) {
            return elm;
        }
    }
}
