package org.planx.msd.lang;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import org.planx.msd.*;
import org.planx.msd.util.*;

/**
 * A <code>Discriminator</code> capable of discriminating objects of
 * different types. Discriminators for types to be handled are added
 * using {@link #addDiscriminator}.
 * When discriminating a list of input values, the values are first
 * separated by type. That is, by the specific <code>Class</code> they
 * are instance of. This is done using a <code>ClassDiscriminator</code>.
 * Next, objects of each type are discriminated by looking up a
 * discriminator for the type based on the class. Note, that a discriminator
 * is associated with a specific <code>Class</code> and not subclasses.
 *
 * @author Thomas Ambus
 */
public class PolymorphicDiscriminator<T> extends AbstractDiscriminator<T> {
    private List<Entry<? extends T>> ds;

    public PolymorphicDiscriminator() {
        this.ds = new ArrayList<Entry<? extends T>>();
    }

    public <E extends T> void addDiscriminator(Class<E> cls,
                                Discriminator<? super E> d) {
        ds.add(new Entry<E>(cls, d));
    }

    /**
     * @throws NonDiscriminableObjectException if an object was
     *         encountered for which
     *         no discriminator was registered
     */
    public <U,S> Collection<List<S>> discriminate(List<? extends U> values,
                                              Extractor<U,? extends T,S> e) {
        if (values.isEmpty()) return Collections.emptyList();

        // Partition input according to class

        int size = ds.size();
        List[] svs = new List[size];

        for (U elm : values) {
            int i = 0;
            for (Entry<? extends T> entry : ds) {
                if (entry.cls.isInstance(e.getLabel(elm))) {
                    // label is instanceof the current class, so
                    // discriminator found
                    List<U> vs = svs[i];
                    if (vs == null) {
                        vs = new ArrayList<U>();
                        svs[i] = vs;
                    }
                    vs.add(elm);
                    break;
                }
                i++;
            }
            if (i >= size) throw new NonDiscriminableObjectException
                ("No Discriminator registered for "+
                    e.getLabel(elm).getClass());
        }

        // Discriminate each block in the partition separately

        Collection<List<S>> result = new ArrayList<List<S>>();

        int j = 0;
        for (Entry<? extends T> entry : ds) {
            List<U> vs = svs[j++];
            if (vs != null) {
                Collection<List<S>> es = entry.d.discriminate(vs,
                                        (Extractor) e);
                result.addAll(es);
            }
        }

        return result;
    }

    private static class Entry<E> {
        Class<E> cls;
        Discriminator<? super E> d;

        public Entry(Class<E> cls, Discriminator<? super E> d) {
            this.cls = cls;
            this.d = d;
        }
    }
}
