package org.planx.msd.array;

import java.util.AbstractList;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.RandomAccess;
import org.planx.msd.*;
import org.planx.msd.util.*;
import org.planx.util.Array;

/**
 * A <code>Discriminator</code> capable of discriminating a multiset of
 * arrays of <code>byte</code>s.
 * <p>
 * <b>Note that this implementation is not synchronized.</b> If multiple
 * threads access an instance of this class concurrently, it must be
 * synchronized externally.
 *
 * @author Thomas Ambus
 */
public class ByteArrayDiscriminator extends AbstractDiscriminator<byte[]> {
    private static final int INIT_CAPACITY = 10;
    private Memory memory;

    /**
     * Constructs a new <code>ByteArrayDiscriminator</code> reusing
     * the memory allocated in the specified <code>Memory</code>.
     */
    public ByteArrayDiscriminator(Memory memory) {
        this.memory = memory;
    }

    public <U,S> Collection<List<S>> discriminate(List<? extends U> values,
                                         Extractor<U,? extends byte[],S> e) {

        // Check fast way out: only zero, one, or two elements

        int vsize = values.size();
        switch (vsize) {
        case 0:
            return Collections.emptyList();
        case 1:
            List<S> l = Collections.singletonList(
                    e.getValue(values.get(0)));
            return Collections.singletonList(l);
        case 2:
            U u1 = values.get(0);
            U u2 = values.get(1);
            byte[] i1 = e.getLabel(u1);
            byte[] i2 = e.getLabel(u2);
            if (Arrays.equals(i1,i2)) {
                List<S> l1 = new ArrayList<S>(2);
                l1.add(e.getValue(u1));
                l1.add(e.getValue(u2));
                return Collections.singletonList(l1);
            } else {
                Collection<List<S>> r = new ArrayList<List<S>>(2);
                List<S> l1 = Collections.singletonList(e.getValue(u1));
                List<S> l2 = Collections.singletonList(e.getValue(u2));
                r.add(l1);
                r.add(l2);
                return r;
            }
        }
        if (!(values instanceof RandomAccess)) values =
            new ArrayList<U>(values);

        // Declarations

        List[] dictionary = memory.dictionary;
        int[] used = memory.used;
        int used_size = 0;

        Collection<List<S>> result = new ArrayList<List<S>>();
        List<S> finished = new ArrayList<S>();

        int[] indexes = new int[INIT_CAPACITY]; // Note: Initialized to 0
        List[] work = new List[INIT_CAPACITY];
        int work_capacity = INIT_CAPACITY;
        int work_head = 1;
        work[0] = values;                  // Start with one big block

        // Refine until there are no unfinished blocks

        while (work_head > 0) {

            if (work_head < work_capacity) work[work_head] = null;
            work_head--;

            List<U> block = work[work_head];
            int blockSize = block.size();
            int initSubSize = (blockSize < 10) ? blockSize : 10;
            int index = indexes[work_head];

            // Subpartition current unfinished block

            used_size = 0;
            for (int i=0; i<blockSize; i++) {
                U elm = block.get(i);
                byte[] seq = e.getLabel(elm);

                if (seq.length <= index) {
                    // No more elms, array is finished
                    finished.add(e.getValue(elm));
                } else {
                    int c = seq[index];

                    List<U> list = dictionary[c];
                    if (list == null) {
                        list = new ArrayList<U>(initSubSize);
                        dictionary[c] = list;
                        used[used_size++] = c;
                    }
                    list.add(elm);
                }
            }

            // If the end of any arrays were reached, add
            // these to result

            if (!finished.isEmpty()) {
                result.add(finished);
                finished = new ArrayList<S>();
            }

            // Add blocks in partition to result or unfinished

            index++;
            for (int i=0; i<used_size; i++) {
                int idx = used[i];
                List<U> subBlock = dictionary[idx];
                dictionary[idx] = null;

                // If the block only contains one element it
                // cannot be partitioned any further

                if (subBlock.size() > 1) {
                    if (work_head+1 >= work_capacity) {
                        work = Array.ensureCapacity(
                                work, work_head, work_capacity+1);
                        indexes = Array.ensureCapacity(
                                indexes, work_head, work_capacity+1);
                        work_capacity = work.length;
                    }
                    work[work_head] = subBlock;
                    indexes[work_head] = index;
                    work_head++;
                } else {
                    result.add(Discriminators.valueList(subBlock, e));
                }
            }
        }
        return result;
    }
}
