/*
 * #%L
 * Nuiton Web :: Nuiton Web
 * 
 * $Id: WinstoneLauncher.java 48 2011-01-26 16:07:53Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/nuiton-web/tags/nuiton-web-1.2/nuiton-web/src/main/java/org/nuiton/web/war/WinstoneLauncher.java $
 * %%
 * Copyright (C) 2011 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.web.war;

import winstone.Launcher;

import javax.swing.ImageIcon;
import java.awt.AWTException;
import java.awt.Desktop;
import java.awt.Image;
import java.awt.MenuItem;
import java.awt.PopupMenu;
import java.awt.SystemTray;
import java.awt.TrayIcon;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.StringWriter;
import java.net.JarURLConnection;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Map;

/**
 * War main class launcher (winstone based).
 * 
 * Based on winstone micro container.
 * 
 * To use it :
 *  java -jar app-xxx.war
 * 
 * @author chatellier
 * @version $Id: WinstoneLauncher.java 48 2011-01-26 16:07:53Z tchemit $
 * @since 1.0
 */
public class WinstoneLauncher implements ActionListener, MouseListener {

    /** Winstone server instance. */
    protected Launcher winstone;

    /** Server URI. */
    protected URI serverUri;

    /** Server name. */
    protected String serverName;

    /**
     * Main method (used by war in manifest).
     * 
     * @param args args
     * @throws IOException
     */
    public static void main(String[] args) throws IOException {
        WinstoneLauncher launcher = new WinstoneLauncher();
        launcher.readInformation();
        launcher.startServer(args);
        launcher.installSystemTray();
        launcher.openBrowser();
    }

    /**
     * Parse WEB-INF/web.xml file and get server display name.
     * 
     * @since 1.1.3
     */
    protected void readInformation() {
        InputStream stream = WinstoneLauncher.class.getResourceAsStream("/WEB-INF/web.xml");
        
        if (stream != null) {
            String content = readAsString(stream);
            if (content != null) {
                int first = content.indexOf("<display-name>");
                if (first >= 0) {
                    serverName = content.substring(first + 14, content.indexOf("</display-name>"));
                    System.out.println("Using server name : " + serverName);
                }
            }
        }

        // if none read, set default
        if (serverName == null || serverName.isEmpty()) {
            serverName = "Server";
        }
    }

    /**
     * Read input stream as string.
     * 
     * Code from commons io.
     * 
     * @param stream stream to read
     * @return content as string
     * @since 1.1.3
     */
    protected String readAsString(InputStream stream) {
        InputStreamReader reader = new InputStreamReader(stream);
        StringWriter sw = new StringWriter();
        char[] buffer = new char[4096];
        int n = 0;
        try {
            while (-1 != (n = reader.read(buffer))) {
                sw.write(buffer, 0, n);
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        return sw.toString();
    }

    /**
     * Launch servlet container.
     * 
     * @param args args
     * @throws IOException
     */
    protected void startServer(String[] args) throws IOException {
        System.out.println("Starting server embedded mode...");

        String fqnLauncherFile = WinstoneLauncher.class.getName().replaceAll("\\.", "/") + ".class";
        System.out.println("Search for launcher class : " + fqnLauncherFile);

        URL classFile = WinstoneLauncher.class.getClassLoader().getResource(fqnLauncherFile);
        System.out.println(" - using classFile : " + classFile);

        // strange following line seams also work for jnlp launch
        File me = new File(((JarURLConnection) classFile.openConnection()).getJarFile().getName());
        System.out.println(" - using warfile file : " + me);

        // hashArgs map, initialized with command line args
        Map<String, String> hashArgs = Launcher.getArgsFromCommandLine(args);

        hashArgs.put("warfile", me.getAbsolutePath()); // or any other command line args, eg port

        System.out.println(" - using args : " + hashArgs);

        Launcher.initLogger(hashArgs);
        winstone = new Launcher(hashArgs); // spawns threads, so your application doesn't block

        // open browser
        int port = 8080;
        String configPort = hashArgs.get("httpPort");
        if (configPort != null && !configPort.isEmpty()) {
            try {
                port = Integer.parseInt(configPort);
            } catch (NumberFormatException e) {
                e.printStackTrace();
            }
        }

        // build server uri
        try {
            serverUri = new URI("http://localhost:" + port);
        } catch (URISyntaxException e) {
            e.printStackTrace();
        }
    }

    /**
     * Shutdown server.
     */
    protected void stopServer() {
        if (winstone != null) {
            winstone.shutdown();
            System.exit(0);
        }
    }

    /**
     * Install system tray to stop server.
     */
    protected void installSystemTray() {
        if (SystemTray.isSupported()) {
            // build menu
            PopupMenu menu = new PopupMenu();
            MenuItem browserItem = new MenuItem("Start browser");
            browserItem.addActionListener(this);
            browserItem.setActionCommand("browser");
            menu.add(browserItem);

            MenuItem stopItem = new MenuItem("Stop server");
            stopItem.addActionListener(this);
            stopItem.setActionCommand("stop");
            menu.add(stopItem);

            // build tray icon
            URL imageURL = WinstoneLauncher.class.getResource("/favicon.png");
            if (imageURL == null) {
                imageURL = WinstoneLauncher.class.getResource("/favicon.jpg");
            }
            if (imageURL == null) {
                System.out.println("No favicon.{png|jpg} found, skip systray installation");
            } else {
                Image image = new ImageIcon(imageURL).getImage();
                TrayIcon icon = new TrayIcon(image, serverName, menu);
                icon.setImageAutoSize(true);
                icon.addMouseListener(this);

                // System tray
                SystemTray systemTray = SystemTray.getSystemTray();
                try {
                    systemTray.add(icon);
                } catch (AWTException ex) {
                    throw new RuntimeException("Can't install tray icon", ex);
                }
            }
        }
    }

    /*
     * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
     */
    @Override
    public void actionPerformed(ActionEvent e) {
        if ("browser".equalsIgnoreCase(e.getActionCommand())) {
            openBrowser();
        } else if ("stop".equalsIgnoreCase(e.getActionCommand())) {
            stopServer();
        }
    }

    /**
     * Open browser.
     * 
     * @throws IOException 
     */
    protected void openBrowser() {
        if (Desktop.isDesktopSupported() && serverUri != null) {
            Desktop desktop = Desktop.getDesktop();
            if (desktop.isSupported(Desktop.Action.BROWSE)) {
                System.out.println("Opening browser at " + serverUri);
                try {
                    desktop.browse(serverUri);
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
    }

    @Override
    public void mouseClicked(MouseEvent e) {
        if (e.getClickCount() == 2) {
            openBrowser();
        }
    }

    @Override
    public void mousePressed(MouseEvent e) {

    }

    @Override
    public void mouseReleased(MouseEvent e) {

    }

    @Override
    public void mouseEntered(MouseEvent e) {

    }

    @Override
    public void mouseExited(MouseEvent e) {

    }
}
