/*
 * #%L
 * ToPIA :: Persistence
 * 
 * $Id: TopiaConnectionProvider.java 2332 2011-09-13 10:09:10Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/topia/tags/topia-2.6.7/topia-persistence/src/main/java/org/nuiton/topia/framework/TopiaConnectionProvider.java $
 * %%
 * Copyright (C) 2004 - 2011 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.topia.framework;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.HibernateException;
import org.hibernate.cfg.Environment;
import org.hibernate.connection.ConnectionProvider;
import org.hibernate.connection.ConnectionProviderFactory;
import org.hibernate.util.PropertiesHelper;
import org.hibernate.util.ReflectHelper;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

/**
 * Customized connection provider.
 * <p/>
 * This provider fix the following bug :
 * http://nuiton.org/issues/show/561
 * <p/>
 * To use this connection provider, add this property to topia configuration
 * <p/>
 * <pre>
 * config.setProperty(Environment.CONNECTION_PROVIDER, TopiaConnectionProvider.class.getName());
 * </pre>
 * <p/>
 * or in a properties file :
 * <p/>
 * <pre>
 * hibernate.connection.provider_class=org.nuiton.topia.framework.TopiaConnectionProvider
 * </pre>
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 2.5.3
 */
public class TopiaConnectionProvider implements ConnectionProvider {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(TopiaConnectionProvider.class);

    /**
     * JDBC url of connection.
     * <p/>
     * This is a mandatory hibernate configuration vi the property
     * {@link Environment#URL}.
     */
    private String url;

    /** All grabbed connection properties */
    private Properties connectionProps;

    /**
     * Sql isolation level to use in connection.
     * <p/>
     * Can be configured by hibernate property {@link Environment#ISOLATION_LEVELS}.
     *
     * @see Connection#getTransactionIsolation()
     */
    private Integer isolation;

    /**
     * auto commit connection state.
     * <p/>
     * Can be configured by hibernate property {@link Environment#AUTOCOMMIT}.
     *
     * @see Connection#getAutoCommit()
     */
    private boolean autocommit;

    /**
     * Size of connection pool.
     * <p/>
     * By default use {@code 20}, can be specify by using the hibernate
     * configuration property {@link Environment#POOL_SIZE}.
     */
    private int poolSize;

    /** Our pool of connections which are not closed and availables. */
    private final List<Connection> pool;

    public TopiaConnectionProvider() {
        pool = new ArrayList<Connection>();
    }

    @Override
    public void configure(Properties props) throws HibernateException {
        String driverClass = props.getProperty(Environment.DRIVER);

        poolSize = PropertiesHelper.getInt(Environment.POOL_SIZE, props, 20); //default pool size 20
        if (log.isDebugEnabled()) {
            log.debug("Connection pool size: " + poolSize);
        }

        autocommit = PropertiesHelper.getBoolean(Environment.AUTOCOMMIT, props);
        if (log.isDebugEnabled())
            log.debug("autocommit mode: " + autocommit);

        isolation = PropertiesHelper.getInteger(Environment.ISOLATION, props);
        if (isolation != null) {
            if (log.isDebugEnabled()) {
                log.debug("JDBC isolation level: " +
                         Environment.isolationLevelToString(isolation));
            }
        }

        if (driverClass == null) {

            if (log.isWarnEnabled()) {
                log.warn("no JDBC Driver class was specified by property " +
                         Environment.DRIVER);
            }
        } else {
            try {
                // trying via forName() first to be as close to DriverManager's semantics
                Class.forName(driverClass);
            } catch (ClassNotFoundException cnfe) {
                try {
                    ReflectHelper.classForName(driverClass);
                } catch (ClassNotFoundException e) {
                    String msg = "JDBC Driver class not found: " + driverClass;
                    log.error(msg, e);
                    throw new HibernateException(msg, e);
                }
            }
        }

        url = props.getProperty(Environment.URL);
        if (url == null) {
            String msg = "JDBC URL was not specified by property " +
                         Environment.URL;
            if (log.isErrorEnabled()) {
                log.error(msg);
            }
            throw new HibernateException(msg);
        }

        connectionProps =
                ConnectionProviderFactory.getConnectionProperties(props);

        if (log.isDebugEnabled()) {
            log.debug("using driver: " + driverClass + " at URL: " + url);
        }
        // if debug level is enabled, then log the password, otherwise mask it
        if (log.isTraceEnabled()) {
            log.debug("connection properties: " + connectionProps);
        } else if (log.isDebugEnabled()) {
            log.debug("connection properties: " +
                     PropertiesHelper.maskOut(connectionProps, "password"));
        }
    }

    @Override
    public Connection getConnection() throws SQLException {

        Connection connection = null;

        synchronized (pool) {

            // try to use a connection from the pool (if any)

            while (!pool.isEmpty() && connection == null) {
                int last = pool.size() - 1;
                if (log.isTraceEnabled()) {
                    log.trace("using pooled JDBC connection, pool size: " +
                              last);
                }

                connection = pool.remove(last);
                if (connection.isClosed()) {

                    // this connection is closed!, don't use it
                    connection = null;

                    if (log.isDebugEnabled()) {
                        log.debug("Remove already closed connection from pool " +
                                  connection);
                    }
                }
            }
        }

        if (connection == null) {

            // the pool was empty, creates a new connection

            if (log.isDebugEnabled()) {
                log.debug("opening new JDBC connection to " + url);
            }
            connection = DriverManager.getConnection(url, connectionProps);
        }

        // configure connection

        if (isolation != null) {
            connection.setTransactionIsolation(isolation);
        }
        if (connection.getAutoCommit() != autocommit) {
            connection.setAutoCommit(autocommit);
        }

        return connection;
    }

    @Override
    public void closeConnection(Connection conn) throws SQLException {

        // if connection is already closed, nothing has to be done
        // we can't keep this connection (and can not be push in pool)

        if (conn.isClosed()) {

            if (log.isDebugEnabled()) {
                log.debug("Connection [" + conn +
                          "] alreay closed!, will not use it any longer ");
            }
            return;
        }

        // connection was not closed, can push it in the pool (if pool is not
        // full)

        synchronized (pool) {
            int currentSize = pool.size();
            if (currentSize < getPoolSize()) {
                if (log.isTraceEnabled()) {
                    log.trace("returning connection to pool, pool size: " +
                              (currentSize + 1));
                }
                pool.add(conn);
                return;
            }
        }

        // pool was full, must release the connection which will be loose

        if (log.isDebugEnabled()) {
            log.debug("closing JDBC connection");
        }

        conn.close();
    }

    @Override
    protected void finalize() throws Throwable {
        super.finalize();
        close();
    }

    @Override
    public void close() {

        if (log.isDebugEnabled()) {
            log.debug("cleaning up connection pool: " + url);
        }

        for (Connection connection : pool) {
            try {
                connection.close();
            } catch (SQLException sqle) {
                if (log.isWarnEnabled()) {
                    log.warn("problem closing pooled connection", sqle);
                }
            }
        }
        pool.clear();

    }

    @Override
    public boolean supportsAggressiveRelease() {
        return false;
    }

    public String getUrl() {
        return url;
    }

    public Properties getConnectionProps() {
        return connectionProps;
    }

    public Integer getIsolation() {
        return isolation;
    }

    public List<Connection> getPool() {
        return pool;
    }

    public int getPoolSize() {
        return poolSize;
    }

    public boolean isAutocommit() {
        return autocommit;
    }

}
