package org.nuiton.spgeed;

/*-
 * #%L
 * spgeed
 * %%
 * Copyright (C) 2017 - 2019 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import org.apache.commons.lang3.ClassUtils;

import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;

public class SpgeedUtils {

    /**
     * Retourne le type qui doit etre retourné, c-a-d que si l'utilisateur
     * demande:
     * - MyObject[] on retourne MyObject
     * - Collection&lt;MyObject&gt; on retourne MyObject
     * - Chunk&lt;MyObject&gt; on retourne MyObject
     *
     * @param c
     * @return
     */
    public static Optional<Class> getElementType(Type c) {
        Class result = null;

        if (c instanceof ParameterizedType) {
            ParameterizedType pType = (ParameterizedType)c;
            Type raw = pType.getRawType();
            if (raw instanceof Class && Collection.class.isAssignableFrom((Class)raw)) {
                Type tmp = pType.getActualTypeArguments()[0];
                if (tmp instanceof Class) {
                    result = (Class) tmp;
                }
            }
        } else if (c instanceof Class && ((Class)c).isArray()) {
            result = ((Class)c).getComponentType();
        }
        return Optional.ofNullable(result);
    }

    public static Collection newCollectionInstance(Class returnType) {
        if (!Collection.class.isAssignableFrom(returnType)) {
            throw new IllegalArgumentException("Class is not a collection: " + returnType);
        }

        Collection result;
        if(Modifier.isAbstract( returnType.getModifiers() ) || returnType.isInterface()) {
            if (returnType.isAssignableFrom(Set.class)) {
                result = new HashSet();
            } else if (returnType.isAssignableFrom(List.class) || returnType.isAssignableFrom(Collection.class)) {
                result = new ArrayList();
            } else {
                throw new IllegalArgumentException("Can't find instanciable class for :" + returnType);
            }
        } else {
            try {
                result = (Collection)returnType.newInstance();
            } catch (InstantiationException | IllegalAccessException eee) {
                throw new IllegalArgumentException("Can't create new instance of class :" + returnType);
            }
        }
        return result;
    }

    public static Object convertToNumber(Object v, Class elementType) {
        try {
            if (v == null || elementType.isInstance(v)) {
                return v;
            } else {
                Method m = Number.class.getMethod(elementType.getSimpleName().toLowerCase().replace("integer", "int") + "Value");
                return m.invoke(v);
            }
        } catch (Exception eee) {
            throw new RuntimeException(String.format("Can't convert '%s' to '%s'", v, elementType.getName()), eee);
        }
    }

    public static boolean returningChunk(Class returnType) {
        boolean result = Chunk.class.isAssignableFrom(returnType);
        return result;
    }

    public static boolean returningCollection(Class returnType) {
        boolean result = Collection.class.isAssignableFrom(returnType);
        return result;
    }

    public static boolean returningArray(Class returnType) {
        boolean result = returnType.isArray();
        return result;
    }

    public static boolean returningPrimitive(Class elementType) {
        boolean result = ClassUtils.isPrimitiveOrWrapper(elementType)
                || elementType.equals(String.class)
                || elementType.equals(UUID.class)
                || elementType.isEnum();
        return result;
    }

    public static <T, C extends Chunk<T>> C newChunk(Class<C> returnType, Class<T> elementType, long fetch, long first, long next, long total) {
        try {
            C result = returnType.newInstance();
            result.setElementType(elementType);
            result.setFetch(fetch);
            result.setFirst(first);
            result.setNext(next);
            result.setTotal(total);
            return result;
        } catch (InstantiationException | IllegalAccessException eee) {
            throw new IllegalArgumentException("Chunk child class must have no args constructor", eee);
        }
    }
}
