/*
 * #%L
 * Nuiton Utils :: Nuiton Validator
 * 
 * $Id: NuitonFieldValidatorSupport.java 2182 2011-08-24 13:13:11Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/nuiton-utils/tags/nuiton-utils-2.6.4/nuiton-validator/src/main/java/org/nuiton/validator/xwork2/field/NuitonFieldValidatorSupport.java $
 * %%
 * Copyright (C) 2011 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.validator.xwork2.field;

import com.opensymphony.xwork2.validator.ValidationException;
import com.opensymphony.xwork2.validator.validators.FieldValidatorSupport;

/**
 * Nuiton default field validator.
 * <p/>
 * This validator offers a {@link #skip} property that can be used to skip or
 * not the validator, this property is a OGNL expression.
 * <p/>
 * To use this new field validator support, just now implements the method
 * {@link #validateWhenNotSkip(Object)}. This method will be invoked only if the skip
 * parameter is evaluated to {@code false}.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 2.3
 */
public abstract class NuitonFieldValidatorSupport extends FieldValidatorSupport {

    /**
     * extra parameter at the very beginning of the
     * {@link #validate(Object)} method to be able to skip (or not) the
     * validator execution.
     * <p/>
     * by default the value is {@code false} : it seems fair to want to
     * validate if the validator is used :D...
     */
    protected String skip = "false";

    /**
     * Sets the value of the {@link #skip} parameter.
     *
     * @param skip the new value of the {@link #skip} parameter
     */
    public void setSkip(String skip) {
        this.skip = skip;
    }

    /**
     * Method to be invoked when skip parameter was not evaludated to {@code true}.
     *
     * @param object the object to be validated.
     * @throws ValidationException is thrown if there is validation error(s).
     */

    protected abstract void validateWhenNotSkip(Object object) throws ValidationException;

    @Override
    public void validate(Object object) throws ValidationException {

        // evaluate the skip parameter
        boolean mustSkip = evaluateSkipParameter(object);

        if (mustSkip) {

            // skip is set to true, so skip the validation
            if (log.isDebugEnabled()) {
                log.debug("Skip the validation from " + this +
                          ", due to skip parameter evaluated to true");
            }
            return;
        }

        // must validate
        validateWhenNotSkip(object);
    }

    /**
     * Evaluate the skip parameter value against the object to validate.
     * <p/>
     * This parameter can be an OGNL expression.
     *
     * @param object the object to validate
     * @return the evaluation of the skip parameter.
     * @throws ValidationException if could not evaluate the parameter
     */
    protected boolean evaluateSkipParameter(Object object) throws ValidationException {

        skip = skip.trim();

        if ("false".equals(skip)) {

            return false;
        }

        if ("true".equals(skip)) {

            return true;
        }

        try {
            Boolean answer = Boolean.FALSE;
            Object obj;
            obj = getFieldValue(skip, object);
            if (obj != null && obj instanceof Boolean) {
                answer = (Boolean) obj;
            }
            return answer;
        } catch (ValidationException e) {
            throw e;
        } catch (Exception e) {
            // let this pass, but it will be logged right below
            throw new ValidationException(
                    "Can not evaluate boolean expression [" + skip +
                    "] for reason " + e.getMessage());
        }

    }

    @Override
    public Object getFieldValue(String name,
                                Object object) throws ValidationException {
        return super.getFieldValue(name, object);
    }
}
