/*
 * #%L
 * Nuiton Utils
 * 
 * $Id: BinderProvider.java 1840 2010-05-06 09:53:21Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/nuiton-utils/tags/nuiton-utils-1.5/src/main/java/org/nuiton/util/beans/BinderProvider.java $
 * %%
 * Copyright (C) 2004 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.util.beans;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.util.HashMap;
import java.util.Map;

/**
 * Manage a cache of {@link Binder} objects.
 * <p/>
 * You must first register some binders via the {@code registerBinder} api.
 * <pre>
 * Binder<User,UserDTO> mybinder = ...;
 * registerBinder(myNewBinder);
 * </pre>
 * To use several binders of the the same types, you can moreover specify
 * a context name of your binder :
 * <pre>
 * Binder<User,UserDTO> mybinder = ...;
 * registerBinder(myBinder, "One");
 * </pre>
 * <p/>
 * Then you can obtained them back via the api
 * <pre>
 * Binder<User,UserDTO> mybinder =  getBinder(User.class,UserDTO);
 * </pre>
 * <p/>
 * or with a context name :
 * <pre>
 * Binder<User,UserDTO> mybinder =  getBinder(User.class,UserDTO.class, "One");
 * </pre>
 *
 * @author tchemit <chemit@codelutin.com>
 * @see Binder
 * @see BinderBuilder
 * @since 1.1.5
 */
public class BinderProvider {

    /** Logger */
    private static final Log log = LogFactory.getLog(BinderProvider.class);

    /** Cache of registred binders indexed by their unique entry */
    protected static Map<BinderEntry, Binder<?, ?>> binders;

    /**
     * Gets the registred mirror binder (source type = target type) with no
     * context name specified.
     *
     * @param sourceType the type of source and target
     * @param <S>        the type of source and target
     * @return the registred binder or {@code null} if not found.
     */
    public static <S> Binder<S, S> getBinder(Class<S> sourceType) {
        return getBinder(sourceType, sourceType, null);
    }

    /**
     * Gets the registred mirror binder (source type = target type) with the
     * given context name.
     *
     * @param sourceType the type of source and target
     * @param name       the context's name of the searched binder
     * @param <S>        the type of source and target
     * @return the registred binder or {@code null} if not found.
     */
    public static <S> Binder<S, S> getBinder(Class<S> sourceType,
                                             String name) {
        return getBinder(sourceType, sourceType, name);
    }


    /**
     * Gets the registred binder given his types with no context name.
     *
     * @param sourceType the type of source
     * @param targetType the type of target
     * @param <S>        the type of source
     * @param <T>        the type of target
     * @return the registred binder or {@code null} if not found.
     */
    public static <S, T> Binder<S, T> getBinder(Class<S> sourceType,
                                                Class<T> targetType) {
        return getBinder(sourceType, targetType, null);
    }

    /**
     * Gets the registred binder given his types and his context's name.
     *
     * @param sourceType the type of source
     * @param targetType the type of target
     * @param name       the context's name of the searched binder
     * @param <S>        the type of source
     * @param <T>        the type of target
     * @return the registred binder or {@code null} if not found.
     */
    public static <S, T> Binder<S, T> getBinder(Class<S> sourceType,
                                                Class<T> targetType,
                                                String name) {
        BinderEntry entry = new BinderEntry(sourceType, targetType, name);
        if (log.isDebugEnabled()) {
            log.debug("for entry " + entry);
        }
        Binder<?, ?> result = getBinders().get(entry);
        if (result == null) {
            // binder not found
            if (log.isWarnEnabled()) {
                log.warn("Could not find binder " + entry);
            }
        }
        return (Binder<S, T>) result;
    }

    /**
     * Register a binder with no context name.
     * <p/>
     * <b>Note: </b> If a previous binder with same definition exists, it will
     * be overriden by the new binder.
     *
     * @param binder the binder to register.
     */
    public static void registerBinder(Binder<?, ?> binder) {
        registerBinder(binder, null);
    }

    /**
     * Register a binder with no context's name from a {@link BinderBuilder}.
     * <p/>
     * <b>Note: </b> If a previous binder with same definition exists, it will
     * be overriden by the new binder.
     *
     * @param builder the builder which contains builder model to use
     * @see BinderBuilder#createBinder(Class)
     */
    public static void registerBinder(BinderBuilder builder) {
        registerBinder(builder, Binder.class);
    }

    /**
     * Register a binder with no context's name from a {@link BinderBuilder}.
     * <p/>
     * <b>Note: </b> If a previous binder with same definition exists, it will
     * be overriden by the new binder.
     *
     * @param builder    the builder which contains builder model to use
     * @param binderType the type of binder to instanciate and register.
     * @see BinderBuilder#createBinder(Class)
     */
    public static <B extends Binder<?, ?>> void registerBinder(BinderBuilder builder,
                                                               Class<B> binderType) {

        registerBinder(builder, binderType, null);
    }

    /**
     * Register a binder with a context's name from a {@link BinderBuilder}.
     * <p/>
     * <b>Note: </b> If a previous binder with same definition exists, it will
     * be overriden by the new binder.
     *
     * @param builder the builder which contains builder model to use
     * @param name    the context's name
     * @see BinderBuilder#createBinder(Class)
     */
    public static void registerBinder(BinderBuilder builder, String name) {

        registerBinder(builder, Binder.class, name);
    }

    /**
     * Register a binder with a context's name from a {@link BinderBuilder}.
     * <p/>
     * <b>Note: </b> If a previous binder with same definition exists, it will
     * be overriden by the new binder.
     *
     * @param builder    the builder which contains builder model to use
     * @param binderType the type of binder to instanciate and register.
     * @param name       the context's name
     * @see BinderBuilder#createBinder(Class)
     */
    public static <B extends Binder<?, ?>> void registerBinder(BinderBuilder builder,
                                                               Class<B> binderType,
                                                               String name) {
        // instanciate the binder
        B binder = builder.createBinder(binderType);
        // register it
        registerBinder(binder, name);
    }

    /**
     * Register a binder with a context name.
     * <p/>
     * <b>Note: </b> If a previous binder with same definition exists, it will
     * be overriden by the new binder.
     *
     * @param binder the binder to register.
     * @param name   the context's name
     */
    public static void registerBinder(Binder<?, ?> binder, String name) {
        BinderModel<?, ?> model = binder.getModel();
        BinderEntry entry = new BinderEntry(
                model.getSourceType(),
                model.getTargetType(),
                name
        );
        if (log.isDebugEnabled()) {
            log.debug("binder to seek : " + entry);
        }
        Binder<?, ?> oldBinder = getBinders().get(entry);
        if (oldBinder != null) {
            // already a binder for this entry
            if (log.isWarnEnabled()) {
                log.warn("Binder already registred for " + entry + " : " +
                         oldBinder);
                log.warn("Will be replace by the new binder " + binder);
            }
        }
        // register the binder
        if (log.isDebugEnabled()) {
            log.debug("entry : " + entry + " : " + binder);
        }
        getBinders().put(entry, binder);
    }

    protected static Map<BinderEntry, Binder<?, ?>> getBinders() {
        if (binders == null) {
            binders = new HashMap<BinderEntry, Binder<?, ?>>();
        }
        return binders;
    }

    public static void clear() {
        if (binders != null) {
            binders.clear();
            binders = null;
        }
    }

    /**
     * Definition of an binder entry (source and target types + context name).
     * <p/>
     * <b>Note :</b>When no context is specified, we always use a
     * {@code null} context name.
     */
    public static class BinderEntry {

        protected final Class<?> sourceType;

        protected final Class<?> targetType;

        protected final String name;


        public BinderEntry(Class<?> sourceType,
                           Class<?> targetType,
                           String name) {
            this.sourceType = sourceType;
            this.targetType = targetType;
            this.name = name;
        }

        public Class<?> getSourceType() {
            return sourceType;
        }

        public Class<?> getTargetType() {
            return targetType;
        }

        public String getName() {
            return name;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }

            BinderEntry that = (BinderEntry) o;

            return (name == null ? that.name == null : name.equals(that.name)) &&
                    sourceType.equals(that.sourceType) &&
                    targetType.equals(that.targetType);
        }

        @Override
        public int hashCode() {
            int result = sourceType.hashCode();
            result = 31 * result + targetType.hashCode();
            result = 31 * result + (name != null ? name.hashCode() : 0);
            return result;
        }

        @Override
        public String toString() {
            StringBuilder buffer = new StringBuilder("<");
            buffer.append(super.toString());
            buffer.append(", sourceType: ").append(getSourceType()).append(',');
            buffer.append(" targetType: ").append(getTargetType()).append(',');
            buffer.append(" name: ").append(getName()).append('>');

            return buffer.toString();
        }
    }
}
