/*
 * #%L
 * Nuiton Utils
 * 
 * $Id: BinderBuilder.java 1841 2010-05-06 15:45:26Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/nuiton-utils/tags/nuiton-utils-1.4/src/main/java/org/nuiton/util/beans/BinderBuilder.java $
 * %%
 * Copyright (C) 2004 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.util.beans;

import java.beans.BeanInfo;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.Collection;
import java.util.Map;
import java.util.TreeMap;

/**
 * A builder of {@link BinderModel} and {@link Binder}.
 * <p/>
 * A {@code binder} permits to copy some properties from a bean to another one.
 *
 * @author tchemit <chemit@codelutin.com>
 * @see BinderModel
 * @see Binder
 * @since 1.1.5
 */
public class BinderBuilder {

    /** current model used to build the binder */
    protected BinderModel<?, ?> model;

    /** source properties descriptors */
    protected Map<String, PropertyDescriptor> sourceDescriptors;

    /** target properties descriptors */
    protected Map<String, PropertyDescriptor> targetDescriptors;

    public BinderBuilder() {
    }

    /**
     * Creates a mirrored binder for the given {@code type} and add the given
     * simple properties.
     *
     * @param type       the type of mirrored binder
     * @param properties simple properties to add
     */
    public BinderBuilder(Class<?> type, String... properties) {
        createBinderModel(type);
        addSimpleProperties(properties);
    }

    /**
     * Creates a binder for the given types and add the given simple properties.
     *
     * @param sourceType type of the source of the binder
     * @param targetType type of the target of the binder
     * @param properties simple properties to add
     */
    public BinderBuilder(Class<?> sourceType,
                         Class<?> targetType,
                         String... properties) {
        createBinderModel(sourceType, targetType);
        addSimpleProperties(properties);
    }

    /**
     * Creates a new binder model for a mirrored binder (source type = target
     * type).
     * <p/>
     * <b>Note:</b> If a previous model was created, but not released via the
     * method {@link #createBinder(Class)}, the method will failed.
     *
     * @param type the type of source and target
     * @return the instance of the builder
     * @throws IllegalStateException if a previous builder model was created
     *                               without been released
     * @throws NullPointerException  if a parameter is null
     */
    public BinderBuilder createBinderModel(Class<?> type)
            throws IllegalStateException, NullPointerException {
        createBinderModel(type, type);
        return this;
    }

    /**
     * Creates a new binder model.
     * <p/>
     * <b>Note:</b> If a previous model was created, but not released via the
     * method {@link #createBinder(Class)}, the method will failed.
     *
     * @param sourceType the type of the source
     * @param targetType the type of the target
     * @return the instance of the builder
     * @throws IllegalStateException if a previous builder model was created
     *                               without been released
     * @throws NullPointerException  if a parameter is null
     */
    public BinderBuilder createBinderModel(Class<?> sourceType,
                                           Class<?> targetType)
            throws IllegalStateException, NullPointerException {
        if (sourceType == null) {
            throw new NullPointerException("sourceType can not be null");
        }
        if (targetType == null) {
            throw new NullPointerException("targetType can not be null");
        }

        if (model != null) {
            throw new IllegalStateException(
                    "there is already a binderModel in construction, release " +
                    "it with the method createBinder before using this method."
            );
        }

        // init model
        model = new BinderModel(sourceType, targetType);

        // obtain source descriptors
        sourceDescriptors = new TreeMap<String, PropertyDescriptor>();
        loadDescriptors(model.getSourceType(), sourceDescriptors);

        // obtain target descriptors
        targetDescriptors = new TreeMap<String, PropertyDescriptor>();
        loadDescriptors(model.getTargetType(), targetDescriptors);

        return this;
    }

    /**
     * Creates a new binder given using the {@link Binder} type of binder
     * from the internal binder model
     * previously created via the method {@code createBinderModel<XXX>} and
     * then filled with methods {@code addXXX(XXX)}.
     * <p/>
     * <b>Note:</b> If no model is present, the method will fail.
     *
     * @return the instance of the new buinder.
     * @throws IllegalStateException if no model was previously created.
     * @throws NullPointerException  if the parameter is {@code null}
     */
    public Binder<?, ?> createBinder()
            throws NullPointerException, IllegalStateException {

        Binder<?, ?> binder = createBinder(Binder.class);
        return binder;
    }

    /**
     * Creates a new binder given his type from the internal binder model
     * previously created via the method {@code createBinderModel<XXX>} and
     * then filled with methods {@code addXXX(XXX)}.
     * <p/>
     * <b>Note:</b> If no model is present, the method will fail.
     *
     * @param binderType the type of binder to instanciate
     * @param <B>        the type of binder to instanciate
     * @return the instance of the new buinder.
     * @throws IllegalStateException if no model was previously created.
     * @throws NullPointerException  if the parameter is {@code null}
     */
    public <B extends Binder<?, ?>> B createBinder(Class<B> binderType)
            throws NullPointerException, IllegalStateException {
        checkModelExists();
        if (binderType == null) {
            throw new NullPointerException("binderType can not be null");
        }
        try {
            B binder = binderType.newInstance();
            binder.setModel(model);
            return binder;
        } catch (Exception e) {
            throw new IllegalStateException(
                    "could not instanciate binder " + binderType, e);
        } finally {
            // release resources of the model
            model = null;
            sourceDescriptors.clear();
            sourceDescriptors = null;
            targetDescriptors.clear();
            targetDescriptors = null;
        }
    }

    /**
     * Add to the binder model some simple properties (says source property name
     * = target property name).
     * <p/>
     * <b>Note:</b> If no model is present, the method will fail.
     *
     * @param properties the name of mirrored property
     * @return the instance of the builder
     * @throws IllegalStateException if no model was previously created
     * @throws NullPointerException  if a property is {@code null}
     */
    public BinderBuilder addSimpleProperties(String... properties)
            throws IllegalStateException, NullPointerException {
        checkModelExists();
        for (String property : properties) {
            if (property == null) {
                throw new NullPointerException(
                        "parameter 'properties' can not contains a null value");
            }
            addProperty0(property, property);
        }
        return this;
    }

    /**
     * Add to the binder model some simple properties (says source property name
     * = target property name).
     * <p/>
     * <b>Note:</b> If no model is present, the method will fail.
     *
     * @param sourceProperty the name of the source property to bind
     * @param targetProperty the name of the target property to bind
     * @return the instance of the builder
     * @throws IllegalStateException if no model was previously created
     * @throws NullPointerException  if a parameter is {@code null}
     */

    public BinderBuilder addProperty(String sourceProperty,
                                     String targetProperty)
            throws IllegalStateException, NullPointerException {
        if (sourceProperty == null) {
            throw new NullPointerException(
                    "parameter 'sourceProperty' can not be null");
        }
        if (targetProperty == null) {
            throw new NullPointerException(
                    "parameter 'targetProperty' can not be null");
        }
        checkModelExists();
        addProperty0(sourceProperty, targetProperty);
        return this;
    }

    /**
     * Add to the binder model some properties.
     * <p/>
     * Parameter {@code sourceAndTargetProperties} must be a array of couple
     * of {@code sourceProperty}, {@code targetProperty}.
     * <p/>
     * Example :
     * <pre>
     * builder.addProperties("name","name2","text","text");
     * </pre>
     * <p/>
     * <b>Note:</b> If no model is present, the method will fail.
     *
     * @param sourceAndTargetProperties the couple of (sourceProperty -
     *                                  targetProperty) to bind
     * @return the instance of the builder
     * @throws IllegalStateException    if no model was previously created
     * @throws IllegalArgumentException if there is not the same number of
     *                                  source and target properties
     * @throws NullPointerException     if a parameter is {@code null}
     */
    public BinderBuilder addProperties(String... sourceAndTargetProperties)
            throws IllegalStateException, IllegalArgumentException,
            NullPointerException {
        checkModelExists();
        if (sourceAndTargetProperties.length % 2 != 0) {
            throw new IllegalArgumentException(
                    "must have couple(s) of sourceProperty,targetProperty) " +
                    "but had " + Arrays.toString(sourceAndTargetProperties));
        }
        for (int i = 0, max = sourceAndTargetProperties.length / 2;
             i < max; i++) {
            String sourceProperty = sourceAndTargetProperties[2 * i];
            String targetProperty = sourceAndTargetProperties[2 * i + 1];
            if (sourceProperty == null) {
                throw new NullPointerException(
                        "parameter 'sourceAndTargetProperties' can not " +
                        "contains a null value");
            }
            if (targetProperty == null) {
                throw new NullPointerException(
                        "parameter 'sourceAndTargetProperties' can not " +
                        "contains a null value");
            }
            addProperty0(sourceProperty, targetProperty);
        }
        return this;
    }

    public BinderBuilder addCollectionStrategy(Binder.CollectionStrategy strategy,
                                               String... propertyNames) {

        for (String propertyName : propertyNames) {

            // check property is registred
            if (!model.containsSourceProperty(propertyName)) {
                throw new IllegalArgumentException(
                        "source property '" + propertyName + "' " +
                        " is NOT registred.");
            }

            // check property is collection type
            PropertyDescriptor descriptor = sourceDescriptors.get(propertyName);
            Class<?> type = descriptor.getPropertyType();
            if (!Collection.class.isAssignableFrom(type)) {
                throw new IllegalStateException(
                        "source property '" + propertyName +
                        "' is not a collection type [" + type + "]");
            }

            // can safely add the strategy
            model.addCollectionStrategy(propertyName, strategy);
        }
        return this;
    }

    public BinderBuilder addBinder(String propertyName, Binder<?, ?> binder) {

        // check property is registred
        if (!model.containsSourceProperty(propertyName)) {
            throw new IllegalArgumentException(
                    "source property '" + propertyName + "' " +
                    " is NOT registred.");
        }

        // check property is the same type of given binder
        PropertyDescriptor descriptor = sourceDescriptors.get(propertyName);
        Class<?> type = descriptor.getPropertyType();

        if (!Collection.class.isAssignableFrom(type) &&
            !binder.model.getSourceType().isAssignableFrom(type)) {
            throw new IllegalStateException(
                    "source property '" + propertyName +
                    "' has not the same type [" + type +
                    "] of the binder [" + binder.model.getSourceType() + "].");
        }

        // can safely add the strategy
        model.addBinder(propertyName, binder);

        return this;
    }

    protected void addProperty0(String sourceProperty,
                                String targetProperty) {
        // check srcProperty does not exist
        if (model.containsSourceProperty(sourceProperty)) {
            throw new IllegalArgumentException("source property '" +
                                               sourceProperty + "' " + " was already registred.");
        }
        // check dstProperty does not exist
        if (model.containsTargetProperty(targetProperty)) {
            throw new IllegalArgumentException("destination property '" +
                                               targetProperty + "' " + " was already registred.");
        }

        // obtain source descriptor
        PropertyDescriptor sourceDescriptor =
                sourceDescriptors.get(sourceProperty);
        if (sourceDescriptor == null) {
            throw new IllegalArgumentException("no property '" +
                                               sourceProperty + "' " + "found on type " +
                                               model.getSourceType());
        }
        // check srcProperty is readable
        Method readMethod = sourceDescriptor.getReadMethod();
        if (readMethod == null) {
            throw new IllegalArgumentException("property '" + sourceProperty +
                                               "' " + "is not readable on type " + model.getSourceType());
        }

        // obtain dst descriptor
        PropertyDescriptor targetDescriptor =
                targetDescriptors.get(targetProperty);
        if (targetDescriptor == null) {
            throw new IllegalArgumentException("no property '" +
                                               targetProperty + "' " + "found on type " +
                                               model.getTargetType());
        }
        // check dstProperty is writable
        Method writeMethod = sourceDescriptor.getWriteMethod();
        if (writeMethod == null) {
            throw new IllegalArgumentException("property '" + targetProperty +
                                               "' " + "is not writable on type " + model.getTargetType());
        }

        // check types are ok
        Class<?> sourceType = sourceDescriptor.getPropertyType();
        Class<?> targetType = targetDescriptor.getPropertyType();
        //TODO-TC20100221 : should check if primitive and boxed it in such case
        if (!sourceType.equals(targetType)) {
            throw new IllegalArgumentException("source property '" +
                                               sourceProperty + "' and target property '" +
                                               targetProperty + "' are not compatible ( sourceType : " +
                                               sourceType + " vs targetType :" + targetType + ')');
        }

        // safe to add the binding
        model.addBinding(sourceDescriptor, targetDescriptor);
    }


    protected void checkModelExists() throws IllegalStateException {
        if (model == null) {
            throw new IllegalStateException("there is not model, must " +
                                            "create one with createBinderModel method");
        }
    }

    protected BinderModel<?, ?> getModel() {
        return model;
    }

    protected static void loadDescriptors(
            Class<?> type,
            Map<String, PropertyDescriptor> descriptors) {
        try {

            BeanInfo beanInfo = Introspector.getBeanInfo(type);
            for (PropertyDescriptor descriptor :
                    beanInfo.getPropertyDescriptors()) {
                if (!descriptors.containsKey(descriptor.getName())) {
                    descriptors.put(descriptor.getName(), descriptor);
                }
            }
        } catch (IntrospectionException e) {
            throw new RuntimeException("Could not obtain bean properties " +
                                       "descriptors for source type " + type, e);
        }
        Class<?>[] interfaces = type.getInterfaces();
        for (Class<?> i : interfaces) {
            loadDescriptors(i, descriptors);
        }
        Class<?> superClass = type.getSuperclass();
        if (superClass != null && !Object.class.equals(superClass)) {
            loadDescriptors(superClass, descriptors);
        }
    }
}
