/*
 * #%L
 * Nuiton Utils
 * 
 * $Id: Log.java 1830 2010-04-15 14:29:20Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/nuiton-utils/tags/nuiton-utils-1.4/src/main/java/org/nuiton/util/Log.java $
 * %%
 * Copyright (C) 2004 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

/* *
 * Log.java
 *
 * Created: 12 août 2004
 *
 * @author Benjamin Poussin <poussin@codelutin.com>
 * @version $Revision: 1830 $
 *
 * Mise a jour: $Date: 2010-04-15 16:29:20 +0200 (jeu., 15 avril 2010) $
 * par : */

package org.nuiton.util;

import java.util.EventListener;
import java.util.EventObject;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Cette classe permet de mettre en place un monitoring d'application simplement.
 * Le développeur a chaque fois qu'il le souhaite ajoute dans son code un
 * Log.logUserInfo("...", "....") qui indique un message que l'utilisateur est
 * suceptible de vouloir, par exemple le résultat de la sauvegarde d'un fichier.
 * Ou bien si l'application effectue un traitement, il peut utiliser
 * Log.logTask("...", "...", max, value) pour indiquer qu'un traitement est
 * en cours.
 * <p/>
 * Il suffit ensuite de creer un objet qui herite de {@link LogListener}, puis
 * de l'enregistrer sur certaine category d'event il recevra alors les
 * evenements de l'utilisateur.
 * <p/>
 * Une utilisation peut-être la bar de status qui afficherait le message.
 * <p/>
 * exemple de code
 * <pre>
 * LogListener l = new StatusBar();
 * Log.addLogListener(l);
 * ...
 * ...
 * ...
 * Log.logTask("SAVE", "Sauvegarde en cours", -1, 0);
 * ... // sauvegarde
 * Log.logUserInfo("SAVE", "Sauvegarde réussie");
 * Log.logTask("SAVE", "Sauvegarde terminée", 0, 0);
 * </pre>
 */
public class Log { // Log

    static private Log LOG_INSTANCE = new Log();

    /** L'interface que doivent respecter un listener */
    public interface LogListener extends EventListener {
        void logMessage(LogEvent e);

        void logTask(LogEvent e);
    }

    /** Les events envoyes aux listeners */
    static public class LogEvent extends EventObject {
        /**  */
        private static final long serialVersionUID = 6597052732707368243L;

        protected String category;
        protected Level level;
        protected String message;
        protected Throwable exception;
        protected int taskMax = 0;
        protected int taskValue = 0;

        public LogEvent(Object source, String category, Level level, String message, Throwable exception) {
            super(source == null ? LOG_INSTANCE : source);
            this.category = category;
            this.level = level;
            this.message = message;
            this.exception = exception;
        }

        public LogEvent(Object source, String category, String message, int taskMax, int taskValue) {
            super(source == null ? LOG_INSTANCE : source);
            this.category = category;
            this.message = message;
            this.taskMax = taskMax;
            this.taskValue = taskValue;
        }

        public String getCategory() {
            return category;
        }

        /** Retourne une valeur que si l'event est un sendMessage, sinon null; */
        public Level getLevel() {
            return level;
        }

        public String getMessage() {
            return message;
        }

        /**
         * L'exception envoyé dans le log, si le log ne contient pas d'exception
         * alors null est retourne.
         */
        public Throwable getException() {
            return exception;
        }

        public int getTaskMax() {
            return taskMax;
        }

        public int getTaskValue() {
            return taskValue;
        }

    }

    /** Le Level pour les log utilisateur */
    static private class UserLevel extends Level {
        /**  */
        private static final long serialVersionUID = -9075227788352473733L;

        public UserLevel(String name, int value) {
            super(name, value);
        }
    }

    /** Tous les listeners */
    static protected CategorisedListenerSet<LogListener> listeners = new CategorisedListenerSet<LogListener>();

    /** Level.INFO = 700 Level.FINE=500 * */
    public static final Level USER_INFO = new UserLevel("USERINFO", 600);

    /** Ajoute un listener sur tous les logs envoye */
    static public void addLogListener(LogListener l) {
        listeners.add(LOG_INSTANCE, l);
    }

    /** enleve un listener sur tous les logs envoye */
    static public void removeLogListener(LogListener l) {
        listeners.remove(LOG_INSTANCE, l);
    }

    /** Ajoute un listener sur une certaine category de log */
    static public void addLogListener(LogListener l, String category) {
        listeners.add(category, l);
    }

    /** enleve un listener sur une certaine category de log */
    static public void removeLogListener(LogListener l, String category) {
        listeners.remove(category, l);
    }

    static protected void fire(String category, Level level, String message, Throwable exception) {
        LogEvent e = new LogEvent(null, category, level, message, exception);
        try {
            logDevFinest("org.nuiton.util.Log.fire", "Category: " + category + " listeners enregistrés: " + listeners);
            listeners.fire(category, "logMessage", e);
            listeners.fire(LOG_INSTANCE, "logMessage", e);
        } catch (Exception eee) {
            Logger.getLogger(Log.class.getName() + ".fire").log(Level.WARNING, "Error during send log event", eee);
        }
    }

    static protected void fire(String category, String message, int max, int value) {
        LogEvent e = new LogEvent(null, category, message, max, value);
        try {
            log("org.nuiton.util.Log.fire", Level.FINEST, "Category: " + category + " listeners enregistrés: " + listeners, null);
            listeners.fire(category, "logTask", e);
            listeners.fire(LOG_INSTANCE, "logTask", e);
        } catch (Exception eee) {
            Logger.getLogger(Log.class.getName() + ".fire").log(Level.WARNING, "Error during send log event", eee);
        }
    }

    /**
     * Ajoute un message dans le USER_LEVEL.
     *
     * @param category la category du message, souvent un nom de module d'une
     *                 application.
     * @param message  le message a envoyer
     */
    static public void logUserInfo(String category, String message) {
        logUserInfo(category, message, null);
    }

    static public void logUserInfo(String category, String message, Throwable e) {
        log(category, USER_INFO, message, e);
        fire(category, USER_INFO, message, e);
    }

    /**
     * Permet d'indiquer l'avancement d'une tache. Si l'on ne connait pas la
     * longueur de la tache il suffit d'indiquer -1 dans max, pour indiquer
     * une tache en cours dont on ne connait pas la fin.
     * lorsque la tache est termine, il suffit d'appeler cette methode avec max
     * valant 0.
     *
     * @param category la category de la tache
     * @param message  le message a afficher, le message peut-etre null
     * @param max      l'entier qui indique la fin de la tache. La tache commence a
     *                 0 et fini lorsque l'on arrive a max. Si max vaut -1 cela veut dire
     *                 que la tache debute mais qu'on ne connait pas sa longueur
     * @param current  la valeur courante de la tache.
     */
    static public void logTask(String category, String message, int max, int current) {
        log(category, USER_INFO, "task: " + message + "[" + current + "/" + max + "]", null);
        fire(category, message, max, current);
    }

    static public void log(String category, Level level, String message, Throwable e) {
        if (e == null) {
            Logger.getLogger(category).log(level, message);
        } else {
            Logger.getLogger(category).log(level, message, e);
        }
    }

    static public void logDevFinest(String category, String message) {
        logDevFinest(category, message, null);
    }

    static public void logDevFinest(String category, String message, Throwable e) {
        log(category, Level.FINEST, message, e);
    }

    static public void logDevFiner(String category, String message) {
        logDevFiner(category, message, null);
    }

    static public void logDevFiner(String category, String message, Throwable e) {
        log(category, Level.FINER, message, e);
    }

    static public void logDevFine(String category, String message) {
        logDevFine(category, message, null);
    }

    static public void logDevFine(String category, String message, Throwable e) {
        log(category, Level.FINE, message, e);
    }

    static public void logDevInfo(String category, String message) {
        logDevInfo(category, message, null);
    }

    static public void logDevInfo(String category, String message, Throwable e) {
        log(category, Level.INFO, message, e);
    }

    static public void logDevWarn(String category, String message) {
        logDevInfo(category, message, null);
    }

    static public void logDevWarn(String category, String message, Throwable e) {
        log(category, Level.WARNING, message, e);
    }

    static public void logDevSevere(String category, String message) {
        logDevSevere(category, message, null);
    }

    static public void logDevSevere(String category, String message, Throwable e) {
        log(category, Level.SEVERE, message, e);
    }

} // Log
