/*
 * #%L
 * Nuiton Utils
 * 
 * $Id: ReflectUtil.java 1830 2010-04-15 14:29:20Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/nuiton-utils/tags/nuiton-utils-1.3/src/main/java/org/nuiton/util/ReflectUtil.java $
 * %%
 * Copyright (C) 2004 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.util;

import static org.nuiton.i18n.I18n._;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.List;

/**
 * Des méthodes utiles d'introspection
 *
 * @author tchemit <chemit@codelutin.com>
 */
public class ReflectUtil {

    /**
     * Pour déterminer si un champ d'une classe est une constante
     * (modifiers sont static, final et public)
     *
     * @param field le champs à tester
     * @return <code>true</code> si les modifiers sont final, static et public
     */
    public static boolean isConstantField(Field field) {
        int modifiers = field.getModifiers();
        return Modifier.isPublic(modifiers) && Modifier.isStatic(modifiers) && Modifier.isFinal(modifiers);
    }

    /**
     * Recherche dans une classe donnée <code>klazz</code>, les constantes d'un
     * certain type <code>searchingClass</code> et les retourne.
     * <p/>
     * L'algorithme parcourt aussi les superclasses.
     *
     * @param <T> enumeration's type
     * @param klass          la classe contenant les constantes
     * @param searchingClass le type des champs constants à récupérer
     * @return la liste des champs du type requis dans
     * @throws RuntimeException si problème lors de la récupération
     */
    @SuppressWarnings({"unchecked"})
    public static <T> List<T> getConstants(Class<?> klass, Class<T> searchingClass) {
        List<T> result = new ArrayList<T>();
        for (Field field : klass.getDeclaredFields()) {
            if (!field.isAccessible()) {
                field.setAccessible(true);
            }
            if (searchingClass.isAssignableFrom(field.getType()) && isConstantField(field)) {
                try {
                    result.add((T) field.get(null));
                } catch (IllegalAccessException e) {
                    throw new RuntimeException(e);
                }
            }
        }
        Class<?> superClass = klass.getSuperclass();
        if (superClass != null) {
            result.addAll(getConstants(superClass, searchingClass));
        }
        return result;
    }

    /**
     * @param <T> enumeration's type
     * @param klass     the required class
     * @param fieldName the required constant name
     * @return the constant value
     */
    @SuppressWarnings({"unchecked"})
    public static <T> T getConstant(Class<?> klass, String fieldName) {
        try {
            T result = null;
            Field f = klass.getDeclaredField(fieldName);
            if (isConstantField(f)) {
                f.setAccessible(true);
                result = (T) f.get(null);
            }
            return result;
        } catch (NoSuchFieldException e) {
            throw new RuntimeException(e);
        } catch (IllegalAccessException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Convertit une classe non typée, en une classe d'enum
     *
     * @param <T> enumeration's type
     * @param type la classe a typer
     * @return la classe typee
     * @throws IllegalArgumentException si le type est null ou non une extension
     *                                  de la classe Enum.
     */
    @SuppressWarnings({"unchecked"})
    public static <T extends Enum<T>> Class<T> getEnumClass(Class<?> type) throws IllegalArgumentException {
        if (type == null) {
            throw new IllegalArgumentException(_("nuitonutil.error.null.parameter", "type"));
        }
        if (!type.isEnum()) {
            throw new IllegalArgumentException(_("nuitonutil.error.not.an.enum", type));
        }
        return (Class<T>) type;
    }
}
