/*
 * *##% 
 * Maven License Plugin
 * Copyright (C) 2008 - 2010 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package org.nuiton.license.plugin;


import org.apache.maven.project.MavenProject;
import org.nuiton.license.plugin.repository.License;
import org.nuiton.license.plugin.repository.LicenseRepository;
import org.nuiton.license.plugin.repository.LicenseRepositoryFactory;
import org.nuiton.plugin.AbstractPlugin;
import org.nuiton.plugin.PluginHelper;

import java.io.File;

/**
 * Le goal pour ajouter le fichier LICENSE.txt dans le classpath (et le generer s'il n'existe pas).
 *
 * @author chemit
 * @goal add-license
 * @phase generate-resources
 * @requiresProject true
 * @requiresDependencyResolution compile
 */
public class AddLicenseFileMojo extends AbstractPlugin {

    /**
     * Dependance du projet.
     *
     * @parameter default-value="${project}"
     * @required
     * @since 1.0.0
     */
    protected MavenProject project;
    /**
     * Fichier de la licence du module.
     *
     * @parameter expression="${license.licenceFile}" default-value="${basedir}/LICENSE.txt"
     * @required
     * @since 1.0.0
     */
    protected File licenseFile;
    /**
     * Le type de license a appliquer.
     * <p/>
     * Pour obtenir la liste des licenses disponibles, utilisez le goal <b>available-licenses</b>
     * <pre>
     *  mvn helper:available-licenses -Ddetail
     * </pre>
     *
     * @parameter expression="${license.licenseName}"
     * @required
     * @since 1.0.0
     */
    protected String licenseName;
    /**
     * Repertoire de sortie des sources.
     *
     * @parameter expression="${license.outputDirectory}" default-value="target/generated-sources/license"
     * @required
     * @since 1.0.0
     */
    protected File outputDirectory;
    /**
     * Repertoire de sortie des classes (classpath).
     *
     * @parameter expression="${license.licenceFilename}" default-value=""
     * @since 1.0.0
     */
    protected String licenseFilename;
    /**
     * Encoding a utiliser pour lire et ecrire les fichiers.
     *
     * @parameter expression="${helper.encoding}" default-value="${project.build.sourceEncoding}"
     * @required
     * @since 1.0.0
     */
    protected String encoding;
    /**
     * Un flag pour conserver un backup des fichiers modifies.
     *
     * @parameter expression="${license.keepBackup}"  default-value="false"
     * @since 1.0.0
     */
    protected boolean keepBackup;
    /**
     * Un flag pour forcer la generation.
     *
     * @parameter expression="${license.force}"  default-value="false"
     * @since 1.0.0
     */
    protected boolean force;
    /**
     * Un flag pour activer le mode verbeux.
     *
     * @parameter expression="${license.verbose}"  default-value="${maven.verbose}"
     * @since 1.0.0
     */
    protected boolean verbose;
    /**
     * Un flag pour faire une copie nommée dans META-INF (prefixe avec le nom de l'artifact).
     * <p/>
     * Cette option n'est utilisable que sur des modules avec un class-path (pas pour un pom)
     *
     * @parameter expression="${license.copyToMETA_INF}"  default-value="false"
     * @since 1.0.0
     */
    protected boolean copyToMETA_INF;
    /**
     * La baseURL d'un resolver de license supplementaire
     *
     * @parameter expression="${license.extraResolver}"
     * @since 1.0.0
     */
    protected String[] extraResolver;
    protected License license;

    protected boolean hasClassPath() {
        return rejectPackaging(Packaging.pom);
    }

    boolean doGenerate;

    @Override
    protected void init() throws Exception {

        // must generate if file does not exist
        doGenerate = true;

        if (!force) {
            // regenerate only if file exists and is newer than pom file
            doGenerate = !isFileNewerThanPomFile(licenseFile);
        }

        // acquire license

        LicenseRepository factory = LicenseRepositoryFactory.newLicenseRepository(true, true, extraResolver);

        license = factory.getLicense(licenseName);

        if (licenseFilename == null || licenseFilename.isEmpty()) {
            licenseFilename = licenseName;
        }
    }

    @Override
    protected void doAction() throws Exception {
        getLog().info("using licence [" + licenseName + "]");

        if (doGenerate) {
            if (verbose) {
                getLog().info("detail : " + license);
            }

            if (licenseFile.exists() && keepBackup) {
                if (verbose) {
                    getLog().info("backup " + licenseFile);
                }
                // copy it to backup file
                backupFile(licenseFile);
            }
        }

        String licenseContent = license.getLicenseContent(encoding);

        if (doGenerate) {
            writeFile(licenseFile, licenseContent, encoding);
        }

        if (hasClassPath()) {
            // copy LICENSE.txt to classpath
            File target = new File(outputDirectory, licenseFile.getName());

            copyFile(licenseFile, target);

            if (copyToMETA_INF) {
                File destFile = PluginHelper.getFile(outputDirectory, "META-INF", project.getArtifactId() + "-" + licenseFile.getName());
                copyFile(licenseFile, destFile);
            }
            addResourceDir(outputDirectory, "**/*.txt");
        }
    }

    public File getLicenseFile() {
        return licenseFile;
    }

    public String getLicenseName() {
        return licenseName;
    }

    public org.nuiton.license.plugin.repository.License getLicense() {
        return license;
    }

    public String[] getExtraResolver() {
        return extraResolver;
    }

    public String getEncoding() {
        return encoding;
    }

    public boolean isKeepBackup() {
        return keepBackup;
    }

    public boolean isForce() {
        return force;
    }

    public File getOutputDirectory() {
        return outputDirectory;
    }

    public void setLicenseFile(File licenseFile) {
        this.licenseFile = licenseFile;
    }

    public void setLicenseName(String licenseName) {
        this.licenseName = licenseName;
    }

    public void setExtraResolver(String[] extraResolver) {
        this.extraResolver = extraResolver;
    }

    public void setLicense(org.nuiton.license.plugin.repository.License license) {
        this.license = license;
    }

    public void setEncoding(String encoding) {
        this.encoding = encoding;
    }

    public void setKeepBackup(boolean keepBackup) {
        this.keepBackup = keepBackup;
    }

    public void setOutputDirectory(File outputDirectory) {
        this.outputDirectory = outputDirectory;
    }

    public void setForce(boolean force) {
        this.force = force;
    }

    @Override
    public boolean isVerbose() {
        return verbose;
    }

    @Override
    public void setVerbose(boolean verbose) {
        this.verbose = verbose;
    }

    @Override
    public MavenProject getProject() {
        return project;
    }

    @Override
    public void setProject(MavenProject project) {
        this.project = project;
    }
}
