/*
 * #%L
 * NuitonMatrix
 * 
 * $Id: MatrixFactory.java 420 2012-03-09 09:22:50Z echatellier $
 * $HeadURL: http://svn.nuiton.org/svn/nuiton-matrix/tags/nuiton-matrix-2.3/nuiton-matrix/src/main/java/org/nuiton/math/matrix/MatrixFactory.java $
 * %%
 * Copyright (C) 2004 - 2011 CodeLutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.math.matrix;

import java.lang.reflect.Constructor;
import java.util.List;

/**
 * Cette classe permet de creer des matrices, toutes les creations de matrice
 * doivent etre faite a travers cette classe. Cette classe permet de modifier la
 * representation interne des matrices de facon simple.
 * <p>
 * Created: 11 octobre 2005 20:15:20 CEST
 * 
 * @author Benjamin POUSSIN <poussin@codelutin.com>
 * @version $Revision: 420 $
 * 
 * Last update: $Date: 2012-03-09 10:22:50 +0100 (Fri, 09 Mar 2012) $ by :
 * $Author: echatellier $
 */
public class MatrixFactory { // MatrixFactory

    /** Valeur par defaut si aucun type de Vector n'est donné */
    protected static Class<?> defaultVectorClass = DoubleBigVector.class;

    protected Class<?> vectorClass = null;

    protected static SemanticMapper defaultSemanticMapper = new SemanticMapper();

    protected MatrixFactory(Class<?> vectorClass) {
        this.vectorClass = vectorClass;
    }

    public static void setDefaultVectorClass(Class<?> vectorClass) {
        defaultVectorClass = vectorClass;
    }

    public static Class<?> getDefaultVectorClass() {
        return defaultVectorClass;
    }

    public static void setSemanticMapper(SemanticMapper semanticMapper) {
        defaultSemanticMapper = semanticMapper;
    }

    public static SemanticMapper getSemanticMapper() {
        return defaultSemanticMapper;
    }

    /**
     * Retourne une factory utilisant vectorClass comme classe de base a
     * l'implantation des matrices.
     * 
     * @param vectorClass vector class implementation
     * @return factory
     */
    public static MatrixFactory getInstance(Class<?> vectorClass) {
        return new MatrixFactory(vectorClass);
    }

    /**
     * Utilise par defaut {@link FloatBigVector}.
     * 
     * @return factory
     */
    public static MatrixFactory getInstance() {
        return getInstance(defaultVectorClass);
    }

    public MatrixND create(int[] dim) {
        return new MatrixNDImpl(this, dim);
    }

    /**
     * Convert a double array into matrix.
     * 
     * @param values The values to fill the matrix
     * @param dim An array representing the dimensions of the matrix
     * @return a 2D matrix filled with the values, null if the dimension is more
     *         than 2
     */
    public MatrixND create(double[] values, int[] dim) {

        if (dim.length > 2) {
            return null;
        }
        MatrixNDImpl matrix = new MatrixNDImpl(this, dim);

        if (dim.length == 2) {
            for (int i = 0; i < dim[0]; i++) {
                for (int j = 0; j < dim[1]; j++) {
                    int[] coordinates = { i, j };
                    matrix.setValue(coordinates, values[i * dim[1] + j]);
                }
            }
        }
        if (dim.length == 1) {
            for (int i = 0; i < dim[0]; i++) {
                int[] coordinates = { i };
                matrix.setValue(coordinates, values[i]);
            }
        }

        return matrix;
    }

    public MatrixND create(List<?>[] semantics) {
        return new MatrixNDImpl(this, semantics);
    }

    public MatrixND create(String name, int[] dim) {
        return new MatrixNDImpl(this, name, dim);
    }

    public MatrixND create(String name, int[] dim, String[] dimNames) {
        return new MatrixNDImpl(this, name, dim, dimNames);
    }

    public MatrixND create(String name, List<?>[] semantics) {
        return new MatrixNDImpl(this, name, semantics);
    }

    public MatrixND create(String name, List<?>[] semantics, String[] dimNames) {
        return new MatrixNDImpl(this, name, semantics, dimNames);
    }

    public MatrixND create(MatrixND matrix) {
        return new MatrixNDImpl(this, matrix);
    }

    /**
     * Crée une nouvelle matrice identité. Une matrice identité est une matrice
     * à 2 dimensions dont tous les éléments de la diagonal vaut 1
     * 
     * @param size la taille de la matrice
     * @return une nouvelle matrice identité
     */
    public MatrixND matrixId(int size) {
        MatrixND result = create(new int[] { size, size });
        for (int i = 0; i < size; i++) {
            result.setValue(i, i, 1);
        }
        return result;
    }

    protected Vector createVector(int length) {
        try {
            Constructor<?> c = vectorClass
                    .getConstructor(new Class<?>[] { Integer.TYPE });
            return (Vector) c.newInstance(new Object[] { length });
        } catch (Exception eee) {
            throw new RuntimeException("Can't create vector", eee);
        }
    }
    
    public MatrixProxy createProxy(List<?>[] semantics, MatrixProvider matrixProvider) {
        MatrixProxy matrixProxy = new MatrixProxy(this, semantics);
        matrixProxy.setMatrixProvider(matrixProvider);
        return matrixProxy;
    }

    public MatrixProxy createProxy(String name, int[] dim, MatrixProvider matrixProvider) {
        MatrixProxy matrixProxy = new MatrixProxy(this, name, dim);
        matrixProxy.setMatrixProvider(matrixProvider);
        return matrixProxy;
    }

    public MatrixProxy createProxy(String name, int[] dim, String[] dimNames, MatrixProvider matrixProvider) {
        MatrixProxy matrixProxy = new MatrixProxy(this, name, dim, dimNames);
        matrixProxy.setMatrixProvider(matrixProvider);
        return matrixProxy;
    }

    public MatrixProxy createProxy(String name, List<?>[] semantics, MatrixProvider matrixProvider) {
        MatrixProxy matrixProxy = new MatrixProxy(this, name, semantics);
        matrixProxy.setMatrixProvider(matrixProvider);
        return matrixProxy;
    }

    public MatrixProxy createProxy(String name, List<?>[] semantics, String[] dimNames, MatrixProvider matrixProvider) {
        MatrixProxy matrixProxy = new MatrixProxy(this, name, semantics, dimNames);
        matrixProxy.setMatrixProvider(matrixProvider);
        return matrixProxy;
    }

} // MatrixFactory
