/*
 * #%L
 * JRedmine :: Client
 * 
 * $Id: RedmineRestClient.java 157 2010-10-08 10:23:16Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jredmine/tags/jredmine-1.1.4/jredmine-client/src/main/java/org/nuiton/jredmine/rest/RedmineRestClient.java $
 * %%
 * Copyright (C) 2009 - 2010 Tony Chemit, CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.jredmine.rest;

import org.apache.commons.httpclient.HttpMethod;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.StatusLine;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.io.rest.RestClient;
import org.nuiton.io.rest.RestClientConfiguration;
import org.nuiton.io.rest.RestRequest;
import org.nuiton.io.rest.RestSession;
import org.nuiton.jredmine.model.*;

import java.io.File;
import java.io.IOException;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * Implementation of a {@link RestClient} to access a Redmine server via the
 * {@code redmine_rest} rails plugin.
 *
 * @author chemit
 * @plexus.component role="org.nuiton.io.rest.RestClient" role-hint="redmine"
 * @since 1.0.0
 */
public class RedmineRestClient extends RestClient {

    private static final Log log = LogFactory.getLog(RedmineRestClient.class);
    public static final DateFormat DATE_FORMAT = new SimpleDateFormat("yyyy-MM-dd");
    public static final String GET_ISSUE_LIST = Issue.class.getName() + "list";
    public static final String GET_ISSUE_STATUS_LIST = IssueStatus.class.getName() + "list";
    public static final String GET_ISSUE_PRIORITY_LIST = IssuePriority.class.getName() + "list";
    public static final String GET_ISSUE_CATEGORY_LIST = IssueCategory.class.getName() + "list";
    public static final String GET_ISSUE_TIME_ENTRY_LIST = TimeEntry.class.getName() + "list";
    public static final String GET_PROJECT_LIST = Project.class.getName() + "list";
    public static final String GET_PROJECT_ISSUES = Project.class.getName() + "issues";
    public static final String GET_PROJECT_OPENED_ISSUES = Project.class.getName() + "opened-issues";
    public static final String GET_PROJECT_CLOSED_ISSUES = Project.class.getName() + "closed-issues";
    public static final String GET_USER_PROJECTS = Project.class.getName() + "forUser";
    public static final String GET_PROJECT = Project.class.getName() + "detail";
    public static final String GET_VERSION_LIST = Version.class.getName() + "list";
    public static final String GET_TRACKER_LIST = Tracker.class.getName() + "list";
    public static final String GET_ATTACHMENTS_LIST = Attachment.class.getName() + "list";
    public static final String GET_USER_LIST = User.class.getName() + "list";
    public static final String GET_NEWS_LIST = News.class.getName() + "list";
    public static final String GET_VERSION = Version.class.getName() + "detail";
    public static final String ADD_VERSION = Version.class.getName() + "add";
    public static final String UPDATE_VERSION = Version.class.getName() + "update";
    public static final String NEXT_VERSION = Version.class.getName() + "next";
    public static final String ADD_NEWS = News.class.getName() + "add";
    public static final String ADD_ATTACHMENT = Attachment.class.getName() + "add";
    public static final String ADD_ISSUE_TIME_ENTRY = TimeEntry.class.getName() + "add";
    public static final String LOGIN = "login";
    public static final String LOGOUT = "logout";
    public static final String PING = "ping";

    public RedmineRestClient() {
        super();
    }

    public RedmineRestClient(RestClientConfiguration configuration) {
        super(configuration);
    }

    @Override
    protected void addDefaultRequests() {

        addRequestBuilder(new DefaultRequestBuilder(PING, "ping"));
        addRequestBuilder(new DefaultRequestBuilder(LOGOUT, "logout"));
        addRequestBuilder(new DefaultRequestBuilder(LOGIN, "login") {

            @Override
            public String[] getParameters(Object... args) {
                String login = (String) args[0];
                String password = (String) args[1];
                return new String[]{"username", login, "password", password};
            }
        });

        // with no scope

        addRequestBuilder(new DefaultRequestBuilder(GET_PROJECT_LIST, "get_projects.xml"));
        addRequestBuilder(new DefaultRequestBuilder(GET_USER_PROJECTS, "get_user_projects.xml"));
        addRequestBuilder(new DefaultRequestBuilder(GET_ISSUE_STATUS_LIST, "get_issue_statuses.xml"));
        addRequestBuilder(new DefaultRequestBuilder(GET_ISSUE_PRIORITY_LIST, "get_issue_priorities.xml"));

        // with project scope

        addRequestBuilder(new ProjectScopeRequestBuilder(GET_PROJECT, "get_project.xml"));
        addRequestBuilder(new ProjectScopeRequestBuilder(GET_PROJECT_ISSUES, "get_project_issues.xml"));
        addRequestBuilder(new ProjectScopeRequestBuilder(GET_PROJECT_OPENED_ISSUES, "get_project_opened_issues.xml"));
        addRequestBuilder(new ProjectScopeRequestBuilder(GET_PROJECT_CLOSED_ISSUES, "get_project_closed_issues.xml"));
        addRequestBuilder(new ProjectScopeRequestBuilder(GET_VERSION_LIST, "get_project_versions.xml"));
        addRequestBuilder(new ProjectScopeRequestBuilder(GET_ISSUE_CATEGORY_LIST, "get_issue_categories.xml"));
        addRequestBuilder(new ProjectScopeRequestBuilder(GET_TRACKER_LIST, "get_project_trackers.xml"));
        addRequestBuilder(new ProjectScopeRequestBuilder(GET_USER_LIST, "get_project_users.xml"));
        addRequestBuilder(new ProjectScopeRequestBuilder(GET_NEWS_LIST, "get_project_news.xml"));

        addRequestBuilder(new ProjectScopeRequestBuilder(ADD_VERSION, "add_version.xml") {

            @Override
            public String[] getParameters(Object... args) {
                Version version = (Version) args[1];
                String date = version.getEffectiveDate() == null ? "" : DATE_FORMAT.format(version.getEffectiveDate());
                return new String[]{
                        "version[name]", version.getName(),
                        "version[description]", version.getDescription(),
                        "version[effective_date]", date
                };
            }
        });

        addRequestBuilder(new ProjectScopeRequestBuilder(UPDATE_VERSION, "update_version.xml") {

            @Override
            public String[] getParameters(Object... args) {
                Version version = (Version) args[1];
                String date = version.getEffectiveDate() == null ? "" : DATE_FORMAT.format(version.getEffectiveDate());
                return new String[]{
                        "version[name]", version.getName(),
                        "version[description]", version.getDescription(),
                        "version[effective_date]", date
                };
            }
        });

        addRequestBuilder(new ProjectScopeRequestBuilder(NEXT_VERSION, "next_version.xml") {

            @Override
            public String[] getParameters(Object... args) {

                String oldVersionId = (String) args[1];
                Version version = (Version) args[2];
                String date = version.getEffectiveDate() == null ? "" : DATE_FORMAT.format(version.getEffectiveDate());
                return new String[]{
                        "oldVersionName", oldVersionId,
                        "version[name]", version.getName(),
                        "version[description]", version.getDescription(),
                        "version[effective_date]", date
                };
            }
        });

        addRequestBuilder(new ProjectScopeRequestBuilder(ADD_NEWS, "add_news.xml") {

            @Override
            public String[] getParameters(Object... args) {
                News news = (News) args[1];
                return new String[]{
                        "news[title]", news.getTitle(),
                        "news[summary]", news.getSummary(),
                        "news[description]", news.getDescription()
                };
            }
        });

        // with version scope

        addRequestBuilder(new VersionScopeRequestBuilder(GET_VERSION, "get_version.xml"));
        addRequestBuilder(new VersionScopeRequestBuilder(GET_ISSUE_LIST, "get_version_issues.xml"));
        addRequestBuilder(new VersionScopeRequestBuilder(GET_ATTACHMENTS_LIST, "get_version_attachments.xml"));


        addRequestBuilder(new VersionScopeRequestBuilder(ADD_ATTACHMENT, "add_version_attachment.xml") {

            @Override
            public String[] getParameters(Object... args) {
                String versionId = (String) args[1];
                Attachment attachment = (Attachment) args[2];
                return new String[]{
                        "version_name", versionId,
                        "attachment[description]", attachment.getDescription()
                };
            }

            @Override
            public Map<String, File> getAttachments(Object... args) {
                Map<String, File> upload = new HashMap<String, File>();
                Attachment attachment = (Attachment) args[2];
                upload.put("attachment[file]", attachment.getToUpload());
                return upload;
            }
        });

        // with issue scope

        addRequestBuilder(new IssueScopeRequestBuilder(GET_ISSUE_TIME_ENTRY_LIST, "get_issue_times.xml"));

        addRequestBuilder(new IssueScopeRequestBuilder(ADD_ISSUE_TIME_ENTRY, "add_issue_time.xml") {

            @Override
            public String[] getParameters(Object... args) {
                String issueId = (String) args[1];
                TimeEntry timeEntry = (TimeEntry) args[2];
                Date d = timeEntry.getSpentOn();
                if (d == null) {
                    d = new Date();
                }
                String date = DATE_FORMAT.format(d);
                return new String[]{
                        "issue_id", issueId,
                        //"timeEntry[issue_id]", issueId,
                        "time_entry[activity_id]", timeEntry.getActivityId() + "",
                        "time_entry[spent_on]", date,
                        "time_entry[hours]", timeEntry.getHours() + "",
                        "time_entry[comments]", timeEntry.getComments() == null ? "" : timeEntry.getComments()
                };
            }
        });

    }

    @Override
    protected void open(RestSession session) throws IOException {

        ping(session);

        if (!configuration.isAnonymous()) {
            login(session);
        }
    }

    @Override
    protected void close(RestSession session) throws IOException {

        if (session == null) {
            return;
        }
        try {
            RestRequest request = getRequest(LOGOUT);
            session.setOpen(false);
            session.doRequest(request);
        } finally {
            session.close();
        }
    }

    protected void ping(RestSession session) throws IOException {

        try {

            RestRequest request = getRequest(PING);
            HttpMethod gm = session.doRequest(request);

            StatusLine sl = gm.getStatusLine();
            int statusCode = sl.getStatusCode();
            if (log.isDebugEnabled()) {
                log.debug("status code " + statusCode + " for " + gm.getPath());
            }

            if (statusCode != HttpStatus.SC_OK) {
                gm.releaseConnection();
                throw new IOException("Got error code <" + statusCode + ":" + sl.getReasonPhrase() + "> on " + gm.getPath());
            }

            String content = gm.getResponseBodyAsString();

            boolean ok = "ping".equals(content);
            if (!ok) {
                throw new IOException("can not connect to " + configuration.getRestUrl());
            }
        } catch (IOException ex) {
            throw ex;
        } catch (Exception ex) {
            throw new IOException("could not ping " + configuration.getRestUrl() + " for reason " + ex.getMessage(), ex);
        }
    }

    protected void login(RestSession session) throws IOException {

        RestRequest request = getRequest(LOGIN, configuration.getRestUsername(), configuration.getRestPassword());

        HttpMethod gm = session.doRequest(request);

        StatusLine sl = gm.getStatusLine();
        int statusCode = sl.getStatusCode();
        if (log.isDebugEnabled()) {
            log.debug("status code " + statusCode + " for " + gm.getPath());
        }

        if (statusCode != HttpStatus.SC_OK) {
            gm.releaseConnection();
            throw new IOException("Got error code <" + statusCode + ":" + sl.getReasonPhrase() + "> on " + gm.getPath());
        }

        // ok session is logged in

    }
}
