/*
 * *##% 
 * JRedmine :: Client
 * Copyright (C) 2009 - 2010 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package org.nuiton.jredmine;

import org.nuiton.io.rest.RestClient;
import org.nuiton.io.rest.RestClientConfiguration;
import org.nuiton.jredmine.rest.RedmineRestClient;

/**
 * Technical contract to implements a redmine service which wrap the
 * {@link org.nuiton.jredmine.rest.RedmineRestClient}.
 * <p/>
 * A default implementation is offered in {@link org.nuiton.jredmine.DefaultRedmineServiceImplementor}.
 * <p/>
 * Any concrete implentation of a redmine service should implements this..
 *
 * @author chemit
 * @see org.nuiton.jredmine.DefaultRedmineServiceImplementor
 * @since 1.0.0
 */
public interface RedmineServiceImplementor {

    /**
     * Tests if the service is loogued to the redmine service.
     *
     * @return {@code true} is service is init and loggued to Redmine service,
     *         {@code false} otherwise.
     */
    boolean isInit();

    /**
     * Initialize the service given a redmine client already initialized.
     *
     * @param session the redmine client to be used by the service
     * @return the initialized service
     * @throws RedmineServiceException if any pb
     * @see RedmineRestClient
     */
    RedmineServiceImplementor init(RestClient session) throws RedmineServiceException;

    /**
     * Initialize the service given a client configuration.
     *
     * @param configuration the configuration to be used to init the internal redmine client
     * @return the initialized service
     * @throws RedmineServiceException if any pb
     */
    RedmineServiceImplementor init(RestClientConfiguration configuration) throws RedmineServiceException;

    /**
     * Close the service and destroy any connexion to the redmine service.
     *
     * @throws RedmineServiceException if any pb
     */
    void destroy() throws RedmineServiceException;

    /**
     * Generic method to obtain a single data from a redmine server.
     *
     * @param <T>         the type of data to obtain
     * @param requestName the name of the request to use
     * @param type        the type of data to obtain
     * @param args        the parameters to obtain the data
     * @return the obtained data
     * @throws RedmineServiceException if any pb
     */
    <T> T getData(String requestName, Class<T> type, Object... args) throws RedmineServiceException;

    /**
     * Generic method to obtain a array of data from a redmine server.
     *
     * @param <T>         the type of data to obtain
     * @param requestName the name of the request to use
     * @param type        the type of data to obtain
     * @param args        the parameters to obtain the datas
     * @return the obtained datas
     * @throws RedmineServiceException if any pb
     */
    <T> T[] getDatas(String requestName, Class<T> type, Object... args) throws RedmineServiceException;

    /**
     * Generic method to send a data to a redmine server and return the single
     * data updated from the redmine server.
     *
     * @param requestName the name of the request used
     * @param klazz       the type of data to treate
     * @param args        the parameters of the request
     * @param <T>         the type of data to treate
     * @return the updated data
     * @throws RedmineServiceException if any pb
     */
    <T> T sendData(String requestName, Class<T> klazz, Object... args) throws RedmineServiceException;


    /**
     * Generic method to send a data (or more ) to a redmine server and
     * return the array of data from the redmine server.
     *
     * @param requestName the name of the request used
     * @param klazz       the type of data to treate
     * @param args        the parameters of the request
     * @param <T>         the type of data to treate
     * @return the updated data
     * @throws RedmineServiceException if any pb
     */
    <T> T[] sendDatas(String requestName, Class<T> klazz, Object... args) throws RedmineServiceException;

    /**
     * Checks if the current session is not a anonymous one.
     *
     * @throws IllegalStateException        if service not init
     * @throws RedmineServiceLoginException if not loggued
     * @throws NullPointerException         if something is null
     */
    void checkLoggued() throws IllegalStateException, RedmineServiceLoginException, NullPointerException;

    /**
     * Checks if the current session is anonymous.
     *
     * @param session the session to test
     * @throws RedmineServiceLoginException if not anonymous
     * @throws NullPointerException         if something is null
     */
    void checkNotLoggued(RestClient session) throws RedmineServiceLoginException, NullPointerException;
}
