/* *##% 
 * JAXX Maven plugin
 * Copyright (C) 2008 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*/
package org.nuiton.jaxx.plugin;

import jaxx.compiler.CompiledObjectDecorator;
import jaxx.compiler.CompilerConfiguration;
import jaxx.compiler.I18nHelper;
import jaxx.compiler.JAXXCompiler;
import jaxx.compiler.JAXXEngine;
import jaxx.compiler.beans.BeanInfoUtil;
import jaxx.compiler.binding.DataBindingHelper;
import jaxx.compiler.spi.DefaultInitializer;
import jaxx.compiler.tags.TagManager;
import jaxx.runtime.JAXXContext;
import jaxx.runtime.JAXXObject;
import jaxx.runtime.swing.help.JAXXHelpBroker;
import org.apache.commons.lang.builder.ToStringBuilder;
import org.apache.commons.lang.builder.ToStringStyle;
import org.apache.maven.plugin.MojoExecutionException;
import org.nuiton.i18n.I18n;
import org.nuiton.io.FileUpdaterHelper;
import org.nuiton.io.MirroredFileUpdater;
import org.nuiton.plugin.PluginHelper;

import java.beans.Introspector;
import java.io.File;
import java.lang.String;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Generates some java code from jaxx files.
 *
 * @author chemit
 * @goal generate
 * @phase process-sources
 * @requiresDependencyResolution compile
 * @requiresProject
 */
public class GenerateMojo extends AbstractJaxxMojo implements CompilerConfiguration {

    /**
     * Default includes to use, if none provided
     */
    private static final String[] INCLUDES = {"**\\/*.jaxx"};
    /**
     * Repertoire sources des fichiers jaxx a generer.
     *
     * @parameter expression="${jaxx.src}" default-value="${basedir}/src/main/java"
     */
    protected File src;
    /**
     * Repertoire de destination des fichiers java a generer.
     *
     * @parameter expression="${jaxx.outJava}" default-value="${basedir}/target/generated-sources/java"
     */
    protected File outJava;
    /**
     * pour filter les fichiers a traiter
     *
     * @parameter expression="${jaxx.includes}"
     */
    protected String[] includes;
    /**
     * pour filter les fichiers a ne pas traiter
     *
     * @parameter expression="${jaxx.excludes}"
     */
    protected String[] excludes;
    /**
     * Le compilateur à utiliser (par défaut celui de Swing)
     *
     * @parameter expression="${jaxx.compilerFQN}" default-value="jaxx.compiler.JAXXCompiler"
     */
    protected String compilerFQN;
    /**
     * Le compilateur à utiliser (par défaut celui de Swing)
     *
     * @parameter expression="${jaxx.validatorFQN}" default-value="jaxx.runtime.validator.swing.SwingValidator"
     */
    protected String validatorFQN;
    /**
     * the name of implementation of {@link JAXXContext}to be used on
     * {@link JAXXObject}.
     * <p/>
     * Must not be abstract.
     *
     * @parameter expression="${jaxx.jaxxContextFQN}" default-value="jaxx.runtime.context.DefaultJAXXContext"
     * @required
     */
    protected String jaxxContextFQN;
    /**
     * the FQN of the ui to use for error notification.
     * <p/>
     * If not given, will use the one defined in validator
     *
     * @parameter expression="${jaxx.defaultErrorUIFQN}"
     */
    protected String defaultErrorUIFQN;
    /**
     * the FQN of the ui to use for error notification.
     * <p/>
     * If not given, will use the one defined in validator
     *
     * @parameter expression="${jaxx.defaultDecoratorFQN}" default-value="jaxx.compiler.decorators.DefaultCompiledObjectDecorator"
     * @see CompiledObjectDecorator
     */
    protected String defaultDecoratorFQN;
    /**
     * flag to include in compiler classpath the java sources directories
     * (src and outJava).
     * <p/>
     * By default, false.
     *
     * @parameter expression="${jaxx.addSourcesToClassPath}" default-value="false"
     */
    protected boolean addSourcesToClassPath;
    /**
     * flag to include in compiler classpath the java resources directories
     * (src and outJava).
     * <p/>
     * By default, false.
     *
     * @parameter expression="${jaxx.addResourcesToClassPath}" default-value="false"
     * @since 1.6.0
     */
    protected boolean addResourcesToClassPath;
    /**
     * flag to include in compiler classpath the compile class-path
     * (can only be used in a test phase).
     * <p/>
     * By default, false.
     *
     * @parameter expression="${jaxx.addCompileClassPath}" default-value="false"
     * @since 1.6.0
     */
    protected boolean addCompileClassPath;
    /**
     * flag to include in compiler classpath the project compile classpath.
     * <p/>
     * By default, false.
     *
     * @parameter expression="${jaxx.addProjectClassPath}" default-value="false"
     */
    protected boolean addProjectClassPath;
    /**
     * A flag to mark themojo to be used in a test phase. This will permits
     * to add generated sources in test compile roots.
     *
     * @parameter expression="${jaxx.testPhase}" default-value="false"
     * @since 1.6.0
     */
    protected boolean testPhase;
    /**
     * to make compiler i18nable, says add the
     * {@link I18n#_(String, Object...)} method invocation on
     * {@link I18nHelper#I18N_ATTRIBUTES} attributes.
     *
     * @parameter expression="${jaxx.i18nable}" default-value="true"
     * @see I18nHelper
     */
    protected boolean i18nable;
    /**
     * pour optimizer le code compile ou genere ?
     *
     * @parameter expression="${jaxx.optimize}" default-value="false"
     */
    protected boolean optimize;
    /**
     * flag to add logger to each generated jaxx file.
     * <p/>
     * By default, always add it.
     *
     * @parameter expression="${jaxx.addLogger}" default-value="true"
     */
    protected boolean addLogger;
    /**
     * flag to keep compilers after the generate operation (usefull for tests.
     * <p/>
     * By default, always reset.
     *
     * @parameter expression="${jaxx.resetAfterCompile}" default-value="true"
     */
    protected boolean resetAfterCompile;
    /**
     * extra path to be added in {@link Introspector#setBeanInfoSearchPath(String[])}.
     * <p/>
     * add beanInfoSearchPath to be registred by
     * {@link BeanInfoUtil#addJaxxBeanInfoPath(String...)}
     * <p/>
     * and then will be use by {@link DefaultInitializer#initialize()}.
     * <p/>
     * <p/>
     * This permit to use real beanInfo of imported graphic libraries.
     *
     * @parameter expression="${jaxx.beanInfoSearchPath}"
     */
    protected String[] beanInfoSearchPath;
    /**
     * list of fqn of class toimport for all generated jaxx files
     *
     * @parameter expression="${jaxx.extraImportList}"
     */
    protected String extraImportList;
    /**
     * a flag to use UIManager to retreave icons.
     *
     * @parameter expression="${jaxx.useUIManagerForIcon}" default-value="false"
     */
    protected boolean useUIManagerForIcon;
    /**
     * flag to activate profile mode.
     * <p/>
     * By default, not active.
     *
     * @parameter expression="${jaxx.profile}" default-value="false"
     */
    protected boolean profile;
    /**
     * To show detected bindings.
     * <p/>
     * By default, do not show them.
     *
     * @parameter expression="${jaxx.showBinding}" default-value="false"
     * @since 2.0.0
     */
    protected boolean showBinding;
    /**
     * the FQN of help broker
     * <p/>
     * By default, use the JAXX implementation {@link JAXXHelpBroker}.
     *
     * @parameter expression="${jaxx.helpBrokerFQN}" default-value="jaxx.runtime.swing.help.JAXXHelpBroker"
     * @since 1.3
     */
    protected String helpBrokerFQN;
    /**
     * detected jaxx files in {@link #init()} method
     */
    protected String[] files;
    /**
     * file updater used to detect jaxx files.
     * <p/>
     * <b>Note:</b> if {@link #verbose} flag is on, will ne be used
     */
    protected MirroredFileUpdater updater;
    /**
     *
     */
    private Class<?> defaultErrorUIClass;
    /**
     *
     */
    private Class<?> validatorClass;
    /**
     *
     */
    private Class<? extends CompiledObjectDecorator> defaultDecoratorClass;
    /**
     *
     */
    private Class<? extends JAXXContext> jaxxContextClass;
    /**
     *
     */
    private Class<? extends JAXXCompiler> compilerClass;
    /**
     *
     */
    private String[] extraImports;
    /**
     *
     */
    private boolean nofiles;
    /**
     *
     */
    protected ClassLoader cl;
    /**
     * JAXX engine
     */
    private JAXXEngine engine;

    @SuppressWarnings("unchecked")
    @Override
    public void init() throws Exception {

        fixCompileSourceRoots();

        if (includes == null || includes.length == 0) {
            // use default includes
            includes = INCLUDES;
        }
        updater = FileUpdaterHelper.newJaxxFileUpdater(src, outJava);

        Map<File, String[]> result = new HashMap<File, String[]>();
        getFilesToTreateForRoots(
                includes,
                excludes,
                Arrays.asList(src.getAbsolutePath()),
                result,
                isForce() ? null : updater);

        files = result.get(src);

        nofiles = files == null || files.length == 0;
        if (nofiles) {
            return;
        }

        cl = initClassLoader(getProject(),
                src,
                addSourcesToClassPath,
                testPhase,
                addResourcesToClassPath,
                addCompileClassPath,
                addProjectClassPath);

        Thread.currentThread().setContextClassLoader(cl);

        compilerClass = (Class<? extends JAXXCompiler>)
                Class.forName(compilerFQN, false, cl);
        defaultDecoratorClass = (Class<? extends CompiledObjectDecorator>)
                Class.forName(defaultDecoratorFQN, false, cl);
        jaxxContextClass = (Class<? extends JAXXContext>)
                Class.forName(jaxxContextFQN, false, cl);
        if (!JAXXContext.class.isAssignableFrom(jaxxContextClass)) {
            throw new MojoExecutionException(
                    "jaxxContextFQN must be an implementation of " +
                    JAXXContext.class + " but was : " + jaxxContextClass);
        }
        validatorClass = Class.forName(validatorFQN, false, cl);

        if (defaultErrorUIFQN != null && !defaultErrorUIFQN.trim().isEmpty()) {
            defaultErrorUIClass = Class.forName(defaultErrorUIFQN, false, cl);
        }

        if (beanInfoSearchPath != null && beanInfoSearchPath.length > 0) {
            // register extra path
            BeanInfoUtil.addJaxxBeanInfoPath(beanInfoSearchPath);
        }

        createDirectoryIfNecessary(getTargetDirectory());

        // compute extra imports (can not use java classes since some of
        // imports can not be still compiled)
        if (extraImportList != null && !extraImportList.isEmpty()) {
            String[] imports = extraImportList.split(",");
            int i = 0;
            for (String importS : imports) {
                imports[i++] = importS.trim();
            }
            if (isVerbose()) {
                getLog().info("extra imports " + Arrays.toString(imports));
            }
            extraImports = imports;
        }

        if (isVerbose()) {
            getLog().info(toString());
            getLog().info("includes : " + Arrays.toString(includes));
            for (String file : files) {
                getLog().info("will parse " + file);
            }
        }
    }

    @Override
    protected boolean checkSkip() {
        if (nofiles) {
            getLog().info("Nothing to generate - all files are up to date.");
            return false;
        }
        return true;
    }

    @Override
    public void doAction() throws Exception {

        getLog().info("Detects " + files.length + " modified jaxx file(s). ");

        if (showBinding) {
            DataBindingHelper.SHOW_LOG = showBinding;
        }
        try {
            long t0 = System.nanoTime();

            // force compiler init from here, not in a static block
            TagManager.reset(isVerbose());

            engine = JAXXEngine.newLaunchor(src, files, this);
            int nbFiles = engine.run();
            report(engine);
            if (nbFiles == -1) {
                throw new MojoExecutionException(
                        "Aborting due to errors reported by jaxxc");
            }
            getLog().info("Generated " + nbFiles + " file(s) in " +
                          PluginHelper.convertTime(System.nanoTime() - t0));

        } catch (MojoExecutionException e) {
            getLog().error(e);
            throw e;
        } catch (Exception e) {
            //getLog().error(e);
            Throwable e2 = e;
            while (e2.getCause() != null) {
                e2 = e.getCause();
            }
            getLog().error(e2);

            throw new MojoExecutionException(e2.getMessage(), e2);
        } finally {
            DataBindingHelper.SHOW_LOG = false;
        }
    }

    @Override
    public File getTargetDirectory() {
        return outJava;
    }

    @Override
    public void setTargetDirectory(File targetDirectory) {
        outJava = targetDirectory;
    }

    @Override
    public boolean getOptimize() {
        return optimize;
    }

    @Override
    public boolean isI18nable() {
        return i18nable;
    }

    @Override
    public boolean isUseUIManagerForIcon() {
        return useUIManagerForIcon;
    }

    @Override
    public boolean isAddLogger() {
        return addLogger;
    }

    @Override
    public Class<? extends JAXXContext> getJaxxContextClass() {
        return jaxxContextClass;
    }

    @Override
    public String[] getExtraImports() {
        return extraImports;
    }

    @Override
    public boolean isResetAfterCompile() {
        return resetAfterCompile;
    }

    @Override
    public boolean isOptimize() {
        return optimize;
    }

    @Override
    public Class<?> getDefaultErrorUI() {
        return defaultErrorUIClass;
    }

    @Override
    public ClassLoader getClassLoader() {
        return cl;
    }

    @Override
    public Class<? extends JAXXCompiler> getCompilerClass() {
        return compilerClass;
    }

    @Override
    public Class<? extends CompiledObjectDecorator> getDefaultDecoratorClass() {
        return defaultDecoratorClass;
    }

    @Override
    public boolean isProfile() {
        return profile;
    }

    @Override
    public boolean isGenerateHelp() {
        return generateHelp;
    }

    @Override
    public String getHelpBrokerFQN() {
        return helpBrokerFQN;
    }

    @Override
    public Class<?> getValidatorClass() {
        return validatorClass;
    }

    public JAXXEngine getEngine() {
        return engine;
    }

    @Override
    public String toString() {
        return ToStringBuilder.reflectionToString(
                this, ToStringStyle.MULTI_LINE_STYLE);
    }

    protected void fixCompileSourceRoots() {
        //FIXME TC20091222 : why this test ? should always have a project ?
        // even in tests...
        if (getProject() == null) {
            // no project defined, can not fix anything
            // this case could appear if we wanted to do some tests of the plugin
            return;
        }       

        if (testPhase) {
            addTestCompileSourceRoots(getTargetDirectory());
        } else {
            addCompileSourceRoots(getTargetDirectory());
        }
    }

    protected void report(JAXXEngine engine) {
        List<String> warnings = engine.getWarnings();
        if (!warnings.isEmpty()) {
            StringBuilder buffer = new StringBuilder(
                    "JAXX detects " +
                    (warnings.size() == 1 ? "1 warning" :
                     warnings.size() + " warnings"));
            buffer.append(" :");
            for (String s : warnings) {
                buffer.append("\n").append(s);
            }
            getLog().warn(buffer.toString());
        }
        List<String> errors = engine.getErrors();
        if (!errors.isEmpty()) {
            StringBuilder buffer = new StringBuilder(
                    "JAXX detects " +
                    (errors.size() == 1 ? "1 error" :
                     errors.size() + " errors"));
            buffer.append(" :");
            for (String s : errors) {
                buffer.append("\n").append(s);
            }
            getLog().error(buffer.toString());
        }
    }
}
