/* *##%
 * Copyright (C) 2007
 *     JaxxPlugin, Code Lutin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *##%*/
package org.nuiton.jaxx;

import org.apache.maven.plugin.MojoFailureException;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.Enumeration;
import java.util.List;
import java.util.Properties;
import java.util.Set;
import java.util.Stack;
import org.apache.maven.model.Resource;
import org.nuiton.i18n.I18n;
import org.nuiton.util.FileUtil;
import org.nuiton.util.SortedProperties;
import org.xml.sax.Attributes;
import org.xml.sax.ContentHandler;
import org.xml.sax.InputSource;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.XMLReaderFactory;
import static org.nuiton.i18n.I18n._;

/**
 * Mojo to generate javax help stuff for your project.
 *
 * HelpIds should have been discovered by the JaxxMojo.
 *
 * @author chemit
 * @goal generate-help
 * @phase process-sources
 * 
 * @requiresProject
 * @requiresDependencyResolution compile
 * @since 1.3
 */
public class JaxxHelpGeneratorMojo extends AbstractJaxxMojo {

    /**
     * The directory where to generate javaHelp skeleton files.
     * <p/>
     * Is required if generateHelp is on.
     *
     * @parameter expression="${jaxx.helpTarget}" alias="target" default-value="${maven.src.dir}/main/help"
     * @required
     * 
     * @since 1.3
     */
    protected File target;
    /**
     * The locale to generate for help.
     *
     * By default, stay in France.
     *
     * @parameter expression="${jaxx.locale}" default-value="fr"
     * @required
     *
     * @since 1.3
     */
    protected String locale;
    /**
     * The package where to generate i18n java file.
     *
     * @parameter expression="${jaxx.packageName}" default-value="${project.groupId}"
     * @required
     *
     * @since 1.3
     */
    protected String packageName;
    /**
     * The file name of the helpset to generate.
     *
     * @parameter expression="${jaxx.helpSetFileName}" default-value="${jaxx.helpSetName}.hs"
     * @required
     *
     * @since 1.3
     */
    protected String helpsetFileName;
    /**
     * The file name of the helpset map to generate.
     *
     * @parameter expression="${jaxx.mapFileName}" default-value="${jaxx.helpSetName}Map.jhm"
     * @required
     *
     * @since 1.3
     */
    protected String mapFileName;
    /**
     * The file name of the helpset index to generate.
     *
     * @parameter expression="${jaxx.indexFileName}" default-value="${jaxx.helpSetName}Index.xml"
     * @required
     *
     * @since 1.3
     */
    protected String indexFileName;
    /**
     * The file name of the helpset toc to generate.
     *
     * @parameter expression="${jaxx.tocFileName}" default-value="${jaxx.helpSetName}TOC.xml"
     * @required
     *
     * @since 1.3
     */
    protected String tocFileName;
    /**
     * The file name of the i18n java file to generate.
     *
     * @parameter expression="${jaxx.i8nFileName}" default-value="${jaxx.helpSetName}I18n.java"
     * @required
     *
     * @since 1.3
     */
    protected String i8nFileName;
    /**
     * The template used to generate helpset file.
     *
     * Must be an existing file or a ressource in classp-ath
     * 
     * @parameter expression="${jaxx.helpSetTemplate}" default-value="/defaultHelpSet.hs.vm"
     * @required
     *
     * @since 1.3
     */
    protected File helpSetTemplate;
    /**
     * The template used to generate helpset map file.
     * 
     * Must be an existing file or a ressource in classp-ath
     * 
     * @parameter expression="${jaxx.mapTemplate}" default-value="/defaultMap.jhm.vm"
     * @required
     *
     * @since 1.3
     */
    protected File mapTemplate;
    /**
     * The template used to generate helpset index file.
     * 
     * Must be an existing file or a ressource in classp-ath
     * 
     * @parameter expression="${jaxx.indexTemplate}" default-value="/defaultIndex.xml.vm"
     * @required
     *
     * @since 1.3
     */
    protected File indexTemplate;
    /**
     * The template used to generate helpset toc file.
     * 
     * Must be an existing file or a ressource in classp-ath
     * 
     * @parameter expression="${jaxx.tocTemplate}" default-value="/defaultToc.xml.vm"
     * @required
     *
     * @since 1.3
     */
    protected File tocTemplate;
    /**
     * The template used to generate helpset content file.
     *
     * Must be an existing file or a ressource in classp-ath
     *
     * @parameter expression="${jaxx.contentTemplate}" default-value="/defaultContent.html.vm"
     * @required
     *
     * @since 1.3
     */
    protected File contentTemplate;
    /**
     * The template used to generate helpset content file.
     *
     * Must be an existing file or a ressource in classp-ath
     *
     * @parameter expression="${jaxx.i18nTemplate}" default-value="/defaultI18n.java.vm"
     * @required
     *
     * @since 1.3
     */
    protected File i18nTemplate;
    /**
     * The help ids discovered by Jaxx compilation
     */
    protected Properties helpIds;
    private static final String AUTOREMOVE_LINE = "REMOVE THS LINE TO DISABLE AUTO-REGENERATE THE FILE";

    @Override
    public void init() throws Exception {

        if (project != null && ("pom".equals(project.getPackaging()) || "site".equals(project.getPackaging()))) {
            // nothing to be done for this type of packaging
            skip = true;
            getLog().info("skip generate goal for packaging " + project.getPackaging());
            return;
        }

        if (!helpIdStore.exists()) {
            skip = true;
            getLog().info("no helpIdStore to react at " + helpIdStore);
            return;
        }

        // check there is some bundle
        if (locale == null) {
            throw new MojoFailureException("you must set the bundles property.");
        }
        // check there is some bundle
        if (target == null) {
            throw new MojoFailureException("you must set the target property.");
        }
        // check ressources
        checkResource(helpSetTemplate);
        checkResource(mapTemplate);
        checkResource(indexTemplate);
        checkResource(tocTemplate);
        checkResource(contentTemplate);

        if (!target.exists()) {
            getLog().info("mkdir " + target);
            target.mkdirs();
        }


        helpIds = new SortedProperties();

        InputStream stream = new FileInputStream(helpIdStore);

        helpIds.load(stream);

        stream.close();

        if (helpIds.isEmpty()) {

            // no ids detected
            getLog().warn("no helpIds detected, will skip.");
            skip = true;
            return;
        }

        skip = false;
    }

    @Override
    public void doAction() throws Exception {

        if (i18nable) {

            List<URL> lUrls = new java.util.ArrayList<URL>();
            List resources = project.getResources();
            for (Object o : resources) {
                Resource resource = (Resource) o;
                lUrls.add(new File(resource.getDirectory()).toURI().toURL());
            }
            I18n.setExtraURL(lUrls.toArray(new URL[lUrls.size()]));
            I18n.init(locale, null);
        }

        File file;

        Properties env = new Properties();

        env.put("helpSetName", helpSetName);
        env.put("helpSetFileName", helpsetFileName);
        env.put("mapFileName", mapFileName);
        env.put("indexFileName", indexFileName);
        env.put("tocFileName", tocFileName);
        env.put("separator", "    ");
        env.put("autoremoveLine", AUTOREMOVE_LINE);


        int touchedFiles = 0;

        // ---------------------------------------------------------------
        // --- main helpset file -----------------------------------------
        // ---------------------------------------------------------------


        file = new File(target, helpsetFileName);

        boolean doCreate = generateHelSetFile(file, env);

        if (doCreate) {
            touchedFiles++;
        }

        // ---------------------------------------------------------------
        // --- helpset map file ------------------------------------------
        // ---------------------------------------------------------------

        file = new File(target, mapFileName);

        Properties mergedHelpIds = generateMapFile(file, env);
        touchedFiles++;

        // ---------------------------------------------------------------
        // --- helpset index file ----------------------------------------
        // ---------------------------------------------------------------

        file = new File(target, indexFileName);

        NodeItem indexRootItem = generateIndexFile(file, env);
        touchedFiles++;

        // ---------------------------------------------------------------
        // --- helpset toc file ------------------------------------------
        // ---------------------------------------------------------------

        file = new File(target, tocFileName);

        NodeItem tocRootItem = generateTocFile(file, env);
        touchedFiles++;

        // ---------------------------------------------------------------
        // --- helpset content files -------------------------------------
        // ---------------------------------------------------------------

        TemplateGenerator gen = prepareGenerator(contentTemplate);

        Enumeration keys = helpIds.keys();

        while (keys.hasMoreElements()) {
            String key = (String) keys.nextElement();
            String url = (String) helpIds.get(key);
            url = helpSetName + File.separator + url;

            File f = new File(target, url);

            boolean exist = f.exists();

            if (exist) {
                // check if there is a autoremoveLine in content
                String content = FileUtil.readAsString(f);
                if (!content.contains(AUTOREMOVE_LINE)) {
                    // no regenerate marker detected, so skip this file
                    if (verbose) {
                        getLog().debug("skip existing file " + f);
                    }
                    continue;
                }
            }

            f.getParentFile().mkdirs();

            if (verbose) {
                if (exist) {
                    getLog().info("regenerate content file " + f);
                } else {
                    getLog().info("generate   content file " + f);
                }
            }

            env.put("helpId", key);
            String i18n = helpsetI18nPrefix + key + helpsetTitleI18nSuffix;
            env.put("helpIdTitle", _(i18n));
            env.put("helpIdUrl", url);

            gen.generate(env, f);
            touchedFiles++;
        }

        // ---------------------------------------------------------------
        // --- i18n file -------------------------------------------------
        // ---------------------------------------------------------------

        String path = packageName.replaceAll("\\.", File.separator);
        path += File.separator + i8nFileName;
        file = new File(outJava, path);
        generateI18nFile(file, env, mergedHelpIds, indexRootItem, tocRootItem);
        touchedFiles++;

        getLog().info(touchedFiles + " file(s) treated.");
    }

    protected void generateI18nFile(File file, Properties env, Properties mergedHelpIds, NodeItem indexRootItem, NodeItem tocRootItem) throws Exception {

        boolean create = !file.exists();

        if (!file.getParentFile().exists()) {
            file.getParentFile().mkdirs();
        }

        Set<String> keys = new java.util.HashSet<String>();

        for (Object k : mergedHelpIds.keySet()) {
            String key = helpsetI18nPrefix + k + helpsetTitleI18nSuffix;
            keys.add(key);
        }
        indexRootItem.extractI18n(keys, helpsetI18nPrefix, helpsetIndexI18nSuffix);
        tocRootItem.extractI18n(keys, helpsetI18nPrefix, helpsetTocI18nSuffix);

        env.put("keys", keys);
        env.put("packageName", packageName);
        String className = file.getName();
        int index = className.lastIndexOf(".");
        className = className.substring(0, index);
        env.put("className", className);

        if (verbose) {
            if (create) {
                getLog().info("generate i18n java file " + file);
            } else {
                getLog().info("udpate i18n java file " + file);
            }
        }

        doGen(i18nTemplate, file, env);

    }

    protected boolean generateHelSetFile(File file, Properties env) throws Exception {

        if (file.exists()) {
            // check the auto removeline presence
            String content = FileUtil.readAsString(file);
            if (!content.contains(AUTOREMOVE_LINE)) {
                // no regenerate marker detected, so skip this file
                if (verbose) {
                    getLog().info("skip existing helpset main file " + file);
                }
                return false;
            }
        }

        if (verbose) {
            if (file.exists()) {
                getLog().info("regenerate helpset main file " + file);
            } else {
                getLog().info("generate helpset main file " + file);
            }
        }
        doGen(helpSetTemplate, file, env);
        return true;
    }

    protected Properties generateMapFile(File file, Properties env) throws Exception {

        boolean create;

        Properties mergedHelpIds = null;
        if (file.exists()) {

            // get back the exisiting data and merge it with incoming ones

            if (verbose) {
                getLog().info("loading existing helpset map file " + file);
            }

            mergedHelpIds = getExistingHelpIds(file);
            create = false;

        } else {

            mergedHelpIds = new SortedProperties();
            create = true;
        }

        // inject new helpIds

        for (Object k : helpIds.keySet()) {
            mergedHelpIds.put(k, helpSetName + "/" + helpIds.get(k));
        }

        if (!mergedHelpIds.contains("top")) {
            // on ajoute une entree vers le root du helpset

            String topUrl = helpSetName + ".html";
            helpIds.put("top", topUrl);
            mergedHelpIds.put("top", helpSetName + "/" + topUrl);
            if (verbose) {
                getLog().debug("add top entry with url " + topUrl);
            }
        }

        if (verbose) {
            if (create) {
                getLog().info("generate helpset map file " + file);
            } else {
                getLog().info("udpate helpset map file " + file);
            }
        }

        env.put("helpIds", mergedHelpIds);
        doGen(mapTemplate, file, env);
        env.remove("helpIds");
        return mergedHelpIds;
    }

    protected NodeItem generateIndexFile(File file, Properties env) throws Exception {
        NodeItem rootItem = null;

        boolean create;

        if (file.exists()) {

            create = false;

            rootItem = getExistingItems("indexitem", file);
        } else {
            create = true;
        }

        if (rootItem == null) {
            rootItem = new NodeItem("top", helpSetName);
        }

        // inject new index entries

        for (Object k : helpIds.keySet()) {
            NodeItem toc = rootItem.findChild(k + "");
            if (verbose) {
                getLog().debug("index " + k + " : " + toc);
            }
        }

        //String prefix = helpsetI18nPrefix;
        //String prefix = helpsetI18nPrefix + helpSetName + ".";
        rootItem.applyI18n(helpsetI18nPrefix, helpsetIndexI18nSuffix);

        if (verbose) {
            if (create) {
                getLog().info("generate helpset index file " + file);
            } else {
                getLog().info("udpate helpset index file " + file);
            }
        }

        env.put("rootItem", rootItem);
        doGen(indexTemplate, file, env);
        env.remove("rootItem");
        return rootItem;
    }

    protected NodeItem generateTocFile(File file, Properties env) throws Exception {
        NodeItem rootItem = null;

        boolean create;

        if (file.exists()) {

            create = false;

            rootItem = getExistingItems("tocitem", file);
        } else {
            create = true;
        }

        if (rootItem == null) {
            rootItem = new NodeItem("top", helpSetName);
        }
        // inject new toc entries

        for (Object k : helpIds.keySet()) {
            NodeItem toc = rootItem.findChild(k + "");
            if (verbose) {
                getLog().debug("toc " + k + " : " + toc);
            }
        }

        //String prefix = helpsetI18nPrefix + helpSetName + ".";
        rootItem.applyI18n(helpsetI18nPrefix, helpsetTocI18nSuffix);

        if (verbose) {
            if (create) {
                getLog().info("generate helpset toc file " + file);
            } else {
                getLog().info("udpate helpset toc file " + file);
            }
        }

        env.put("rootItem", rootItem);
        doGen(tocTemplate, file, env);
        env.remove("rootItem");
        return rootItem;
    }

    protected void doGen(File template, File f, Properties env) throws Exception {
        TemplateGenerator gen = prepareGenerator(template);
        gen.generate(env, f);
    }

    protected TemplateGenerator prepareGenerator(File template) throws Exception {
        URL templateURL = getTemplate(template);

        if (verbose) {
            getLog().info("using template " + templateURL);
        }
        TemplateGenerator gen = new TemplateGenerator(project, templateURL);
        return gen;
    }

    protected URL getTemplate(File f) throws IOException {
        URL r = null;
        InputStream s = null;
        if (f.exists()) {
            r = f.toURI().toURL();
        } else {
            r = getClass().getResource(f.toString());
        }
        return r;
    }

    protected void checkResource(File f) throws MojoFailureException {
        if (!f.exists()) {
            // test in classPath
            InputStream r = getClass().getResourceAsStream(f.toString());
            if (r == null) {
                throw new MojoFailureException("could not find ressource " + f);
            }
        }
    }

    protected Properties getExistingHelpIds(File file) throws SAXException, IOException {

        final Properties result = new SortedProperties();

        XMLReader parser = XMLReaderFactory.createXMLReader();

        parser.setContentHandler(new ContentHandlerAdapter() {

            String target;
            String url;

            @Override
            public void startElement(String uri, String localName, String qName, Attributes atts) throws SAXException {
                if ("mapID".equals(localName)) {
                    target = atts.getValue("target");
                    url = atts.getValue("url");
                    if (verbose) {
                        getLog().debug("detect map entry : " + target + " : " + url);
                    }
                    result.put(target, url);
                }
            }
        });

        InputStream s = new FileInputStream(file);
        try {
            parser.parse(new InputSource(s));
        } finally {
            s.close();
        }
        return result;
    }

    protected NodeItem getExistingItems(String tagName, File file) throws SAXException, IOException {

        XMLReader parser = XMLReaderFactory.createXMLReader();
        NodeItemHandler handler = new NodeItemHandler(tagName);

        parser.setContentHandler(handler);

        NodeItem rootItem = null;
        InputStream s = new FileInputStream(file);
        try {
            parser.parse(new InputSource(s));
            rootItem = handler.rootItem;
        } finally {
            s.close();
        }
        return rootItem;
    }

    static class NodeItemHandler extends ContentHandlerAdapter {

        NodeItem rootItem;
        NodeItem currentItem;
        final Stack<NodeItem> stack;
        final String tagName;

        public NodeItemHandler(String tagName) {
            this.tagName = tagName;
            this.stack = new Stack<NodeItem>();
        }

        @Override
        public void startDocument() throws SAXException {
            rootItem = new NodeItem("top", null);
        }

        @Override
        public void startElement(String uri, String localName, String qName, Attributes atts) throws SAXException {
            if (tagName.equals(localName)) {

                String target = atts.getValue("target");
                String text = atts.getValue("text");

                // debut d'un item
                if (currentItem == null) {
                    // premier item
                    if (rootItem.getTarget().equals(target)) {
                        // le premier item est bien top
                        //rootItem.setText(text);
                        currentItem = rootItem;
                    } else {
                        // le premier noeud n'est pas top
                        // en l'encapsule
                        stack.push(rootItem);
                        currentItem = new NodeItem(target, text);
                        rootItem.addChild(currentItem);
                    }
                } else {
                    NodeItem newItem = new NodeItem(target, text);
                    currentItem.addChild(newItem);
                    currentItem = newItem;
                }
                currentItem.adjustTarget();
                stack.push(currentItem);

            }
        }

        @Override
        public void endElement(String uri, String localName, String qName) throws SAXException {
            if (tagName.equals(localName)) {
                // fin d'un item
                stack.pop();
                if (!stack.isEmpty()) {
                    currentItem = stack.peek();
                }
            }
        }
    }

    static class ContentHandlerAdapter implements ContentHandler {

        @Override
        public void setDocumentLocator(Locator locator) {
        }

        @Override
        public void startDocument() throws SAXException {
        }

        @Override
        public void endDocument() throws SAXException {
        }

        @Override
        public void startPrefixMapping(String prefix, String uri) throws SAXException {
        }

        @Override
        public void endPrefixMapping(String prefix) throws SAXException {
        }

        @Override
        public void startElement(String uri, String localName, String qName, Attributes atts) throws SAXException {
        }

        @Override
        public void endElement(String uri, String localName, String qName) throws SAXException {
        }

        @Override
        public void characters(char[] ch, int start, int length) throws SAXException {
        }

        @Override
        public void ignorableWhitespace(char[] ch, int start, int length) throws SAXException {
        }

        @Override
        public void processingInstruction(String target, String data) throws SAXException {
        }

        @Override
        public void skippedEntity(String name) throws SAXException {
        }
    }
}