package jaxx.runtime.swing.editor.bean;

/*
 * #%L
 * JAXX :: Widgets
 * $Id: BeanDoubleListHandler.java 2536 2012-12-14 09:22:33Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jaxx/tags/jaxx-2.5.8/jaxx-widgets/src/main/java/jaxx/runtime/swing/editor/bean/BeanDoubleListHandler.java $
 * %%
 * Copyright (C) 2008 - 2012 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import jaxx.runtime.swing.renderer.DecoratorListCellRenderer;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.decorator.JXPathDecorator;
import org.nuiton.util.decorator.MultiJXPathDecorator;

import javax.swing.JList;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListDataEvent;
import javax.swing.event.ListDataListener;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import java.awt.event.MouseEvent;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.List;

/**
 * The handler of a {@link BeanDoubleList}.
 * <p/>
 *
 * @param <O> the type of the objects contained in the list.
 * @author kmorin <morin@codelutin.com>
 * @see BeanDoubleList
 * @since 2.5.8
 */
public class BeanDoubleListHandler<O> {

    private static final Log log = LogFactory.getLog(BeanDoubleListHandler.class);

    /** the mutator method on the property of boxed bean in the ui */
    protected Method mutator;

    /** ui of the handler */
    protected BeanDoubleList<O> ui;

    /** the decorator of data */
    protected MultiJXPathDecorator<O> decorator;

    public BeanDoubleListHandler(BeanDoubleList<O> ui) {
        this.ui = ui;
    }

    /**
     * Initializes the handler of the UI
     *
     * @param decorator the decorator to use to display the data nicely
     * @param universe  the list of all the available items
     * @param selected  the list of selected items
     */
    public void init(JXPathDecorator<O> decorator, List<O> universe, List<O> selected) {
        this.decorator = BeanUIUtil.createDecorator(decorator);

        mutator = BeanUIUtil.getMutator(ui.getBean(), ui.getProperty());

        ui.getModel().setUniverse(universe);
        ui.getModel().setSelected(selected);

        JList universeList = ui.getUniverseList();
        JList selectedList = ui.getSelectedList();

        universeList.setCellRenderer(new DecoratorListCellRenderer(this.decorator));
        selectedList.setCellRenderer(new DecoratorListCellRenderer(this.decorator));

        // When universe list selection model changed, update the add button enabled property
        universeList.getSelectionModel().addListSelectionListener(new ListSelectionListener() {
            @Override
            public void valueChanged(ListSelectionEvent e) {
                ListSelectionModel source = (ListSelectionModel) e.getSource();
                ui.getModel().setAddEnabled(!source.isSelectionEmpty());
            }
        });

        // When selected list selection model changed, update the add button enabled property
        selectedList.getSelectionModel().addListSelectionListener(new ListSelectionListener() {
            @Override
            public void valueChanged(ListSelectionEvent e) {
                ListSelectionModel source = (ListSelectionModel) e.getSource();
                ui.getModel().setRemoveEnabled(!source.isSelectionEmpty());
            }
        });

        // When selected list model changed, push back selected list to bean
        selectedList.getModel().addListDataListener(new ListDataListener() {
            public void intervalAdded(ListDataEvent e) {
                fireSelectionUpdate();
            }

            public void intervalRemoved(ListDataEvent e) {
                fireSelectionUpdate();
            }

            public void contentsChanged(ListDataEvent e) {
                fireSelectionUpdate();
            }
        });
    }

    /**
     * When universe list was double clicked, move selected items to selected list.
     *
     * @param event mouse event
     */
    public void onUniverseListClicked(MouseEvent event) {
        JList universeList = ui.getUniverseList();
        if (event.getClickCount() == 2) {
            int index = universeList.locationToIndex(event.getPoint());
            O item = (O) universeList.getModel().getElementAt(index);
            ui.getModel().addToSelected(item);
        }
    }


    /**
     * When selected list was double clicked, move selected items to universe list.
     *
     * @param event mouse event
     */
    public void onSelectedListClicked(MouseEvent event) {

        JList selectedList = ui.getSelectedList();

        if (event.getClickCount() == 2) {
            int index = selectedList.locationToIndex(event.getPoint());
            O item = (O) selectedList.getModel().getElementAt(index);
            ui.getModel().removeFromSelected(item);
        }
    }

    /** When add button was hit, move selected items (from universe list) to selected list. */
    public void select() {
        List<O> selection = (List<O>) Arrays.asList(ui.getUniverseList().getSelectedValues());
        ui.getModel().addToSelected(selection);
    }

    /** When remove button was hit, move selected items (from selected list) to universe list. */
    public void unselect() {
        List<O> selection = (List<O>) Arrays.asList(ui.getSelectedList().getSelectedValues());
        ui.getModel().removeFromSelected(selection);
    }

    protected void fireSelectionUpdate() {
        if (ui.getBean() == null) {
            return;
        }
        try {
            mutator.invoke(ui.getBean(), ui.getModel().getSelected());
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

}
