
package jaxx.runtime.validator.swing;

/*
 * #%L
 * JAXX :: Validator
 * $Id: SwingValidatorMessageWidget.java 2560 2013-02-07 22:26:31Z kmorin $
 * $HeadURL: http://svn.nuiton.org/svn/jaxx/tags/jaxx-2.5.10/jaxx-validator/src/main/java/jaxx/runtime/validator/swing/SwingValidatorMessageWidget.java $
 * %%
 * Copyright (C) 2008 - 2013 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import java.awt.Component;
import java.awt.Point;
import jaxx.runtime.SwingUtil;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.swing.AbstractAction;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JRootPane;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JToggleButton;
import javax.swing.KeyStroke;
import javax.swing.ListSelectionModel;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.HierarchyBoundsAdapter;
import java.awt.event.HierarchyEvent;
import java.awt.event.KeyEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import jaxx.runtime.swing.ComponentMover;
import jaxx.runtime.swing.ComponentResizer;
import org.jdesktop.swingx.JXTitledPanel;
import org.nuiton.validator.NuitonValidatorScope;

import static org.nuiton.i18n.I18n._;
import static org.nuiton.i18n.I18n.n_;

/**
 * Button which opens a popup containing a table with the errors found
 * by registered validators.
 *
 * @author kmorin <kmorin@codelutin.com>
 * @since 2.5.10
 */
public class SwingValidatorMessageWidget extends JToggleButton {

    private static final Log log =
            LogFactory.getLog(SwingValidatorMessageWidget.class);

    private static final long serialVersionUID = 1L;

    protected SwingValidatorMessageTableModel errorTableModel = new SwingValidatorMessageTableModel();

    protected JDialog popup = new JDialog();

    protected JTable errorTable = new JTable();

    protected Point popupPosition = null;

    public SwingValidatorMessageWidget() {
        super(SwingUtil.createActionIcon("alert-none"));
        setToolTipText(_("tutti.validator.alert.none"));

        errorTableModel.addTableModelListener(new TableModelListener() {

            public void tableChanged(TableModelEvent e) {
                int alerts = errorTableModel.getRowCount();
                String label;
                switch (alerts) {
                    case 0:
                        label = n_("validator.messageWidget.alert.none");
                        break;
                    case 1:
                        label = n_("validator.messageWidget.alert.one");
                        break;
                    default:
                        label = n_("validator.messageWidget.alert.several");
                }

                NuitonValidatorScope maxScope;
                String icon;
                if (alerts == 0) {
                    icon = "alert-none";

                } else {
                    maxScope = NuitonValidatorScope.INFO;
                    for (int i = 0; i < alerts; i++) {
                        NuitonValidatorScope scope = errorTableModel.getRow(i).getScope();
                        int diff = scope.compareTo(maxScope);
                        if (diff < 0) {
                            maxScope = scope;
                        }
                    }
                    switch (maxScope) {
                        case INFO:
                            icon = "alert-info";
                            break;
                        case WARNING:
                            icon = "alert-warning";
                            break;
                        default:
                            icon = "alert-error";

                    }
                }

                setToolTipText(_(label, alerts));
                setIcon(SwingUtil.createActionIcon(icon));
            }
        });

        errorTable.setModel(errorTableModel);
        errorTable.setRowSelectionAllowed(true);
        errorTable.setAutoCreateRowSorter(true);
        errorTable.setAutoResizeMode(JTable.AUTO_RESIZE_SUBSEQUENT_COLUMNS);
        errorTable.setCellSelectionEnabled(false);
        errorTable.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        errorTable.setFillsViewportHeight(true);

        SwingValidatorUtil.installUI(errorTable,
                                     new SwingValidatorMessageTableRenderer());

        JScrollPane scrollPanel = new JScrollPane(errorTable);
        scrollPanel.setColumnHeaderView(errorTable.getTableHeader());

        JXTitledPanel titledPanel = new JXTitledPanel(_("validator.messageWidget.title"), scrollPanel);
        popup.add(titledPanel);
        popup.setTitle(_("validator.messageWidget.title"));
        popup.setSize(800, 300);
        popup.setAlwaysOnTop(true);
        popup.setUndecorated(true);

        ComponentResizer cr = new ComponentResizer();
        cr.registerComponent(popup);
        ComponentMover cm = new ComponentMover();
        cm.setDragInsets(cr.getDragInsets());
        cm.registerComponent(popup);

        popup.addWindowListener(new WindowAdapter() {

            @Override
            public void windowClosing(WindowEvent e) {
                setSelected(false);
            }

        });

        popup.addComponentListener(new ComponentAdapter() {

            @Override
            public void componentMoved(ComponentEvent e) {
                Component component = e.getComponent();
                if (component.isShowing()) {
                    popupPosition = component.getLocationOnScreen();
                }
            }

        });

        addActionListener(new ActionListener() {

            @Override
            public void actionPerformed(ActionEvent e) {
                if (isSelected()) {
                    popup.setVisible(true);
                } else {
                    popup.dispose();
                }
            }
        });

        addHierarchyBoundsListener(new HierarchyBoundsAdapter() {

            @Override
            public void ancestorMoved(HierarchyEvent e) {
                if (popupPosition == null && isShowing()) {
                    Point point = new Point(getLocationOnScreen());
                    point.translate(-popup.getWidth() + getWidth(), -popup.getHeight());
                    popup.setLocation(point);
                }
            }
        });

        // add a auto-close action
        JRootPane rootPane = popup.getRootPane();

        KeyStroke shortcutClosePopup = KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0);

        rootPane.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(
                shortcutClosePopup, "close");
        rootPane.getActionMap().put("close", new AbstractAction() {
            private static final long serialVersionUID = 1L;

            @Override
            public void actionPerformed(ActionEvent e) {
                popup.dispose();
                setSelected(false);
            }
        });

    }

    /**
     * Registers a validator.
     *
     * @param validator
     */
    public void registerValidator(SwingValidator validator) {
        errorTableModel.registerValidator(validator);
        validator.reloadBean();
    }

    /** Clears all the validators. */
    public void clearValidators() {
        errorTableModel.clearValidators();
        errorTableModel.clear();
    }

}
