package jaxx.runtime;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.beans.PropertyDescriptor;
import java.lang.reflect.Method;

/**
 * Simple property decorator based on {@link String#format(String, Object[])} method.
 * <p/>
 * To use it, give him a class and the property name to render.
 * <p/>
 * For example :
 * <pre>
 * Decorator&lt;Object&gt; d = PropertyDecorator.newDecorator(PropertyDecorator.class,"expressions");
 * </pre>
 *
 * @param <O> type of decorated objects
 * @author chemit
 * @see Decorator
 */
public class PropertyDecorator<O> extends Decorator<O> {

    private static final long serialVersionUID = 1L;

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private final Log log = LogFactory.getLog(PropertyDecorator.class);

    /**
     * Factory method to instanciate a new {@link jaxx.runtime.PropertyDecorator} for the given class {@link O} and a
     * readable property name.
     *
     * @param internalClass the class of the objects decorated by the new decorator
     * @param property      the property
     * @param <O>           the generic type of class to be decorated by the new decorator
     * @return the new instanciated decorator
     * @throws IllegalArgumentException if the expression is not valid, says:
     *                                  <p/>
     *                                  - a missing right brace was detected.
     *                                  <p/>
     *                                  - a ${ was found in a jxpath token.
     * @throws NullPointerException     if internalClass parameter is null.
     */
    public static <O> PropertyDecorator<O> newDecorator(Class<O> internalClass, String property)
            throws IllegalArgumentException, NullPointerException {
        return new PropertyDecorator<O>(internalClass, property);
    }

    protected String property;

    protected transient Method m;

    @Override
    public String toString(Object bean) {
        try {
            return getM().invoke(bean) + "";
        } catch (Exception e) {
            log.error("could not convert for reason : " + e, e);
            return "";
        }
    }

    public String getProperty() {
        return property;
    }

    protected PropertyDecorator(Class<O> internalClass, String property) throws NullPointerException {
        super(internalClass);
        if (property == null) {
            throw new NullPointerException("property can not be null.");
        }
        this.property = property;
        // init method
        getM();
    }


    protected Method getM() {
        if (m == null) {
            for (PropertyDescriptor propertyDescriptor : PropertyUtils.getPropertyDescriptors(internalClass)) {
                if (propertyDescriptor.getName().equals(property)) {
                    this.m = propertyDescriptor.getReadMethod();
                    break;
                }
            }
            if (m == null) {
                throw new IllegalArgumentException("could not find the property " + property + " in " + internalClass);
            }
        }
        return m;
    }
}
