/*
 * #%L
 * JAXX :: Compiler
 * 
 * $Id: ValidatorFinalizer.java 2178 2011-01-26 14:41:27Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jaxx/tags/jaxx-2.3/jaxx-compiler/src/main/java/jaxx/compiler/finalizers/ValidatorFinalizer.java $
 * %%
 * Copyright (C) 2008 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package jaxx.compiler.finalizers;

import jaxx.compiler.CompiledObject;
import jaxx.compiler.CompiledObject.ChildRef;
import jaxx.compiler.CompilerException;
import jaxx.compiler.JAXXCompiler;
import jaxx.compiler.JAXXCompilerFinalizer;
import jaxx.compiler.java.JavaArgument;
import jaxx.compiler.java.JavaElement;
import jaxx.compiler.java.JavaField;
import jaxx.compiler.java.JavaFile;
import jaxx.compiler.java.JavaFileGenerator;
import jaxx.compiler.reflect.ClassDescriptor;
import jaxx.compiler.reflect.ClassDescriptorHelper;
import jaxx.compiler.tags.validator.BeanValidatorHandler;
import jaxx.compiler.tags.validator.BeanValidatorHandler.CompiledBeanValidator;
import jaxx.compiler.types.TypeManager;
import jaxx.runtime.JAXXValidator;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.validator.swing.SwingValidator;
import jaxx.runtime.validator.swing.SwingValidatorUtil;
import jaxx.runtime.validator.swing.meta.Validator;
import jaxx.runtime.validator.swing.meta.ValidatorField;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.lang.reflect.Modifier;
import java.util.List;
import java.util.Map;

/**
 * To finalize validators fields.
 *
 * @author tchemit <chemit@codelutin.com>
 * @plexus.component role-hint="validators" role="jaxx.compiler.JAXXCompilerFinalizer"
 */
public class ValidatorFinalizer implements JAXXCompilerFinalizer {

    /** Logger. */
    static Log log = LogFactory.getLog(ValidatorFinalizer.class);

    protected static final JavaField VALIDATOR_IDS_FIELD =
            JavaFileGenerator.newField(
                    Modifier.PROTECTED,
                    "java.util.List<String>",
                    "validatorIds",
                    true,
                    "new ArrayList<String>()"
            );

    @Override
    public boolean accept(JAXXCompiler compiler) {
        return BeanValidatorHandler.hasValidator(compiler);
    }

    @Override
    public void finalizeCompiler(CompiledObject root,
                                 JAXXCompiler compiler,
                                 JavaFile javaFile,
                                 String packageName,
                                 String className) {

        for (CompiledObject object : compiler.getObjects().values()) {
            List<ChildRef> childs = object.getChilds();
            if (childs == null || childs.isEmpty()) {
                continue;
            }
            for (ChildRef child : childs) {
                String javaCode = child.getChildJavaCode();
                // some validators are defined on this object
                boolean found =
                        BeanValidatorHandler.isComponentUsedByValidator(
                                compiler,
                                child.getChild().getId()
                        );
                if (found) {
                    // box the child component in a JxLayer
                    child.setChildJavaCode(
                            SwingUtil.class.getSimpleName() +
                            ".boxComponentWithJxLayer(" + javaCode + ")");
                }
            }
        }
        String eol = JAXXCompiler.getLineSeparator();
        StringBuilder builder = new StringBuilder();
        // register validators
        List<CompiledBeanValidator> validators =
                BeanValidatorHandler.getValidators(compiler);
        javaFile.addImport(Validator.class);
        javaFile.addImport(ValidatorField.class);
        javaFile.addImport(SwingValidatorUtil.class);
        compiler.getJavaFile().addMethod(JavaFileGenerator.newMethod(
                Modifier.PUBLIC,
                "void",
                "registerValidatorFields",
                SwingValidatorUtil.class.getSimpleName() + ".installFields(this);",
                true)
        );
        builder.append("// register ");
        builder.append(validators.size());
        builder.append(" validator(s)");
        builder.append(eol);
        builder.append("validatorIds = ");
        builder.append(SwingValidatorUtil.class.getSimpleName()).append(".initUI(this);");
        builder.append(eol);
//        builder.append("registerValidatorFields();");
//        builder.append(eol);
        builder.append(SwingValidatorUtil.class.getSimpleName()).append(".installUI(this);");
        builder.append(eol);
        compiler.appendLateInitializer(builder.toString());
        for (CompiledBeanValidator validator : validators) {

            registerValidator(validator, compiler, javaFile);
        }

//        StringBuilder registerValidatorFieldsMethod = new StringBuilder();
//        registerValidatorFieldsMethod.append(SwingValidatorUtil.class.getSimpleName()).append(".installFields(this);");

//        for (CompiledBeanValidator validator : validators) {
//            validator.registerValidator(compiler, javaFile);
//
//            String id = TypeManager.getJavaCode(validator.getId());


//            builder.append("validatorIds.add(");
//            builder.append(id);
//            builder.append(");");
//            builder.append(eol);
//            builder.append("getValidator(");
//            builder.append(id);
//            builder.append(").installUIs();");
//            builder.append(eol);
//            builder.append("getValidator(");
//            builder.append(id);
//            builder.append(").reloadBean();");
//            builder.append(eol);
        // init fields
//            validator.addFieldRepresentations(compiler, registerValidatorFieldsMethod);
//        }
//        builder.append("validatorIds = java.util.Collections.unmodifiableList(validatorIds);");
//        builder.append(eol);
//        compiler.appendLateInitializer(builder.toString());
//        compiler.getJavaFile().addMethod(JavaFileGenerator.newMethod(
//                Modifier.PUBLIC,
//                "void",
//                "registerValidatorFields",
//                SwingValidatorUtil.class.getSimpleName() + ".installFields(this);",
//                true)
//        );
    }

    @Override
    public void prepareJavaFile(CompiledObject root,
                                JAXXCompiler compiler,
                                JavaFile javaFile,
                                String packageName,
                                String className) throws ClassNotFoundException {
        Class<?> validatorClass = SwingValidator.class;
//                compiler.getConfiguration().getValidatorClass();
        String validatorFQN = validatorClass.getName();
        javaFile.addImport(validatorFQN);

        //TODO use the specific JAXXValidator interface (swing, gwt,...)
        Class<?> validatorInterface = JAXXValidator.class;

        //TC-20091202 : pass this test if we want to interact with non generated code ?
//        if (javaFile.isSuperclassIsJAXXObject()) {
        ClassDescriptor superClass =
                ClassDescriptorHelper.getClassDescriptor(javaFile.getSuperClass());
        ClassDescriptor validatorInterfaceDescriptor =
                ClassDescriptorHelper.getClassDescriptor(validatorInterface);
        boolean parentIsValidator =
                validatorInterfaceDescriptor.isAssignableFrom(superClass);

        if (parentIsValidator) {
            // nothing to generate (use the parent directly)
            return;
        }
//        }

        // add JAXXValidator interface
        javaFile.addInterface(JAXXCompiler.getCanonicalName(validatorInterface));

        // implements JAXXValidator
        javaFile.addField(VALIDATOR_IDS_FIELD);
        javaFile.addMethod(JavaFileGenerator.newMethod(
                Modifier.PUBLIC,
                validatorFQN + "<?>",
                "getValidator",
                "return (" + validatorFQN + "<?>) (validatorIds.contains(validatorId) ? getObjectById(validatorId) : null);",
                true,
                new JavaArgument("String", "validatorId"))
        );
    }

    public void registerValidator(CompiledBeanValidator validator,
                                  JAXXCompiler compiler,
                                  JavaFile javaFile) {

        JavaField validatorField = javaFile.getField(validator.getId());

        String validatorId = TypeManager.getJavaCode(validator.getId());

        String validatorAnnotation = Validator.class.getSimpleName() +
                                     "( validatorId = " + validatorId +
                                     ")";
        validatorField.addAnnotation(validatorAnnotation);
        Map<String, String> fields = validator.getFields();

        for (Map.Entry<String, String> entry : fields.entrySet()) {
            String propertyName = entry.getKey();
            String component = entry.getValue();

            if (!validator.checkBeanProperty(compiler, propertyName)) {
                // property not find on bean
                continue;
            }

            String keyCode = TypeManager.getJavaCode(propertyName);
            String editorCode = TypeManager.getJavaCode(component);
            JavaElement editor = javaFile.getField(component);
            if (editor == null) {
                String message = "Could not find editor [" +
                                 component + "] for property [" + propertyName +
                                 "] for file " + javaFile.getName();
                log.warn(message);

                // find in the compiler the object
                CompiledObject compiledObject = compiler.getCompiledObject(component);

                if (compiledObject == null) {

                    // this is an error, editor is unknown (this case should
                    // not happen)

                    String errorMessage = "Could not find editor [" +
                                          component + "] for property [" + propertyName +
                                          "] for file " + javaFile.getName();

                    throw new CompilerException(errorMessage);
                }

                // now must add a getter in the javaFile

                String fqn = JAXXCompiler.getCanonicalName(compiledObject);

                editor = javaFile.addGetterMethod(component,
                                                  Modifier.PUBLIC,
                                                  fqn,
                                                  true,
                                                  true);
            }

            String annotation = ValidatorField.class.getSimpleName() +
                                "( validatorId = " + validatorId + "," +
                                "  propertyName = " + keyCode + "," +
                                "  editorName = " + editorCode + "" +
                                ")";
            editor.addAnnotation(annotation);
        }
    }

}
