/*
 * #%L
 * JAXX :: Compiler
 * 
 * $Id: ValidatorFinalizer.java 2118 2010-10-26 17:44:57Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jaxx/tags/jaxx-2.2.4/jaxx-compiler/src/main/java/jaxx/compiler/finalizers/ValidatorFinalizer.java $
 * %%
 * Copyright (C) 2008 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package jaxx.compiler.finalizers;

import jaxx.compiler.CompiledObject;
import jaxx.compiler.CompiledObject.ChildRef;
import jaxx.compiler.JAXXCompiler;
import jaxx.compiler.JAXXCompilerFinalizer;
import jaxx.compiler.java.JavaArgument;
import jaxx.compiler.java.JavaField;
import jaxx.compiler.java.JavaFile;
import jaxx.compiler.java.JavaFileGenerator;
import jaxx.compiler.reflect.ClassDescriptor;
import jaxx.compiler.reflect.ClassDescriptorHelper;
import jaxx.compiler.tags.validator.BeanValidatorHandler;
import jaxx.compiler.tags.validator.BeanValidatorHandler.CompiledBeanValidator;
import jaxx.compiler.types.TypeManager;
import jaxx.runtime.JAXXValidator;
import jaxx.runtime.SwingUtil;

import java.lang.reflect.Modifier;
import java.util.List;

/**
 * To finalize validators fields.
 *
 * @author tchemit <chemit@codelutin.com>
 * @plexus.component role-hint="validators" role="jaxx.compiler.JAXXCompilerFinalizer"
 */
public class ValidatorFinalizer implements JAXXCompilerFinalizer {

    protected static final JavaField VALIDATOR_IDS_FIELD =
            JavaFileGenerator.newField(
                    Modifier.PROTECTED,
                    "java.util.List<String>",
                    "validatorIds",
                    true,
                    "new ArrayList<String>()"
            );

    @Override
    public boolean accept(JAXXCompiler compiler) {
        return BeanValidatorHandler.hasValidator(compiler);
    }

    @Override
    public void finalizeCompiler(CompiledObject root,
                                 JAXXCompiler compiler,
                                 JavaFile javaFile,
                                 String packageName,
                                 String className) {

        for (CompiledObject object : compiler.getObjects().values()) {
            List<ChildRef> childs = object.getChilds();
            if (childs == null || childs.isEmpty()) {
                continue;
            }
            for (ChildRef child : childs) {
                String javaCode = child.getChildJavaCode();
                // some validators are defined on this object
                boolean found =
                        BeanValidatorHandler.isComponentUsedByValidator(
                                compiler,
                                child.getChild().getId()
                        );
                if (found) {
                    // box the child component in a JxLayer
                    child.setChildJavaCode(
                            SwingUtil.class.getSimpleName() +
                            ".boxComponentWithJxLayer(" + javaCode + ")");
                }
            }
        }
        String eol = JAXXCompiler.getLineSeparator();
        StringBuilder builder = new StringBuilder();
        // register validator
        List<CompiledBeanValidator> validators =
                BeanValidatorHandler.getValidators(compiler);
        builder.append("// register ");
        builder.append(validators.size());
        builder.append(" validator(s)");
        builder.append(eol);
        builder.append("registerValidatorFields();");
        StringBuilder registerValidatorFieldsMethod = new StringBuilder();
        for (CompiledBeanValidator validator : validators) {
            String id = TypeManager.getJavaCode(validator.getId());
            builder.append("validatorIds.add(");
            builder.append(id);
            builder.append(");");
            builder.append(eol);
            builder.append("getValidator(");
            builder.append(id);
            builder.append(").installUIs();");
            builder.append(eol);
            builder.append("getValidator(");
            builder.append(id);
            builder.append(").reloadBean();");
            builder.append(eol);
            // init fields
            validator.addFieldRepresentations(compiler, registerValidatorFieldsMethod);
        }
        builder.append("validatorIds = java.util.Collections.unmodifiableList(validatorIds);");
        builder.append(eol);
        compiler.appendLateInitializer(builder.toString());
        compiler.getJavaFile().addMethod(JavaFileGenerator.newMethod(
                Modifier.PUBLIC,
                "void",
                "registerValidatorFields",
                registerValidatorFieldsMethod.toString(),
                true)
        );
    }

    @Override
    public void prepareJavaFile(CompiledObject root,
                                JAXXCompiler compiler,
                                JavaFile javaFile,
                                String packageName,
                                String className) throws ClassNotFoundException {
        Class<?> validatorClass =
                compiler.getConfiguration().getValidatorClass();
        String validatorFQN = validatorClass.getName();
        javaFile.addImport(validatorFQN);

        //TODO use the specific JAXXValidator interface (swing, gwt,...)
        Class<?> validatorInterface = JAXXValidator.class;

        //TC-20091202 : pass this test if we want to interact with non generated code ?
//        if (javaFile.isSuperclassIsJAXXObject()) {
        ClassDescriptor superClass =
                ClassDescriptorHelper.getClassDescriptor(javaFile.getSuperClass());
        ClassDescriptor validatorInterfaceDescriptor =
                ClassDescriptorHelper.getClassDescriptor(validatorInterface);
        boolean parentIsValidator =
                validatorInterfaceDescriptor.isAssignableFrom(superClass);

        if (parentIsValidator) {
            // nothing to generate (use the parent directly)
            return;
        }
//        }

        // add JAXXValidator interface
        javaFile.addInterface(JAXXCompiler.getCanonicalName(validatorInterface));

        // implements JAXXValidator
        javaFile.addField(VALIDATOR_IDS_FIELD);
        javaFile.addMethod(JavaFileGenerator.newMethod(
                Modifier.PUBLIC,
                validatorFQN + "<?>",
                "getValidator",
                "return (" + validatorFQN + "<?>) (validatorIds.contains(validatorId) ? getObjectById(validatorId) : null);",
                true,
                new JavaArgument("String", "validatorId"))
        );
    }
}
