/*
 * *##% 
 * JAXX Compiler
 * Copyright (C) 2008 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package jaxx.compiler.finalizers;

import jaxx.compiler.*;
import jaxx.compiler.binding.DataBinding;
import jaxx.compiler.binding.DefaultJAXXBindingWriter;
import jaxx.compiler.binding.JAXXBindingWriter;
import jaxx.compiler.binding.SimpleJAXXObjectBindingWriter;
import jaxx.compiler.java.*;
import static jaxx.compiler.java.JavaFileGenerator.newField;
import static jaxx.compiler.java.JavaFileGenerator.newMethod;
import jaxx.compiler.reflect.ClassDescriptor;
import jaxx.compiler.reflect.ClassDescriptorLoader;
import jaxx.compiler.reflect.FieldDescriptor;
import jaxx.compiler.reflect.MethodDescriptor;
import jaxx.compiler.types.TypeManager;
import jaxx.runtime.*;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.io.IOException;
import static java.lang.reflect.Modifier.*;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * This class is a refactoring of the {@link jaxx.compiler.JAXXCompiler}.
 * <p/>
 * We delegate now the generation of a {@link jaxx.runtime.JAXXObject} to this class, the
 * {@link jaxx.compiler.JAXXCompiler} now only deals with the compilation of files.
 *
 * @author chemit
 */
public class DefaultFinalizer implements JAXXCompilerFinalizer {

    /**
     * Logger
     */
    protected static final Log log = LogFactory.getLog(DefaultFinalizer.class);

    private static final String PARAMETER_NAME_$BINDING = "$binding";

    public static final String FIELD_NAME_$BINDING_SOURCES = "$bindingSources";
    public static final String FIELD_NAME_$OBJECT_MAP = "$objectMap";
    public static final String FIELD_NAME_$ACTIVE_BINDINGS = "$activeBindings";
    public static final String FIELD_NAME_ALL_COMPONENTS_CREATED = "allComponentsCreated";
    public static final String FIELD_NAME_CONTEXT_INITIALIZED = "contextInitialized";
    public static final String FIELD_NAME_$PREVIOUS_VALUES = "$previousValues";
    public static final String FIELD_NAME_$BINDINGS = "$bindings";
    public static final String FIELD_NAME_$PROPERTY_CHANGE_SUPPORT = "$propertyChangeSupport";
    public static final String FIELD_NAME_DELEGATE_CONTEXT = "delegateContext";
    public static final String FIELD_NAME_SERIAL_VERSION_UID = "serialVersionUID";
    public static final String FIELD_NAME_$JAXX_OBJECT_DESCRIPTOR = "$jaxxObjectDescriptor";

    public static final String METHOD_NAME_$GET_JAXXOBJECT_DESCRIPTOR = "$getJAXXObjectDescriptor";
    public static final String METHOD_NAME_$REGISTER_DEFAULT_BINDINGS = "$registerDefaultBindings";
    public static final String METHOD_NAME_REGISTER_DATA_BINDING = "registerDataBinding";
    public static final String METHOD_NAME_REMOVE_DATA_BINDING = "removeDataBinding";
    public static final String METHOD_NAME_APPLY_DATA_BINDING = "applyDataBinding";
    public static final String METHOD_NAME_PROCESS_DATA_BINDING = "processDataBinding";
    public static final String METHOD_NAME_FIRE_PROPERTY_CHANGE = "firePropertyChange";
    public static final String METHOD_NAME_$GET_PROPERTY_CHANGE_SUPPORT = "$getPropertyChangeSupport";
    public static final String METHOD_NAME_$INITIALIZE = "$initialize";
    public static final String METHOD_NAME_$COMPLETE_SETUP = "$completeSetup";
    public static final String METHOD_NAME_$AFTER_COMPLETE_SETUP = "$afterCompleteSetup";

    public static final String TYPE_STRING = "String";
    public static final String TYPE_VOID = "void";
    public static final String TYPE_BOOLEAN = "boolean";
    public static final String TYPE_OBJECT = "Object";

    /**
     * serialVersionUID field
     */
    protected static final JavaField SERIAL_VERSION_UID_FIELD = newField(PRIVATE | STATIC | FINAL,
            "long", FIELD_NAME_SERIAL_VERSION_UID, false, "1L");
    /**
     *
     */
    protected static final JavaField ACTIVE_BINDINGS_FIELD = newField(PROTECTED,
            "java.util.List<" + TYPE_OBJECT + ">", FIELD_NAME_$ACTIVE_BINDINGS, false, "new ArrayList<" + TYPE_OBJECT + ">()");
    /**
     *
     */
    protected static final JavaField BINDING_SOURCES_FIELD = newField(PROTECTED,
            "Map<" + TYPE_STRING + ", " + TYPE_OBJECT + ">", FIELD_NAME_$BINDING_SOURCES, false, "new HashMap<" + TYPE_STRING + ", " + TYPE_OBJECT + ">()");
    /**
     *
     */
    protected static final JavaField OBJECT_MAP_FIELD = newField(PROTECTED,
            "Map<" + TYPE_STRING + ", " + TYPE_OBJECT + ">", FIELD_NAME_$OBJECT_MAP, true, "new HashMap<" + TYPE_STRING + ", " + TYPE_OBJECT + ">()");
    /**
     *
     */
    protected static final JavaField ALL_COMPONENTS_CREATED_FIELD = newField(PRIVATE,
            TYPE_BOOLEAN, FIELD_NAME_ALL_COMPONENTS_CREATED, false);
    /**
     *
     */
    protected static final JavaField CONTEXT_INITIALIZED = newField(PRIVATE,
            TYPE_BOOLEAN, FIELD_NAME_CONTEXT_INITIALIZED, false, "true");
    /**
     *
     */
    protected static final JavaField PREVIOUS_VALUES_FIELD = newField(PROTECTED,
            "Map<?,?>", FIELD_NAME_$PREVIOUS_VALUES, false, "new java.util.HashMap<" + TYPE_OBJECT + ", " + TYPE_OBJECT + ">()");
    /**
     *
     */
    protected static final JavaField BINDINGS_FIELD = newField(PROTECTED | FINAL,
            "Map<" + TYPE_STRING + ", JAXXBinding>", FIELD_NAME_$BINDINGS, false, "new java.util.TreeMap<" + TYPE_STRING + ", JAXXBinding>()");
    /**
     *
     */
    protected static final JavaField PROPERTY_CHANGE_SUPPORT_FIELD = newField(PROTECTED,
            "PropertyChangeSupport", FIELD_NAME_$PROPERTY_CHANGE_SUPPORT, false);
    /**
     *
     */
    protected static final JavaMethod GET_CONTEXT_VALUE_METHOD = newMethod(PUBLIC, "<T> T", "getContextValue",
            "return " + FIELD_NAME_DELEGATE_CONTEXT + ".getContextValue(clazz, null);", true,
            new JavaArgument("Class<T>", "clazz"));
    /**
     *
     */
    protected static final JavaMethod GET_CONTEXT_VALUE_NAMED_METHOD = newMethod(PUBLIC, "<T> T", "getContextValue",
            "return " + FIELD_NAME_DELEGATE_CONTEXT + ".getContextValue(clazz, name);", true,
            new JavaArgument("Class<T>", "clazz"), new JavaArgument(TYPE_STRING, "name"));
    /**
     *
     */
    protected static final JavaMethod SET_CONTEXT_VALUE_NAMED_METHOD = newMethod(PUBLIC, "<T> " + TYPE_VOID, "setContextValue",
            FIELD_NAME_DELEGATE_CONTEXT + ".setContextValue(o, name);", true,
            new JavaArgument("T", "o"), new JavaArgument(TYPE_STRING, "name"));
    /**
     *
     */
    protected static final JavaMethod SET_CONTEXT_VALUE_METHOD = newMethod(PUBLIC, "<T> " + TYPE_VOID, "setContextValue",
            FIELD_NAME_DELEGATE_CONTEXT + ".setContextValue(o, null);", true,
            new JavaArgument("T", "o"));
    /**
     *
     */
    protected static final JavaMethod REMOVE_CONTEXT_VALUE_NAMED_METHOD = newMethod(PUBLIC, "<T> " + TYPE_VOID, "removeContextValue",
            FIELD_NAME_DELEGATE_CONTEXT + ".removeContextValue(clazz, name);", true,
            new JavaArgument("Class<T>", "clazz"), new JavaArgument(TYPE_STRING, "name"));
    /**
     *
     */
    protected static final JavaMethod REMOVE_CONTEXT_VALUE_METHOD = newMethod(PUBLIC, "<T> " + TYPE_VOID, "removeContextValue",
            FIELD_NAME_DELEGATE_CONTEXT + ".removeContextValue(clazz, null);", true,
            new JavaArgument("Class<T>", "clazz"));
    /**
     *
     */
    protected static final JavaMethod GET_PARENT_CONTAINER_MORE_METHOD = newMethod(PUBLIC, "<O extends Container> O", "getParentContainer",
            "return SwingUtil.getParentContainer(source, clazz);", true,
            new JavaArgument("Object", "source"), new JavaArgument("Class<O>", "clazz"));
    /**
     *
     */
    protected static final JavaMethod GET_PARENT_CONTAINER_METHOD = newMethod(PUBLIC, "<O extends Container> O", "getParentContainer",
            "return SwingUtil.getParentContainer(this, clazz);", true,
            new JavaArgument("Class<O>", "clazz"));
    /**
     *
     */
    protected static final JavaMethod GET_OBJECT_BY_ID_METHOD = newMethod(PUBLIC, TYPE_OBJECT, "getObjectById",
            "return " + FIELD_NAME_$OBJECT_MAP + ".get(id);", true,
            new JavaArgument(TYPE_STRING, "id"));
    /**
     *
     */
    protected static final JavaMethod GET_JAXX_OBJECT_DESCRIPTOR_METHOD = newMethod(PUBLIC | STATIC, "JAXXObjectDescriptor", METHOD_NAME_$GET_JAXXOBJECT_DESCRIPTOR,
            "return JAXXUtil.decodeCompressedJAXXObjectDescriptor(" + FIELD_NAME_$JAXX_OBJECT_DESCRIPTOR + ");", false);
    /**
     *
     */
    protected static final JavaMethod PROCESS_DATA_BINDING_METHOD = newMethod(PUBLIC, TYPE_VOID, METHOD_NAME_PROCESS_DATA_BINDING,
            METHOD_NAME_PROCESS_DATA_BINDING + "(" + PARAMETER_NAME_$BINDING + ", false);", true,
            new JavaArgument(TYPE_STRING, PARAMETER_NAME_$BINDING));

    /**
     *
     */
    protected static final JavaMethod REGISTER_DATA_BINDING_METHOD = newMethod(PUBLIC, TYPE_VOID, METHOD_NAME_REGISTER_DATA_BINDING,
            FIELD_NAME_$BINDINGS + ".put(binding.getId(), binding);",
            true,
            new JavaArgument(JAXXBinding.class.getSimpleName(), "binding"));
    /**
     *
     */
    protected static final JavaMethod GET_DATA_BINDING_METHOD = newMethod(PUBLIC, JAXXBinding.class.getSimpleName() + "[]", "getDataBindings",
            "return " + FIELD_NAME_$BINDINGS + ".values().toArray(new " + JAXXBinding.class.getSimpleName() + "[" + FIELD_NAME_$BINDINGS + ".size()]);",
            true);


    /**
     *
     */
    protected static final JavaMethod FIRE_PROPERTY_CHANGE_METHOD = newMethod(PUBLIC, TYPE_VOID, METHOD_NAME_FIRE_PROPERTY_CHANGE,
            "super." + METHOD_NAME_FIRE_PROPERTY_CHANGE + "(propertyName, oldValue, newValue);", true,
            new JavaArgument(TYPE_STRING, "propertyName"), new JavaArgument(TYPE_OBJECT, "oldValue"), new JavaArgument(TYPE_OBJECT, "newValue"));
    /**
     *
     */
    protected static final JavaMethod FIRE_PROPERTY_CHANGE_NAMED_METHOD = newMethod(PUBLIC, TYPE_VOID, METHOD_NAME_FIRE_PROPERTY_CHANGE,
            METHOD_NAME_$GET_PROPERTY_CHANGE_SUPPORT + "()." + METHOD_NAME_FIRE_PROPERTY_CHANGE + "(propertyName, oldValue, newValue);", true,
            new JavaArgument(TYPE_STRING, "propertyName"), new JavaArgument(TYPE_OBJECT, "oldValue"), new JavaArgument(TYPE_OBJECT, "newValue"));
    /**
     *
     */
    protected static final JavaMethod GET_PROPERTY_CHANGE_SUPPORT_METHOD = newMethod(0, PropertyChangeSupport.class.getSimpleName(), METHOD_NAME_$GET_PROPERTY_CHANGE_SUPPORT,
            "if (" + FIELD_NAME_$PROPERTY_CHANGE_SUPPORT + " == null)\n" +
                    "    " + FIELD_NAME_$PROPERTY_CHANGE_SUPPORT + " = new PropertyChangeSupport(this);\n" +
                    "return " + FIELD_NAME_$PROPERTY_CHANGE_SUPPORT + ";", false);
    /**
     *
     */
    protected static final JavaMethod ADD_PROPERTY_CHANGE_SUPPORT_METHOD = newMethod(PUBLIC, TYPE_VOID, "addPropertyChangeListener",
            METHOD_NAME_$GET_PROPERTY_CHANGE_SUPPORT + "().addPropertyChangeListener(listener);", true,
            new JavaArgument(PropertyChangeListener.class.getSimpleName(), "listener"));
    /**
     *
     */
    protected static final JavaMethod ADD_PROPERTY_CHANGE_SUPPORT_NAMED_METHOD = newMethod(PUBLIC, TYPE_VOID, "addPropertyChangeListener",
            METHOD_NAME_$GET_PROPERTY_CHANGE_SUPPORT + "().addPropertyChangeListener(property, listener);", true,
            new JavaArgument(TYPE_STRING, "property"), new JavaArgument(PropertyChangeListener.class.getSimpleName(), "listener"));
    /**
     *
     */
    protected static final JavaMethod REMOVE_PROPERTY_CHANGE_SUPPORT_METHOD = newMethod(PUBLIC, TYPE_VOID, "removePropertyChangeListener",
            METHOD_NAME_$GET_PROPERTY_CHANGE_SUPPORT + "().removePropertyChangeListener(listener);", true,
            new JavaArgument(PropertyChangeListener.class.getSimpleName(), "listener"));
    /**
     *
     */
    protected static final JavaMethod REMOVE_PROPERTY_CHANGE_SUPPORT_NAMED_METHOD = newMethod(PUBLIC, TYPE_VOID, "removePropertyChangeListener",
            METHOD_NAME_$GET_PROPERTY_CHANGE_SUPPORT + "().removePropertyChangeListener(property, listener);", true,
            new JavaArgument(TYPE_STRING, "property"), new JavaArgument(PropertyChangeListener.class.getSimpleName(), "listener"));
    private static final String PARAMETER_NAME_PARENT_CONTEXT = "parentContext";


    @Override
    public boolean accept(JAXXCompiler compiler) {
        return true;
    }

    @Override
    public void finalizeCompiler(CompiledObject root, JAXXCompiler compiler, JavaFile javaFile, String packageName, String className) {

        String fullClassName = packageName != null ? packageName + "." + className : className;
        if (root == null) {
            throw new CompilerException("root tag can not be null");
        }
        ClassDescriptor superclass = root.getObjectClass();
        boolean superclassIsJAXXObject = ClassDescriptorLoader.getClassDescriptor(JAXXObject.class).isAssignableFrom(superclass);
        javaFile.setModifiers(PUBLIC);
        javaFile.setName(fullClassName);
        javaFile.setSuperClass(JAXXCompiler.getCanonicalName(superclass));
        javaFile.setSuperclassIsJAXXObject(superclassIsJAXXObject);

        javaFile.addInterfaces(compiler.getExtraInterfaces());
        javaFile.setAbstractClass(compiler.isAbstractClass());
        javaFile.setGenericType(compiler.getGenericType());
        javaFile.setSuperGenericType(compiler.getSuperGenericType());

//        // finalize all objects via their decorator
//        for (CompiledObject object : compiler.getObjects().values()) {
//            CompiledObjectDecorator decorator = object.getDecorator();
//            decorator.finalizeCompiler(compiler, root, object, javaFile, packageName, className, fullClassName);
//        }

//        // compile bindings and fill the initDataBindings field of binding helper
//        compiler.getBindingHelper().finalizeBindings();

        if (!superclassIsJAXXObject) {
            javaFile.addInterface(JAXXObject.class.getSimpleName());
        }
    }

    @Override
    public void prepareJavaFile(CompiledObject root, JAXXCompiler compiler, JavaFile javaFile, String packageName, String className) throws ClassNotFoundException {

        String fullClassName = javaFile.getName();

        String jaxxContextImplementorClass = compiler.getConfiguration().getJaxxContextClass().getName();
        boolean superclassIsJAXXObject = javaFile.isSuperclassIsJAXXObject();
        if (!superclassIsJAXXObject) {
            javaFile.addImport(JAXXObject.class);
        }
        javaFile.addImport(JAXXContext.class);
        javaFile.addImport(JAXXObjectDescriptor.class);
        javaFile.addImport(JAXXBinding.class);

        if (!superclassIsJAXXObject) {
            // add logger
            if (compiler.getConfiguration().isAddLogger()) {
                javaFile.addImport(Log.class);
                javaFile.addImport(LogFactory.class);
                //TC-20091127 : let the logger stays protected
                javaFile.addSimpleField(newField(PROTECTED | STATIC | FINAL, "Log", "log", false, "LogFactory.getLog(" + fullClassName + ".class)"));
            }

            // JAXXObject
            javaFile.addField(OBJECT_MAP_FIELD);
            javaFile.addMethod(GET_OBJECT_BY_ID_METHOD);
            javaFile.addSimpleField(BINDING_SOURCES_FIELD);
            javaFile.addSimpleField(ACTIVE_BINDINGS_FIELD);
            javaFile.addSimpleField(BINDINGS_FIELD);
            javaFile.addMethod(REGISTER_DATA_BINDING_METHOD);
            javaFile.addMethod(GET_DATA_BINDING_METHOD);

            javaFile.addMethod(createApplyDataBindingMethod());
            javaFile.addMethod(createProcessDataBindingMethod());
            javaFile.addMethod(createRemoveDataBindingMethod());

            // JAXXContext
            javaFile.addField(newField(PROTECTED | FINAL, JAXXContext.class.getSimpleName(), FIELD_NAME_DELEGATE_CONTEXT, true, "new " + jaxxContextImplementorClass + "()"));
            javaFile.addMethod(SET_CONTEXT_VALUE_METHOD);
            javaFile.addMethod(SET_CONTEXT_VALUE_NAMED_METHOD);
            javaFile.addMethod(GET_CONTEXT_VALUE_METHOD);
            javaFile.addMethod(GET_CONTEXT_VALUE_NAMED_METHOD);
            javaFile.addMethod(REMOVE_CONTEXT_VALUE_METHOD);
            javaFile.addMethod(REMOVE_CONTEXT_VALUE_NAMED_METHOD);
            javaFile.addMethod(GET_PARENT_CONTAINER_METHOD);
            javaFile.addMethod(GET_PARENT_CONTAINER_MORE_METHOD);

            // PropertyChangeSupport
            addPropertyChangeSupport(root, javaFile);

            // DataBinding
            javaFile.addMethod(PROCESS_DATA_BINDING_METHOD);
        }
        javaFile.addSimpleField(SERIAL_VERSION_UID_FIELD);
        javaFile.addSimpleField(ALL_COMPONENTS_CREATED_FIELD);
        boolean overrideContextInitialized = false;
        FieldDescriptor[] scriptFields = compiler.getScriptFields();
        for (FieldDescriptor f : scriptFields) {
            if (FIELD_NAME_CONTEXT_INITIALIZED.equals(f.getName())) {
                overrideContextInitialized = true;
                break;
            }
        }
        if (!overrideContextInitialized) {
            javaFile.addSimpleField(CONTEXT_INITIALIZED);
        }
        javaFile.addSimpleField(createJAXXObjectDescriptorField(compiler, javaFile));

        if (compiler.getStylesheet() != null) {
            boolean needField = true;
            if (superclassIsJAXXObject) {
                // check alreay exists on parent
                ClassDescriptor superclass = root.getObjectClass();
                if (log.isDebugEnabled()) {
                    log.debug("superclass : " + superclass);
                }
                JAXXCompiler parentCompiler = compiler.getEngine().getJAXXCompiler(superclass.getName());
                if (parentCompiler != null) {
                    needField = parentCompiler.getStylesheet() == null;
                } else {
                    try {
                        superclass.getDeclaredFieldDescriptor(PREVIOUS_VALUES_FIELD.getName());
                        needField = false;
                    } catch (NoSuchFieldException ex) {
                        // field not found                        
                    }
                }
                if (needField && log.isDebugEnabled()) {
                    log.debug("no " + PREVIOUS_VALUES_FIELD.getName() + " field in super class");
                }
            }
            if (needField) {
                javaFile.addSimpleField(PREVIOUS_VALUES_FIELD);
            }
        }
        //TC 20090228 - only generate constructors if not done in scripts
        boolean constructorDetected = false;
        MethodDescriptor[] methods = compiler.getScriptMethods();
        for (MethodDescriptor m : methods) {
            try {
                m.getReturnType();
                if (className.equals(m.getName())) {
                    constructorDetected = true;
                    break;
                }
            } catch (Exception e) {
                log.warn("could not find return type " + m);
            }
        }
        if (!constructorDetected) {
            javaFile.addMethod(createConstructor(compiler, className, superclassIsJAXXObject));
            javaFile.addMethod(createConstructorWithInitialContext(compiler, className, superclassIsJAXXObject));
        }

        DataBinding[] bindings = compiler.getBindingHelper().getDataBindings();

        int nbBindings = bindings.length;
        boolean hasDataBindings = nbBindings > 0;

        javaFile.addMethod(createInitializer(compiler, nbBindings));
        javaFile.addMethod(GET_JAXX_OBJECT_DESCRIPTOR_METHOD);

        javaFile.addBodyCode(compiler.getBodyCode().toString());

//        javaFile.addMethod(REGISTER_DATA_BINDING_METHOD);
//        javaFile.addMethod(GET_DATA_BINDING_METHOD);

        if (hasDataBindings) {

            // create the $registerDefaultBindings method

            javaFile.addMethod(createRegisterDefaultBindingsMethod(compiler));

            // add import on each type of JAXXBinding used

            for (JAXXBindingWriter<?> writer : bindingWriters) {
                if (writer.isUsed()) {
                    compiler.getJavaFile().addImport(writer.getType());
                }
            }

            // for each binding declare the constant Id

            for (DataBinding binding : bindings) {

                String constantId = binding.getConstantId();

                // add the data binding constant Id

                compiler.addSimpleField(JavaFileGenerator.newField(
                        (constantId.startsWith("BINDING_$") ? PRIVATE : PUBLIC) | FINAL | STATIC,
                        TYPE_STRING, constantId, false, TypeManager.getJavaCode(binding.getRealId())));
            }
        }

        javaFile.addMethod(createCompleteSetupMethod(compiler, javaFile));
//        javaFile.addMethod(createApplyDataBindingMethod(superclassIsJAXXObject, hasDataBindings));
//        javaFile.addMethod(createProcessDataBindingMethod(superclassIsJAXXObject, hasDataBindings));
//        javaFile.addMethod(createRemoveDataBindingMethod(superclassIsJAXXObject, hasDataBindings));

        addEventHandlers(compiler, javaFile);
    }

    protected static final JAXXBindingWriter[] bindingWriters = new JAXXBindingWriter[]{
            new SimpleJAXXObjectBindingWriter(),
            new DefaultJAXXBindingWriter()
    };

    protected JavaMethod createRegisterDefaultBindingsMethod(JAXXCompiler compiler) {
        DataBinding[] bindings = compiler.getBindingHelper().getDataBindings();
        StringBuilder initCode = new StringBuilder();
//        if (bindings.legth > 0) {

        String eol = JAXXCompiler.getLineSeparator();
        JavaFileGenerator generator = new JavaFileGenerator(eol, true);
        //TODO use optimized writer for simple cases

        initCode.append("// register ").append(bindings.length).append(" data bindings").append(eol);

        for (DataBinding binding : bindings) {

            for (JAXXBindingWriter<?> writer : bindingWriters) {
                if (writer.accept(binding)) {
                    writer.write(binding, generator, initCode);
                    break;
                }
            }
        }
        return JavaFileGenerator.newMethod(PRIVATE, TYPE_VOID, METHOD_NAME_$REGISTER_DEFAULT_BINDINGS, initCode.toString(), false);
    }

    /*---------------------------------------------------------------------------------*/
    /*-- Create fields ----------------------------------------------------------------*/
    /*---------------------------------------------------------------------------------*/
    protected JavaField createJAXXObjectDescriptorField(JAXXCompiler compiler, JavaFile javaFile) {
        try {
            JAXXObjectDescriptor descriptor = compiler.getJAXXObjectDescriptor();
            String data = Base64Coder.serialize(descriptor, true);
            /*ByteArrayOutputStream buffer = new ByteArrayOutputStream();
            ObjectOutputStream out = new ObjectOutputStream(new GZIPOutputStream(buffer));
            out.writeObject(descriptor);
            out.close();
            // the use of the weird deprecated constructor is deliberate -- we need to store the data as a String
            // in the compiled class file, since byte array initialization is horribly inefficient compared to
            // String initialization.  So we store the bytes in the String, and we quite explicitly want a 1:1
            // mapping between bytes and chars, with the high byte of the char set to zero.  We can then safely
            // reconstitute the original byte[] at a later date.  This is unquestionably an abuse of the String
            // type, but if we could efficiently store a byte[] we wouldn't have to do this.
            String data = new String(buffer.toByteArray(), 0);*/

            int sizeLimit = 65000; // constant strings are limited to 64K, and I'm not brave enough to push right up to the limit
            if (data.length() < sizeLimit) {
                return newField(PRIVATE | STATIC | FINAL, TYPE_STRING, FIELD_NAME_$JAXX_OBJECT_DESCRIPTOR, false, TypeManager.getJavaCode(data));
            } else {
                StringBuffer initializer = new StringBuffer();
                for (int i = 0; i < data.length(); i += sizeLimit) {
                    String name = FIELD_NAME_$JAXX_OBJECT_DESCRIPTOR + i;
                    javaFile.addField(new JavaField(PRIVATE | STATIC, TYPE_STRING, name, false,
                            TypeManager.getJavaCode(data.substring(i, Math.min(i + sizeLimit, data.length())))));
                    if (initializer.length() > 0) {
                        initializer.append(" + ");
                    }
                    initializer.append(TYPE_STRING + ".valueOf(").append(name).append(")");
                }
                return newField(PRIVATE | STATIC | FINAL, TYPE_STRING, FIELD_NAME_$JAXX_OBJECT_DESCRIPTOR, false, initializer.toString());
            }
        } catch (IOException e) {
            throw new RuntimeException("Internal error: can't-happen error", e);
        }
    }

    /*---------------------------------------------------------------------------------*/
    /*-- Create methods ---------------------------------------------------------------*/
    /*---------------------------------------------------------------------------------*/
    protected void addPropertyChangeSupport(CompiledObject root, JavaFile javaFile) {
        ClassDescriptor currentClass = root.getObjectClass();
        MethodDescriptor firePropertyChange = null;
        while (firePropertyChange == null && currentClass != null) {
            try {
                firePropertyChange = currentClass.getDeclaredMethodDescriptor(METHOD_NAME_FIRE_PROPERTY_CHANGE, ClassDescriptorLoader.getClassDescriptor(String.class),
                        ClassDescriptorLoader.getClassDescriptor(Object.class),
                        ClassDescriptorLoader.getClassDescriptor(Object.class));

            } catch (NoSuchMethodException e) {
                currentClass = currentClass.getSuperclass();
            }
        }

        int modifiers = firePropertyChange != null ? firePropertyChange.getModifiers() : 0;
        if (isPublic(modifiers)) {
            // we have all the support we need
            return;
        }
        if (isProtected(modifiers)) {
            // there is property change support but the firePropertyChange method is protected
            javaFile.addMethod(FIRE_PROPERTY_CHANGE_METHOD);
        } else {
            // either no support at all or firePropertyChange isn't accessible
            javaFile.addImport(PropertyChangeListener.class);
            javaFile.addImport(PropertyChangeSupport.class);

            javaFile.addField(PROPERTY_CHANGE_SUPPORT_FIELD);
            javaFile.addMethod(GET_PROPERTY_CHANGE_SUPPORT_METHOD);
            javaFile.addMethod(ADD_PROPERTY_CHANGE_SUPPORT_METHOD);
            javaFile.addMethod(ADD_PROPERTY_CHANGE_SUPPORT_NAMED_METHOD);
            javaFile.addMethod(REMOVE_PROPERTY_CHANGE_SUPPORT_METHOD);
            javaFile.addMethod(REMOVE_PROPERTY_CHANGE_SUPPORT_NAMED_METHOD);
            javaFile.addMethod(FIRE_PROPERTY_CHANGE_NAMED_METHOD);
        }
    }

    protected void addEventHandlers(JAXXCompiler compiler, JavaFile javaFile) {
        for (Map.Entry<String, Map<ClassDescriptor, List<EventHandler>>> e1 : compiler.getEventHandlers().entrySet()) {
            // outer loop is iterating over different objects (well, technically, different Java expressions)
            for (Map.Entry<ClassDescriptor, List<EventHandler>> e2 : e1.getValue().entrySet()) {
                // iterate over different types of listeners for this particular object (MouseListener, ComponentListener, etc.)
                for (EventHandler handler : e2.getValue()) {
                    // iterate over individual event handlers of a single type
                    String methodName = compiler.getEventHandlerMethodName(handler);
                    MethodDescriptor listenerMethod = handler.getListenerMethod();
                    if (listenerMethod.getParameterTypes().length != 1) {
                        throw new CompilerException("Expected event handler " + listenerMethod.getName() + " of class " + handler.getListenerClass() + " to have exactly one argument");
                    }
                    javaFile.addMethod(newMethod(PUBLIC, TYPE_VOID, methodName, handler.getJavaCode(), false,
                            new JavaArgument(JAXXCompiler.getCanonicalName(listenerMethod.getParameterTypes()[0]), "event")));
                }
            }
        }
    }

    protected JavaMethod createConstructor(JAXXCompiler compiler, String className, boolean superclassIsJAXXObject) throws CompilerException {
        StringBuffer code = new StringBuffer();
        String constructorParams = compiler.getRootObject().getConstructorParams();
        String eol = JAXXCompiler.getLineSeparator();
        if (constructorParams != null) {
            code.append("        super(").append(constructorParams).append(");").append(eol);
        } else {
            if (superclassIsJAXXObject) {
                code.append("        super();").append(eol);
            }
        }
        code.append(METHOD_NAME_$INITIALIZE + "();");
        code.append(eol);
        return newMethod(PUBLIC, null, className, code.toString(), false);
    }

    protected JavaMethod createConstructorWithInitialContext(JAXXCompiler compiler, String className, boolean superclassIsJAXXObject) throws CompilerException {
        StringBuffer code = new StringBuffer();
        String constructorParams = compiler.getRootObject().getConstructorParams();
        String eol = JAXXCompiler.getLineSeparator();
        if (constructorParams != null) {
            if (superclassIsJAXXObject) {
                //TODO-TC20091127 This MUST be a convention in JAXX : all specialized constructor must a second
                //TODO constructor with extra first parameter as parentContext
                constructorParams = PARAMETER_NAME_PARENT_CONTEXT + ", " + constructorParams;
            }
            code.append("        super(").append(constructorParams).append(");").append(eol);
        } else {
            if (superclassIsJAXXObject) {
                code.append("        super(" + PARAMETER_NAME_PARENT_CONTEXT + ");").append(eol);
            }
        }
        if (!superclassIsJAXXObject) {
            code.append(JAXXUtil.class.getSimpleName()).append(".initContext(this, " + PARAMETER_NAME_PARENT_CONTEXT + ");");
            code.append(eol);
        }
        code.append(METHOD_NAME_$INITIALIZE + "();");
        code.append(eol);
        return newMethod(PUBLIC, null, className, code.toString(), false, new JavaArgument(JAXXContext.class.getSimpleName(), PARAMETER_NAME_PARENT_CONTEXT));
    }

    public JavaMethod createInitializer(JAXXCompiler compiler, int nbBindings) throws CompilerException {
        String eol = JAXXCompiler.getLineSeparator();
        StringBuffer code = new StringBuffer();
        CompiledObject root = compiler.getRootObject();
        code.append("if (" + FIELD_NAME_ALL_COMPONENTS_CREATED + " || !" + FIELD_NAME_CONTEXT_INITIALIZED + ") {").append(eol);
        code.append("    return;").append(eol);
        code.append("}").append(eol);
        // register bindings before anything else
        if (nbBindings > 0) {
            // ajout invocation a la methode d'enregistrement des bindings
            code.append("// registers ").append(nbBindings).append(" data bindings").append(eol);
            code.append(METHOD_NAME_$REGISTER_DEFAULT_BINDINGS + "();").append(eol);
        }
//        String dataBindingsCode = createRegisterDefaultBindingsMethod(compiler);
//        if (dataBindingsCode != null && !dataBindingsCode.isEmpty()) {
//            code.append(dataBindingsCode).append(eol);
//        }
        //TODO-TC20091025 we should remove this if no used anywhere
        code.append(FIELD_NAME_$OBJECT_MAP + ".put(").append(TypeManager.getJavaCode(root.getId())).append(", this);").append(eol);

        Iterator<CompiledObject> i = compiler.getObjectCreationOrder();
        boolean lastWasMethodCall = false;
        //TODO-TC20091025 should do init of root first ?
//        root.getDecorator().createInitializer(compiler, root, root, code, lastWasMethodCall);
        while (i.hasNext()) {
            CompiledObject object = i.next();
            if (object == root) {
                continue;
            }
            CompiledObjectDecorator decorator = object.getDecorator();
            lastWasMethodCall = decorator.createInitializer(compiler, root, object, code, lastWasMethodCall);
        }
        root.getDecorator().createInitializer(compiler, root, root, code, lastWasMethodCall);
        if (compiler.getInitializer().length() > 0) {
            code.append(compiler.getInitializer());
        }
        code.append(METHOD_NAME_$COMPLETE_SETUP + "();").append(eol);
        return newMethod(PRIVATE, TYPE_VOID, METHOD_NAME_$INITIALIZE, code.toString(), false);
    }

    protected JavaMethod createCompleteSetupMethod(JAXXCompiler compiler, JavaFile javaFile) {
        StringBuffer code = new StringBuffer();
        code.append(FIELD_NAME_ALL_COMPONENTS_CREATED + " = true;");
        String eol = JAXXCompiler.getLineSeparator();
        code.append(eol);
        for (CompiledObject object : compiler.getObjects().values()) {
            CompiledObjectDecorator decorator = object.getDecorator();
            code.append(decorator.createCompleteSetupMethod(compiler, object, javaFile));
        }
        String simpleBindingsCode = createInitBindingsCode(compiler);
        if (simpleBindingsCode != null && !simpleBindingsCode.isEmpty()) {
            code.append(simpleBindingsCode).append(eol);
        }

        if (compiler.getLateInitializer().length() > 0) {
            code.append("// late initializer").append(eol);
            code.append(compiler.getLateInitializer()).append(eol);
        }
        //TC-20090313 add an extra method after complete setup
        MethodDescriptor method = compiler.getScriptMethod(METHOD_NAME_$AFTER_COMPLETE_SETUP);
        if (method != null) {
            code.append(METHOD_NAME_$AFTER_COMPLETE_SETUP + "();").append(eol);
        }
        return newMethod(PRIVATE, TYPE_VOID, METHOD_NAME_$COMPLETE_SETUP, code.toString(), false);
    }

    protected String createInitBindingsCode(JAXXCompiler compiler) {
        String eol = JAXXCompiler.getLineSeparator();
        DataBinding[] bindings;

        StringBuilder result = new StringBuilder();

        bindings = compiler.getBindingHelper().getDataBindings();
        if (bindings.length > 0) {

            result.append(eol).append("// apply ").append(bindings.length).append(" data bindings").append(eol);
            result.append(JAXXUtil.class.getSimpleName()).append("." + METHOD_NAME_APPLY_DATA_BINDING + "(this, " + FIELD_NAME_$BINDINGS + ".keySet());").append(eol);

        }
        bindings = compiler.getBindingHelper().getSimpleBindings();
        if (bindings.length > 0) {

            StringBuilder initCode = new StringBuilder();

            for (DataBinding binding : bindings) {

                String binding1 = binding.getInitDataBinding();
                if (binding1 != null && !binding1.trim().isEmpty()) {
                    initCode.append(binding1);
                }
            }

            if (initCode.length() > 0) {

                result.append(eol).append("// apply ").append(bindings.length).append(" property setters").append(eol);
                result.append(initCode.toString().trim());

            }
        }
        return result.toString();
    }

    //    protected JavaMethod createApplyDataBindingMethod(boolean superclassIsJAXXObject, boolean haveBinding) {
    protected JavaMethod createApplyDataBindingMethod() {
        StringBuilder buffer = new StringBuilder();
        String eol = JAXXCompiler.getLineSeparator();

//        if (haveBinding) {
//            if (superclassIsJAXXObject)
        buffer.append("if (" + FIELD_NAME_ALL_COMPONENTS_CREATED + " && " + FIELD_NAME_$BINDINGS + ".containsKey(" + PARAMETER_NAME_$BINDING + ")) {").append(eol);
//            else {
//                buffer.append("if (" + FIELD_NAME_$BINDINGS + ".containsKey(" + PARAMETER_NAME_$BINDING + ")) {").append(eol);
//            }
        buffer.append("    " + FIELD_NAME_$BINDINGS + ".get(" + PARAMETER_NAME_$BINDING + ")." + METHOD_NAME_APPLY_DATA_BINDING + "();").append(eol);
        buffer.append("}");

//        }
//        if (superclassIsJAXXObject) {
//
//            if (haveBinding) {
//                buffer.append(" else {").append(eol).append("    ");
//            }
//            buffer.append("super." + METHOD_NAME_APPLY_DATA_BINDING + "(" + PARAMETER_NAME_$BINDING + ");").append(eol);
//
//            if (haveBinding) {
//                buffer.append("    return;").append(eol).append("}");
//            }
//
//        }

//        if (haveBinding) {
//            if (superclassIsJAXXObject)
//                buffer.append("if (" + FIELD_NAME_ALL_COMPONENTS_CREATED + " && " + FIELD_NAME_$BINDINGS + ".containsKey(" + PARAMETER_NAME_$BINDING + ")) {").append(eol);
//            else {
//                buffer.append("if (" + FIELD_NAME_$BINDINGS + ".containsKey(" + PARAMETER_NAME_$BINDING + ")) {").append(eol);
//            }
//            buffer.append("    " + FIELD_NAME_$BINDINGS + ".get(" + PARAMETER_NAME_$BINDING + ")." + METHOD_NAME_APPLY_DATA_BINDING + "();").append(eol);
//            buffer.append("}");
//
//        }
//        if (superclassIsJAXXObject) {
//
//            if (haveBinding) {
//                buffer.append(" else {").append(eol).append("    ");
//            }
//            buffer.append("super." + METHOD_NAME_APPLY_DATA_BINDING + "(" + PARAMETER_NAME_$BINDING + ");").append(eol);
//
//            if (haveBinding) {
//                buffer.append("    return;").append(eol).append("}");
//            }
//
//        }
        buffer.append(eol).append(METHOD_NAME_PROCESS_DATA_BINDING + "(" + PARAMETER_NAME_$BINDING + ");");
        return newMethod(PUBLIC, TYPE_VOID, METHOD_NAME_APPLY_DATA_BINDING, buffer.toString(), true,
                new JavaArgument(TYPE_STRING, PARAMETER_NAME_$BINDING));
    }

    protected JavaMethod createRemoveDataBindingMethod() {
//    protected JavaMethod createRemoveDataBindingMethod(boolean superclassIsJAXXObject, boolean haveBinding) {
        StringBuilder buffer = new StringBuilder();
        String eol = JAXXCompiler.getLineSeparator();

        buffer.append("if (" + FIELD_NAME_ALL_COMPONENTS_CREATED + " && " + FIELD_NAME_$BINDINGS + ".containsKey(" + PARAMETER_NAME_$BINDING + ")) {").append(eol);
        buffer.append("    " + FIELD_NAME_$BINDINGS + ".get(" + PARAMETER_NAME_$BINDING + ")." + METHOD_NAME_REMOVE_DATA_BINDING + "();").append(eol);
        buffer.append("}");

//        if (haveBinding) {
//
//            if (superclassIsJAXXObject) {
//                buffer.append("if (" + FIELD_NAME_ALL_COMPONENTS_CREATED + " && " + FIELD_NAME_$BINDINGS + ".containsKey(" + PARAMETER_NAME_$BINDING + ")) {").append(eol);
//            } else {
//                buffer.append("if (" + FIELD_NAME_$BINDINGS + ".containsKey(" + PARAMETER_NAME_$BINDING + ")) {").append(eol);
//            }
//            buffer.append("    " + FIELD_NAME_$BINDINGS + ".get(" + PARAMETER_NAME_$BINDING + ")." + METHOD_NAME_REMOVE_DATA_BINDING + "();").append(eol);
//            buffer.append("}");
//
//
//        }
//        if (superclassIsJAXXObject) {
//
//            if (haveBinding) {
//                buffer.append(" else {").append(eol).append("    ");
//            }
//            buffer.append("super." + METHOD_NAME_REMOVE_DATA_BINDING + "(" + PARAMETER_NAME_$BINDING + ");").append(eol);
//
//            if (haveBinding) {
//                buffer.append("}");
//            }
//        }
        return newMethod(PUBLIC, TYPE_VOID, METHOD_NAME_REMOVE_DATA_BINDING, buffer.toString(), true,
                new JavaArgument(TYPE_STRING, PARAMETER_NAME_$BINDING));
    }

    protected JavaMethod createProcessDataBindingMethod() {
//    protected JavaMethod createProcessDataBindingMethod(boolean superclassIsJAXXObject, boolean haveBinding) {
        StringBuffer code = new StringBuffer();
        String eol = JAXXCompiler.getLineSeparator();
        //boolean superclassIsJAXXObject = ClassDescriptorLoader.getClassDescriptor(JAXXObject.class).isAssignableFrom(compiler.getRootObject().getObjectClass());
        // the force parameter forces the update to happen even if it is already in activeBindings.  This
        // is used on superclass invocations b/c by the time the call gets to the superclass, it is already
        // marked active and would otherwise be skipped

        code.append("    if (!$force && " + FIELD_NAME_$ACTIVE_BINDINGS + ".contains(" + PARAMETER_NAME_$BINDING + ")) { ").append(eol);
        code.append("    return;").append(eol);
        code.append("}").append(eol);
        code.append(FIELD_NAME_$ACTIVE_BINDINGS + ".add(" + PARAMETER_NAME_$BINDING + ");").append(eol);
        code.append("try {").append(eol);
        code.append("    if (" + FIELD_NAME_ALL_COMPONENTS_CREATED + " && " + FIELD_NAME_$BINDINGS + ".containsKey(" + PARAMETER_NAME_$BINDING + ")) {").append(eol);
        code.append("        " + FIELD_NAME_$BINDINGS + ".get(" + PARAMETER_NAME_$BINDING + ")." + METHOD_NAME_PROCESS_DATA_BINDING + "();").append(eol);
        code.append("    }").append(eol);
        code.append("} finally {").append(eol);
        code.append("    " + FIELD_NAME_$ACTIVE_BINDINGS + ".remove(" + PARAMETER_NAME_$BINDING + ");").append(eol);
        code.append("}").append(eol);

//        if (haveBinding) {
//            code.append("    if (!$force && " + FIELD_NAME_$ACTIVE_BINDINGS + ".contains(" + PARAMETER_NAME_$BINDING + ")) { ");
//            code.append(eol);
//            code.append("    return;").append(eol);
//            code.append("}").append(eol);
//            code.append(FIELD_NAME_$ACTIVE_BINDINGS + ".add(" + PARAMETER_NAME_$BINDING + ");").append(eol);
//            code.append("try {").append(eol);
//            if (superclassIsJAXXObject) {
//                code.append("    if (" + FIELD_NAME_ALL_COMPONENTS_CREATED + " && " + FIELD_NAME_$BINDINGS + ".containsKey(" + PARAMETER_NAME_$BINDING + ")) {").append(eol);
//            } else {
//                code.append("    if (" + FIELD_NAME_$BINDINGS + ".containsKey(" + PARAMETER_NAME_$BINDING + ")) {").append(eol);
//            }
//            code.append("        " + FIELD_NAME_$BINDINGS + ".get(" + PARAMETER_NAME_$BINDING + ")." + METHOD_NAME_PROCESS_DATA_BINDING + "();").append(eol);
//            code.append("    }");
//            if (superclassIsJAXXObject) {
//                code.append(" else {").append(eol);
//                code.append("        super." + METHOD_NAME_PROCESS_DATA_BINDING + "(" + PARAMETER_NAME_$BINDING + ", true);").append(eol);
//                code.append("    }");
//            }
//            code.append(eol);
//            code.append("} finally {").append(eol);
//            code.append("    " + FIELD_NAME_$ACTIVE_BINDINGS + ".remove(" + PARAMETER_NAME_$BINDING + ");").append(eol);
//            code.append("}").append(eol);
//        } else if (superclassIsJAXXObject) {
//            code.append("super." + METHOD_NAME_PROCESS_DATA_BINDING + "(" + PARAMETER_NAME_$BINDING + ", true);").append(eol);
//        }
        return newMethod(PUBLIC, TYPE_VOID, METHOD_NAME_PROCESS_DATA_BINDING, code.toString(), true,
                new JavaArgument(TYPE_STRING, PARAMETER_NAME_$BINDING), new JavaArgument(TYPE_BOOLEAN, "$force"));
    }
}
