/*
 * #%L
 * I18n :: Maven Plugin
 * 
 * $Id: ParserGWTJavaMojo.java 1903 2011-05-04 19:30:55Z fdesbois $
 * $HeadURL: http://svn.nuiton.org/svn/i18n/tags/i18n-2.4.1/maven-i18n-plugin/src/main/java/org/nuiton/i18n/plugin/parser/impl/ParserGWTJavaMojo.java $
 * %%
 * Copyright (C) 2011 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.i18n.plugin.parser.impl;

import org.apache.maven.plugin.logging.Log;
import org.nuiton.i18n.plugin.parser.AbstractFileParser;
import org.nuiton.i18n.plugin.parser.AbstractI18nParserMojo;
import org.nuiton.i18n.plugin.parser.FileParser;
import org.nuiton.i18n.plugin.parser.I18nSourceEntry;
import org.nuiton.i18n.plugin.parser.ParserException;
import org.nuiton.i18n.plugin.parser.SourceEntry;
import org.nuiton.io.FileUpdater;
import org.nuiton.io.MirroredFileUpdater;
import org.nuiton.io.SortedProperties;
import org.nuiton.processor.ProcessorUtil;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.util.HashSet;
import java.util.Set;

/**
 * To detect from GWT java files new I18n keys, says content of patterns :
 * <ul>
 * <li>{@code @Key("XXX")}</li>
 * <li>{@code @LocalizableResource.Key("XXX")}</li>
 * <li>{@code @com.google.gwt.i18n.client.LocalizableResource.Key("XXX")}</li>
 * </ul>
 * <b>Note: </b> this goal must always be invoked before the {@code process-resources}
 * phase, otherwise all files will be considered as uptodate.
 *
 * @author tchemit <chemit@codelutin.com>
 * @goal parserGWTJava
 * @phase generate-resources
 * @since 2.1
 */
public class ParserGWTJavaMojo extends AbstractI18nParserMojo {

    /**
     * Root directory of the default entry.
     *
     * @parameter expression="${i18n.defaultBasedir}" default-value="${basedir}/src/main/java"
     */
    protected File defaultBasedir;

    /**
     * Default included files to process (ant-like expression).
     *
     * @parameter expression="${i18n.defaultIncludes}" default-value="**\/*.java"
     */
    protected String defaultIncludes;

    /**
     * Defines the file name of the getter where to put detected i18n keys
     * while getter phase.
     *
     * @parameter expression="${i18n.outputGetter}" default-value="gwt-java.getter"
     * @since 2.0
     */
    protected String outputGetter;

    /**
     * Where to generated temporary processed files.
     *
     * @parameter expression="${i18n.workdir}" default-value="${basedir}/target/i18n-workdir"
     * @since 2.0
     */
    protected File workdir;

    protected MirroredFileUpdater entryUpdater;

    @Override
    public String[] getDefaultIncludes() {
        return new String[]{defaultIncludes};
    }

    @Override
    public String[] getDefaultExcludes() {
        return I18nSourceEntry.EMPTY_STRING_ARRAY;
    }

    @Override
    public File getDefaultBasedir() {
        return defaultBasedir;
    }

    @Override
    protected boolean onEnterEntry(I18nSourceEntry entry) {
        boolean b = super.onEnterEntry(entry);
        if (!b) {

            // no skipped entry
            // keep the file updater
            entryUpdater = (MirroredFileUpdater) entry.getUpdater();
        }
        return b;
    }

    @Override
    public FileUpdater newFileUpdater(SourceEntry entry) {
        return new MirroredFileUpdater("", "", entry.getBasedir(), workdir) {

            @Override
            public File getMirrorFile(File f) {
                String file =
                        f.getAbsolutePath().substring(prefixSourceDirecotory);
                return new File(destinationDirectory + File.separator + file);
            }
        };
    }

    @Override
    protected String getOutGetter() {
        return outputGetter;
    }

    @Override
    public FileParser newFileParser() {

        return new GWTJavaFileParser(getLog(),
                                     encoding,
                                     oldParser,
                                     isShowTouchedFiles()
        );
    }

    protected class GWTJavaFileParser extends AbstractFileParser {


        public GWTJavaFileParser(Log log,
                                 String encoding,
                                 SortedProperties oldParser,
                                 boolean showTouchedFiles) {
            super(log, encoding, oldParser, showTouchedFiles);
        }


        protected File prepareFile(File file) throws IOException {

            // clean the jsp to make it xml

            File result = entryUpdater.getMirrorFile(file);
            if (isVerbose()) {
                getLog().info("Will process [" + file + "] to " + result);
            }
            try {
                createDirectoryIfNecessary(result.getParentFile());
            } catch (IOException e) {
                // don't care about it...
            }
            GWTJavaProcessor processor = new GWTJavaProcessor();
            processor.setVerbose(isVerbose());
            processor.process(file, result, getEncoding());
            return result;
        }

        @Override
        public void parseFile(File file) throws IOException {

            // clean the jsp to make it xml

            File processedFile = prepareFile(file);

            // process file to obtain
            String line = null;
            LineNumberReader lnr = new LineNumberReader(new InputStreamReader(
                    new FileInputStream(processedFile), getEncoding()));
            try {
                while ((line = lnr.readLine()) != null) {
                    parseLine(processedFile, line);
                }
            } catch (Exception e) {
                if (line != null) {
                    getLog().error(
                            "could not parse line (" + lnr.getLineNumber() + ") '"
                            + line + "' of file " + file);
                }
                throw new ParserException(e);
            } finally {
                lnr.close();
            }
        }

        @Override
        public void parseLine(File file, String line) throws IOException {

            String key = line.trim();

            if (key.isEmpty()) {
                // no key detected on this line
                return;
            }

            // one key found in file, so file is marked as touched
            setTouched(true);
            // Found a i18n key, register it.
            if (getLog().isDebugEnabled()) {
                getLog().debug(file.getName() + " detected key = " + key);
            }
            registerKey(key);
        }
    }

    /**
     * To Extract i18n keys from GWT java files.
     * <p/>
     * Says keep content only of those patterns :
     * <ul>
     * <li>{@code @Key("XXX")}</li>
     * <li>{@code @LocalizableResource.Key("XXX")}</li>
     * <li>{@code @com.google.gwt.i18n.client.LocalizableResource.Key("XXX")}</li>
     * </ul>
     *
     * @author tchemit <chemit@codelutin.com>
     * @since 2.1
     */
    public static class GWTJavaProcessor extends ProcessorHelper.AbstractParserProcessor {

        protected ProcessorUtil.FragmentExtractor filter1 =
                new ProcessorUtil.FragmentExtractor(
                        "@Key(\"",
                        "\")"
                );

        protected ProcessorUtil.FragmentExtractor filter2 =
                new ProcessorUtil.FragmentExtractor(
                        "@LocalizableResource.Key(\"",
                        "\")"
                );

        protected ProcessorUtil.FragmentExtractor filter3 =
                new ProcessorUtil.FragmentExtractor(
                        "@com.google.gwt.i18n.client.LocalizableResource.Key(\"",
                        "\")"
                );


        @Override
        public void process(File filein,
                            File fileout,
                            String encoding) throws IOException {

            Set<String> keys = new HashSet<String>();
            extractKeys(filter1, filein, encoding, verbose, keys);
            extractKeys(filter2, filein, encoding, verbose, keys);
            extractKeys(filter3, filein, encoding, verbose, keys);
            saveKeysToFile(fileout, keys);
        }

    }
}