/*
 * #%L
 * EUGene :: Maven plugin
 * 
 * $Id: AvailableDataMojo.java 1079 2011-06-28 09:15:23Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/eugene/tags/eugene-2.4.1/maven-eugene-plugin/src/main/java/org/nuiton/eugene/plugin/AvailableDataMojo.java $
 * %%
 * Copyright (C) 2006 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.eugene.plugin;

import org.apache.commons.lang.StringUtils;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.nuiton.eugene.ModelPropertiesUtil;
import org.nuiton.eugene.ModelReader;
import org.nuiton.eugene.Template;
import org.nuiton.eugene.models.Model;
import org.nuiton.eugene.models.object.ObjectModel;
import org.nuiton.eugene.writer.ChainedFileWriter;

import java.util.Arrays;
import java.util.EnumSet;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;

/**
 * Obtain the list of some known data informations.
 * <p/>
 * Use the {@code dataTypes} property to specify a specific data type to use
 * (otherwise will display all known data types).
 * <p/>
 *
 * @author tchemit <chemit@codelutin.com>
 * @goal available-data
 * @requiresProject true
 * @requiresDirectInvocation true
 * @requiresDependencyResolution test
 * @since 2.0.0
 */
public class AvailableDataMojo extends AbstractMojo {

    /**
     * Data type to display (let empty to see all datas).
     * Can specify more than one separated by comma.
     * <p/>
     * Available types are :
     * <pre>
     * modeltype,
     * modelreader,
     * modeltemplate,
     * writer,
     * stereotype,
     * tagvalue
     * </pre>
     * <p/>
     * <b>Note:</b> Let empty to display all data types.
     *
     * @parameter expression="${dataTypes}" default-value=""
     * @since 2.0.0
     */
    protected String dataTypes;

    /**
     * All available models (obtain by plexus, keys are plexus roles, values
     * are a instance of corresponding model).
     *
     * @component role="org.nuiton.eugene.models.Model"
     */
    protected Map<String, Model> modelTypes;

    /**
     * All available writers introspects via plexus
     *
     * @component role="org.nuiton.eugene.ModelReader"
     */
    protected Map<String, ModelReader<?>> modelReaders;

    /**
     * All available templates introspects via plexus
     *
     * @component role="org.nuiton.eugene.Template"
     */
    protected Map<String, Template<?>> modelTemplates;

    /**
     * All available writers introspects via plexus
     *
     * @component role="org.nuiton.eugene.writer.ChainedFileWriter"
     */
    protected Map<String, ChainedFileWriter> writers;

    /**
     * All available model properties providers introspects via plexus
     *
     * @component role="org.nuiton.eugene.ModelPropertiesUtil$ModelPropertiesProvider"
     * @since 2.3.1
     */
    protected Map<String, ModelPropertiesUtil.ModelPropertiesProvider> modelPropertiesProviders;

    protected Map<String, ModelPropertiesUtil.ModelPropertiesProvider> loadedProviders;

    protected ModelPropertiesUtil.ModelPropertiesProvider currentPropertiesProvider;

    @Override
    public void execute() throws MojoExecutionException, MojoFailureException {
        StringBuilder buffer = new StringBuilder();
        dataTypes = dataTypes == null ? "" : dataTypes.trim();

        Set<AvailableData> safeDataTypes;

        if (StringUtils.isEmpty(dataTypes)) {
            // treate all data types
            safeDataTypes = EnumSet.allOf(AvailableData.class);

            if (getLog().isDebugEnabled()) {
                getLog().debug("will use all data types : " + safeDataTypes);
            }

        } else {

            safeDataTypes = EnumSet.noneOf(AvailableData.class);
            for (String s : dataTypes.split(",")) {
                s = s.trim().toLowerCase();
                try {
                    AvailableData data = AvailableData.valueOf(s);
                    if (getLog().isDebugEnabled()) {
                        getLog().debug("will use data type " + data);
                    }
                    safeDataTypes.add(data);
                } catch (IllegalArgumentException e) {
                    getLog().warn(
                            "does not know data type : " + s + " use one of " +
                            Arrays.toString(AvailableData.values()));
                }
            }
        }

        if (safeDataTypes.contains(AvailableData.stereotype) ||
            safeDataTypes.contains(AvailableData.tagvalue)) {

            loadedProviders = new LinkedHashMap<String, ModelPropertiesUtil.ModelPropertiesProvider>();
            // init stores
            for (Map.Entry<String, ModelPropertiesUtil.ModelPropertiesProvider> e : modelPropertiesProviders.entrySet()) {
                ModelPropertiesUtil.ModelPropertiesProvider provider = ModelPropertiesUtil.newStore(
                        Arrays.asList(e.getValue()),
                        false
                );
                loadedProviders.put(e.getKey(), provider);
            }
        }
        for (AvailableData data : safeDataTypes) {
            buffer.append("\n");
            appendData(data, buffer);
        }

        getLog().info("Get datas for data types : " + safeDataTypes +
                      buffer.toString());
    }

    protected void appendData(AvailableData data, StringBuilder buffer) {

        String dataType = data.name();

        if (data == AvailableData.tagvalue || data == AvailableData.stereotype) {

            int nbData = 0;
            for (ModelPropertiesUtil.ModelPropertiesProvider provider : loadedProviders.values()) {
                currentPropertiesProvider = provider;
                nbData += data.getData(this).size();
            }
            currentPropertiesProvider = null;

            String format = "\nFound %s %ss in %s provider(s) : %s\n";

            buffer.append(String.format(format,
                                        nbData,
                                        dataType,
                                        loadedProviders.size(),
                                        loadedProviders.keySet())
            );

            for (Map.Entry<String, ModelPropertiesUtil.ModelPropertiesProvider> e : loadedProviders.entrySet()) {
                String providerName = e.getKey();
                ModelPropertiesUtil.ModelPropertiesProvider provider = e.getValue();
                currentPropertiesProvider = provider;
                Map<String, ?> map = data.getData(this);

                int size = map.size();
                buffer.append("\nProvider [").append(providerName).append("] - ");
                if (size == 0) {
                    buffer.append("No available ").append(dataType).append(".");
                } else if (size == 1) {
                    buffer.append("Found one ").append(dataType).append(" : ");
                } else {
                    buffer.append("Found ");
                    buffer.append(size);
                    buffer.append(" ");
                    buffer.append(dataType);
                    buffer.append("s : ");
                }
                for (Map.Entry<String, ?> e2 : map.entrySet()) {
                    data.toString(buffer, e2);
                }
            }
        } else {
            Map<String, ?> map = data.getData(this);

            int size = map.size();
            if (size == 0) {
                buffer.append("\nNo available ").append(dataType).append(".");
            } else if (size == 1) {
                buffer.append("\nFound one ").append(dataType).append(" : ");
            } else {
                buffer.append("\nFound ");
                buffer.append(size);
                buffer.append(" ");
                buffer.append(dataType);
                buffer.append("s : ");
            }
            for (Map.Entry<String, ?> e : map.entrySet()) {
                data.toString(buffer, e);
            }
        }
    }

    enum AvailableData {
        modeltype {
            @Override
            public Map<String, ?> getData(AvailableDataMojo mojo) {
                return mojo.modelTypes;
            }
        },
        writer {
            @Override
            public Map<String, ?> getData(AvailableDataMojo mojo) {
                return mojo.writers;
            }

            @Override
            String toString(Object data) {
                ChainedFileWriter w = (ChainedFileWriter) data;
                StringBuilder b = new StringBuilder(super.toString(data));
                b.append("\n").append("  inputProtocol             : ");
                b.append(w.getInputProtocol());
                b.append("\n").append("  outputProtocol            : ");
                b.append(w.getOutputProtocol(ObjectModel.NAME));
                b.append("\n").append("  defaultIncludes           : ");
                b.append(w.getDefaultIncludes());
                b.append("\n").append("  defaultInputDirectory     : ");
                b.append(w.getDefaultInputDirectory());
                b.append("\n").append("  defaultTestInputDirectory : ");
                b.append(w.getDefaultTestInputDirectory());
                return b.toString();
            }
        },
        modelreader {
            @Override
            public Map<String, ?> getData(AvailableDataMojo mojo) {
                return mojo.modelReaders;
            }
        },
        modeltemplate {
            @Override
            public Map<String, ?> getData(AvailableDataMojo mojo) {
                return mojo.modelTemplates;
            }
        },
        tagvalue {
            @Override
            public Map<String, ?> getData(AvailableDataMojo mojo) {
                return mojo.getCurrentPropertiesProvider().getTagValueStore();
            }

            @Override
            String toString(Object data) {
                ModelPropertiesUtil.TagValueDefinition d = (ModelPropertiesUtil.TagValueDefinition) data;
                StringBuilder sb = new StringBuilder();
                Class<?>[] targets = d.target();
                for (Class<?> aClass : targets) {
                    sb.append(", ").append(aClass.getSimpleName());
                }
                String result = sb.toString();
                if (targets.length > 0) {
                    result = result.substring(2);
                }
                return result;
            }

            @Override
            void toString(StringBuilder buffer, Map.Entry<String, ?> e) {
                String name = e.getKey();
                Object value = e.getValue();
                buffer.append("\n [");
                buffer.append(name);
                buffer.append("] targets : '");
                buffer.append(toString(value));
                buffer.append("\' : ");
                buffer.append(((ModelPropertiesUtil.TagValueDefinition) value).documentation());
            }
        },
        stereotype {
            @Override
            public Map<String, ?> getData(AvailableDataMojo mojo) {
                return mojo.getCurrentPropertiesProvider().getStereotypeStore();
            }

            @Override
            String toString(Object data) {
                ModelPropertiesUtil.StereotypeDefinition d = (ModelPropertiesUtil.StereotypeDefinition) data;
                StringBuilder sb = new StringBuilder();
                Class<?>[] targets = d.target();
                for (Class<?> aClass : targets) {
                    sb.append(", ").append(aClass.getSimpleName());
                }
                String result = sb.toString();
                if (targets.length > 0) {
                    result = result.substring(2);
                }
                return result;
            }

            @Override
            void toString(StringBuilder buffer, Map.Entry<String, ?> e) {
                String name = e.getKey();
                Object value = e.getValue();
                buffer.append("\n [");
                buffer.append(name);
                buffer.append("] targets : '");
                buffer.append(toString(value));
                buffer.append("\' : ");
                buffer.append(((ModelPropertiesUtil.StereotypeDefinition) value).documentation());
            }
        };

        abstract Map<String, ?> getData(AvailableDataMojo mojo);

        String toString(Object data) {
            return data.getClass().getName();
        }

        void toString(StringBuilder buffer, Map.Entry<String, ?> e) {
            String name = e.getKey();
            Object value = e.getValue();
            buffer.append("\n [");
            buffer.append(name);
            buffer.append("] with implementation '");
            buffer.append(toString(value));
            buffer.append('\'');
        }
    }

    protected ModelPropertiesUtil.ModelPropertiesProvider getCurrentPropertiesProvider() {
        return currentPropertiesProvider;
    }

    protected ModelPropertiesUtil.ModelPropertiesProvider getModelPropertiesProvider() {

        ModelPropertiesUtil.ModelPropertiesProvider provider;

        if (modelPropertiesProviders == null ||
            modelPropertiesProviders.isEmpty()) {

            // could not find any model properties via plexus
            // try to obtain them by ServiceLoader

            provider = ModelPropertiesUtil.newStore(
                    getClass().getClassLoader(),
                    false
            );
        } else {
            provider = ModelPropertiesUtil.newStore(
                    modelPropertiesProviders.values(),
                    false
            );
        }
        return provider;
    }

}
