package org.kth.dks.dks_comm;

import java.net.MalformedURLException;
import java.util.regex.Pattern;

public class DKSOverlayAddress {
  private long id = 0;
  private long groupId = 0;
  private long guid = 0;
  private static final Pattern dksUrl =
      Pattern.compile("^dksoverlay://(\\d*?)/(\\d*?)/(\\d*?)$", Pattern.CASE_INSENSITIVE);


  public DKSOverlayAddress(long newId, long newGroupId, long newGUID) {
    id = newId;
    groupId = newGroupId;
    guid = newGUID;
  }

  /**
   * Constructs a DKSOverlayAddress from a DKSOverlayURL string having the following format:
   * <b>dksoverlay://[GROUP_ID]/[NODE_ID]/[UNIQUE_GUID])</b><br>
   * GROUP_ID must be a number describing the unique DKS ring/group identifier,
   * NODE_ID must be a number describing the node logical identifier,
   * UNIQUE_GUID must be a number describing the node unique identifier (only for identification).
   * The scheme of the url can be in any case, e.g. "dksoverlay://" or "DKSOverlay://".<br><br>
   * Example DKS URL <i>"dksoverlay://2/431/5156164356"</i>
   * @param dksURL String with the above described format
   * @throws MalformedURLException if the string does not match the above description
   */
  public DKSOverlayAddress( String dksURL ) throws MalformedURLException
  {

    if (!dksUrl.matcher(dksURL).matches()) {
      throw new MalformedURLException("Only DKSOverlay scheme supported (DKSOverlay://<GROUP_ID>/<NODE_ID>/<UNIQUE_GUID>)");
    }

    String colons = dksUrl.matcher(dksURL).replaceAll("$1:$2:$3");

    String [] sp = colons.split(":");

    groupId = Long.parseLong(sp[0]);
    id = Long.parseLong(sp[1]);
    guid = Long.parseLong(sp[2]);
  }

  /**
   * Returns the DKSOverlayURL representing this DKSOverlayNetwork in the following format:
   * <b>DKSOverlay://[GROUP_ID]/[NODE_ID]/[UNIQUE_GUID])</b><br>
   * GROUP_ID must be a number describing the unique DKS ring/group identifier,
   * NODE_ID must be a number describing the node logical identifier,
   * UNIQUE_GUID must be a number describing the node unique identifier (only for identification).
   * The scheme of the url can be in any case, e.g. "dks://" or "DKS://".<br><br>
   * Example DKS URL <i>"DKSOverlay://2/431/5156164356"</i>
   * @return String the DKS URL representing this object
   */
  public String getDKSOverlayURL() {
    return "DKSOverlay://"+groupId+"/"+id+"/"+guid;
  }

  public String toString() { return getDKSOverlayURL(); }

  public long getID() {
    return id;
  }

  public long getGroupId() {
    return groupId;
  }

  public long getGUID() {
    return guid;
  }

  public boolean equals(Object o) {
    if (o==this)                                 // optimization
      return true;
    else if (!(o instanceof DKSOverlayAddress))  // avoid nulls and avoid ClassCastExceptions
      return false;

    DKSOverlayAddress addr = (DKSOverlayAddress) o;
    return (addr.id==id && addr.groupId==groupId && addr.guid==guid);
  }

  public int hashCode() {
    int tmp = 17;
    tmp = 37*tmp + (int) (id ^ (id >>> 32));
    tmp = 37*tmp + (int) (groupId ^ (groupId >>> 32));
    tmp = 37*tmp + (int) (guid ^ (guid >>> 32));
    return tmp;
  }

}//DKSOverlayAddress class
