/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.reflect.plugins.javassist.bytecode;

import java.lang.reflect.Method;

import javassist.CtConstructor;
import javassist.Modifier;
import javassist.bytecode.Bytecode;
import javassist.bytecode.CodeAttribute;
import javassist.bytecode.ConstPool;
import javassist.bytecode.MethodInfo;
import javassist.bytecode.Opcode;
import javassist.bytecode.StackMapTable;
import javassist.util.proxy.RuntimeSupport;

import org.jboss.reflect.plugins.javassist.JavassistConstructor;

/**
 * <p>Class to create implementations of the {{@link JavassistConstructor} interface.</p>
 * 
 * <p>This implementation generates raw bytecode to avoid the overhead of compilation via javassist. If 
 * <code>sun.reflect.MagicAccessorImpl</code> is used as the <code>superClass</code> field the implementation
 * classes can access private and protected members of the target class.</p>
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
class JavassistConstructorFactory extends JavassistBehaviorFactory
{
   /** The names of the interfaces we are implementing */ 
   private static final String[] interfaceNames = new String[] {JavassistConstructor.class.getName()};
   
   /** The methods from the interface that are being implemented */ 
   private static final Method[] methods;
   
   static
   {
      methods = new Method[1];
      try
      {
         methods[0] = SecurityActions.getDeclaredMethod(JavassistConstructor.class, "newInstance", Object[].class);
      }
      catch (NoSuchMethodException e)
      {
         throw new RuntimeException(e);
      }
   }

   /** The constructor we are targeting */
   private final CtConstructor ctConstructor;
   
   /** The name of the class being generated */
   private final String className = JavassistConstructor.class.getName() + counter.incrementAndGet();
   
   /**
    * Constructor
    * 
    * @param superClass the super class to use for the implementation
    * @param ctConstructor the constructor we are generating a {@link JavassistConstructor} for
    * @param debug true to cause the class bytes to be output to the file system so they can be inspected with a decompiler/javap
    */      
   JavassistConstructorFactory(Class<?> superClass, CtConstructor ctConstructor, boolean debug)
   {
      super(superClass, ctConstructor, debug);
      this.ctConstructor = ctConstructor;
   }

   String getGeneratedClassName()
   {
      return className;
   }
   

   String[] getInterfaceNames()
   {
      return interfaceNames;
   }

   @Override
   MethodInfo implementMethod(int index, ConstPool cp)
   {
      if (index >= methods.length)
         return null;

      Method method = methods[index]; 

      String desc = RuntimeSupport.makeDescriptor(method);
      MethodInfo minfo = new MethodInfo(cp, method.getName(), desc);
      minfo.setAccessFlags(Modifier.PUBLIC);
      setThrows(minfo, cp, method.getExceptionTypes());
      Bytecode code = new Bytecode(cp, 0, 0);
      
      int pc = code.currentPc();
      
      code.addNew(ctConstructor.getDeclaringClass().getName());
      code.addOpcode(Opcode.DUP);

      addParameters(code, getParameterTypes(), 1);
      
      code.addInvokespecial(ctConstructor.getDeclaringClass(), "<init>", ctConstructor.getSignature());
      code.addOpcode(Opcode.ARETURN);
      
      //We need 3 local variable slots.
      //One for 'this', one for the target reference and one for the argument array.
      //These are all object references and so take one slot each 
      code.setMaxLocals(3);
      CodeAttribute ca = code.toCodeAttribute();
      minfo.setCodeAttribute(ca);

      StackMapTable.Writer writer = new StackMapTable.Writer(32);
      writer.sameFrame(pc);
      ca.setAttribute(writer.toStackMapTable(cp));
      return minfo;
   }
}