/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.reflect.plugins.javassist.bytecode;

import javassist.CtBehavior;
import javassist.CtClass;
import javassist.NotFoundException;
import javassist.bytecode.Bytecode;
import javassist.bytecode.Opcode;

import org.jboss.reflect.plugins.javassist.JavassistConstructor;
import org.jboss.reflect.plugins.javassist.JavassistMethod;

/**
 * <p>Abstract class to create implementations of the {@link JavassistMethod} and {@link JavassistConstructor}
 * interfaces.</p>
 * <p>This implementation generates raw bytecode to avoid the overhead of compilation via javassist. If 
 * <code>sun.reflect.MagicAccessorImpl</code> is used as the <code>superClass</code> field the implementation
 * classes can access private and protected members of the target class.</p>
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
public abstract class JavassistBehaviorFactory extends JavassistMemberFactory
{
   /** The constructor or method being targeted */
   CtBehavior ctBehavior;
   
   /**
    * Constructor
    * 
    * @param superClass the super class to use for the implementation
    * @param ctBehavior the method or constructor we are generating a {@link JavassistMethod} or {@link JavassistConstructor} for
    * @param debug true to cause the class bytes to be output to the file system so they can be inspected with a decompiler/javap
    */      
   JavassistBehaviorFactory(Class<?> superClass, CtBehavior ctBehavior, boolean debug)
   {
      super(superClass, debug);
      this.ctBehavior = ctBehavior;
   }
   
   /**
    * Gets the parameter types of the method or constructor we are generating a {@link JavassistMethod} or {@link JavassistConstructor} for.
    * 
    * @return the parameter types
    */
   CtClass[] getParameterTypes()
   {
      try
      {
         return ctBehavior.getParameterTypes();
      }
      catch (NotFoundException e)
      {
         throw new RuntimeException(e);
      }
   }
   
   @Override
   String initAccessedMember()
   {
      return ctBehavior.getDeclaringClass().getName() + "." + ctBehavior.getName() + ctBehavior.getSignature();
   }

   /**
    * Adds the byte code to push the parameters for a method/constructor 
    * on to the stack
    * 
    * @param code the byte code of the method we are generating
    * @param params the target parameter types
    */
   void addParameters(Bytecode code, CtClass[] params, int parameterIndex)
   {
      for (int i = 0 ; i < params.length ; i++)
      {
         code.addAload(parameterIndex);
         code.addIconst(i);
         code.addOpcode(Opcode.AALOAD);
         if (!params[i].getName().equals(OBJECT_NAME))
         {
            castAndUnbox(code, params[i]);
         }
      }
   }
}
