/*     */ package org.jboss.virtual.plugins.context;
/*     */ 
/*     */ import java.io.IOException;
/*     */ import java.io.ObjectInputStream;
/*     */ import java.io.ObjectInputStream.GetField;
/*     */ import java.io.ObjectOutputStream;
/*     */ import java.io.ObjectOutputStream.PutField;
/*     */ import java.io.ObjectStreamField;
/*     */ import java.net.MalformedURLException;
/*     */ import java.net.URI;
/*     */ import java.net.URISyntaxException;
/*     */ import java.net.URL;
/*     */ import java.util.List;
/*     */ import java.util.concurrent.atomic.AtomicInteger;
/*     */ import org.jboss.logging.Logger;
/*     */ import org.jboss.virtual.VFSUtils;
/*     */ import org.jboss.virtual.VirtualFile;
/*     */ import org.jboss.virtual.plugins.vfs.helpers.PathTokenizer;
/*     */ import org.jboss.virtual.spi.VFSContext;
/*     */ import org.jboss.virtual.spi.VFSContextFactory;
/*     */ import org.jboss.virtual.spi.VFSContextFactoryLocator;
/*     */ import org.jboss.virtual.spi.VirtualFileHandler;
/*     */ 
/*     */ public abstract class AbstractVirtualFileHandler
/*     */   implements VirtualFileHandler
/*     */ {
/*  56 */   protected static final Logger log = Logger.getLogger(AbstractVirtualFileHandler.class);
/*     */   private static final long serialVersionUID = 1L;
/*  60 */   private static final ObjectStreamField[] serialPersistentFields = { new ObjectStreamField("rootURI", URI.class), new ObjectStreamField("parent", VirtualFileHandler.class), new ObjectStreamField("name", String.class), new ObjectStreamField("vfsUrl", URL.class) };
/*     */   private VFSContext context;
/*     */   private VirtualFileHandler parent;
/*     */   private String name;
/*     */   private transient String vfsPath;
/*     */   protected URL vfsUrl;
/*  89 */   private transient AtomicInteger references = new AtomicInteger(0);
/*     */   protected transient long cachedLastModified;
/*     */ 
/*     */   protected AbstractVirtualFileHandler(VFSContext context, VirtualFileHandler parent, String name)
/*     */   {
/* 104 */     if (context == null)
/* 105 */       throw new IllegalArgumentException("Null context");
/* 106 */     if (name == null)
/* 107 */       throw new IllegalArgumentException("Null name");
/* 108 */     this.context = context;
/* 109 */     this.parent = parent;
/* 110 */     this.name = VFSUtils.fixName(name);
/*     */   }
/*     */ 
/*     */   public boolean hasBeenModified() throws IOException
/*     */   {
/* 115 */     boolean hasBeenModified = false;
/* 116 */     long last = getLastModified();
/* 117 */     if (this.cachedLastModified != last)
/*     */     {
/* 119 */       hasBeenModified = this.cachedLastModified != 0L;
/* 120 */       this.cachedLastModified = last;
/*     */     }
/* 122 */     return hasBeenModified;
/*     */   }
/*     */ 
/*     */   public String getName()
/*     */   {
/* 128 */     return this.name;
/*     */   }
/*     */ 
/*     */   public String getPathName()
/*     */   {
/* 133 */     if (this.vfsPath == null)
/*     */     {
/* 135 */       StringBuilder pathName = new StringBuilder();
/* 136 */       initPath(pathName);
/* 137 */       this.vfsPath = pathName.toString();
/*     */     }
/* 139 */     return this.vfsPath;
/*     */   }
/*     */ 
/*     */   public void setPathName(String path)
/*     */   {
/* 149 */     this.vfsPath = path;
/*     */   }
/*     */ 
/*     */   public URL toURL()
/*     */     throws MalformedURLException, URISyntaxException
/*     */   {
/* 156 */     return toURI().toURL();
/*     */   }
/*     */ 
/*     */   public URL toVfsUrl()
/*     */     throws MalformedURLException, URISyntaxException
/*     */   {
/* 162 */     return this.vfsUrl;
/*     */   }
/*     */ 
/*     */   private boolean initPath(StringBuilder pathName)
/*     */   {
/* 173 */     if (this.parent != null)
/*     */     {
/* 175 */       if ((this.parent instanceof AbstractVirtualFileHandler))
/*     */       {
/* 177 */         AbstractVirtualFileHandler handler = (AbstractVirtualFileHandler)this.parent;
/* 178 */         if (handler.initPath(pathName))
/* 179 */           pathName.append('/');
/*     */       }
/*     */       else
/*     */       {
/* 183 */         pathName.append(this.parent.getPathName());
/*     */       }
/* 185 */       pathName.append(getName());
/* 186 */       return true;
/*     */     }
/* 188 */     return false;
/*     */   }
/*     */ 
/*     */   public VirtualFile getVirtualFile()
/*     */   {
/* 193 */     checkClosed();
/* 194 */     increment();
/* 195 */     return new VirtualFile(this);
/*     */   }
/*     */ 
/*     */   public VirtualFileHandler getParent() throws IOException
/*     */   {
/* 200 */     checkClosed();
/* 201 */     return this.parent;
/*     */   }
/*     */ 
/*     */   public VFSContext getVFSContext()
/*     */   {
/* 206 */     checkClosed();
/* 207 */     return this.context;
/*     */   }
/*     */ 
/*     */   protected int increment()
/*     */   {
/* 217 */     return this.references.incrementAndGet();
/*     */   }
/*     */ 
/*     */   protected int decrement()
/*     */   {
/* 227 */     return this.references.decrementAndGet();
/*     */   }
/*     */ 
/*     */   protected void checkClosed()
/*     */     throws IllegalStateException
/*     */   {
/* 237 */     if (this.references.get() < 0)
/* 238 */       throw new IllegalStateException("Closed " + this);
/*     */   }
/*     */ 
/*     */   public void close()
/*     */   {
/* 243 */     if (decrement() == 0)
/* 244 */       doClose();
/*     */   }
/*     */ 
/*     */   protected void doClose()
/*     */   {
/*     */   }
/*     */ 
/*     */   public VirtualFileHandler structuredFindChild(String path)
/*     */     throws IOException
/*     */   {
/* 265 */     checkClosed();
/*     */ 
/* 268 */     String[] tokens = PathTokenizer.getTokens(path);
/* 269 */     if ((tokens == null) || (tokens.length == 0)) {
/* 270 */       return this;
/*     */     }
/*     */ 
/* 274 */     VirtualFileHandler current = this;
/* 275 */     for (int i = 0; i < tokens.length; i++)
/*     */     {
/* 277 */       if (current.isLeaf()) {
/* 278 */         throw new IOException("File cannot have children: " + current);
/*     */       }
/* 280 */       if (PathTokenizer.isReverseToken(tokens[i]))
/*     */       {
/* 282 */         VirtualFileHandler parent = current.getParent();
/* 283 */         if (parent == null) {
/* 284 */           throw new IOException("Using reverse path on top file handler: " + current + ", " + path);
/*     */         }
/* 286 */         current = parent;
/*     */       }
/* 288 */       else if ((current instanceof StructuredVirtualFileHandler))
/*     */       {
/* 290 */         StructuredVirtualFileHandler structured = (StructuredVirtualFileHandler)current;
/* 291 */         current = structured.createChildHandler(tokens[i]);
/*     */       }
/*     */       else
/*     */       {
/* 295 */         String remainingPath = PathTokenizer.getRemainingPath(tokens, i);
/* 296 */         return current.findChild(remainingPath);
/*     */       }
/*     */ 
/*     */     }
/*     */ 
/* 301 */     return current;
/*     */   }
/*     */ 
/*     */   public VirtualFileHandler simpleFindChild(String path)
/*     */     throws IOException
/*     */   {
/* 314 */     if (path == null) {
/* 315 */       throw new IllegalArgumentException("Null path");
/*     */     }
/* 317 */     if (path.length() == 0) {
/* 318 */       return this;
/*     */     }
/*     */ 
/* 321 */     String appliedPath = PathTokenizer.applyReversePaths(path);
/* 322 */     List children = getChildren(false);
/* 323 */     for (VirtualFileHandler child : children)
/*     */     {
/* 325 */       if (child.getName().equals(appliedPath))
/* 326 */         return child;
/*     */     }
/* 328 */     throw new IOException("Child not found " + path + " for " + this);
/*     */   }
/*     */ 
/*     */   public String toString()
/*     */   {
/* 334 */     StringBuilder buffer = new StringBuilder();
/* 335 */     buffer.append(getClass().getSimpleName());
/* 336 */     buffer.append('@');
/* 337 */     buffer.append(System.identityHashCode(this));
/* 338 */     buffer.append("[path=").append(getPathName());
/* 339 */     buffer.append(" context=").append(this.context.getRootURI());
/* 340 */     buffer.append(" real=").append(safeToURLString());
/* 341 */     buffer.append(']');
/* 342 */     return buffer.toString();
/*     */   }
/*     */ 
/*     */   public int hashCode()
/*     */   {
/* 348 */     return getPathName().hashCode();
/*     */   }
/*     */ 
/*     */   public boolean equals(Object obj)
/*     */   {
/* 354 */     if (this == obj)
/* 355 */       return true;
/* 356 */     if ((obj == null) || (!(obj instanceof VirtualFileHandler)))
/* 357 */       return false;
/* 358 */     VirtualFileHandler other = (VirtualFileHandler)obj;
/* 359 */     if (!getVFSContext().equals(other.getVFSContext())) {
/* 360 */       return false;
/*     */     }
/* 362 */     return getPathName().equals(other.getPathName());
/*     */   }
/*     */ 
/*     */   private String safeToURLString()
/*     */   {
/*     */     try
/*     */     {
/* 383 */       return toURI().toString();
/*     */     }
/*     */     catch (URISyntaxException ignored) {
/*     */     }
/* 387 */     return "<unknown>";
/*     */   }
/*     */ 
/*     */   private void writeObject(ObjectOutputStream out)
/*     */     throws IOException
/*     */   {
/* 394 */     ObjectOutputStream.PutField fields = out.putFields();
/* 395 */     fields.put("rootURI", getVFSContext().getRootURI());
/* 396 */     fields.put("parent", this.parent);
/* 397 */     fields.put("name", this.name);
/* 398 */     fields.put("vfsUrl", this.vfsUrl);
/* 399 */     out.writeFields();
/*     */   }
/*     */ 
/*     */   private void readObject(ObjectInputStream in)
/*     */     throws IOException, ClassNotFoundException
/*     */   {
/* 406 */     ObjectInputStream.GetField fields = in.readFields();
/* 407 */     URI rootURI = (URI)fields.get("rootURI", null);
/* 408 */     this.parent = ((VirtualFileHandler)fields.get("parent", null));
/* 409 */     this.name = ((String)fields.get("name", null));
/* 410 */     VFSContextFactory factory = VFSContextFactoryLocator.getFactory(rootURI);
/* 411 */     this.context = factory.getVFS(rootURI);
/* 412 */     this.references = new AtomicInteger(0);
/* 413 */     this.vfsUrl = ((URL)fields.get("vfsUrl", null));
/*     */   }
/*     */ }

/* Location:           /home/mnovotny/projects/EMBEDDED_JBOSS_BETA3_COMMUNITY/embedded/output/lib/embedded-jboss/lib/jboss-embedded-all.jar
 * Qualified Name:     org.jboss.virtual.plugins.context.AbstractVirtualFileHandler
 * JD-Core Version:    0.6.0
 */