/*     */ package org.jboss.util.threadpool;
/*     */ 
/*     */ import java.util.ArrayList;
/*     */ import java.util.Collections;
/*     */ import java.util.Map;
/*     */ import java.util.PriorityQueue;
/*     */ import java.util.concurrent.LinkedBlockingQueue;
/*     */ import java.util.concurrent.RejectedExecutionHandler;
/*     */ import java.util.concurrent.ThreadFactory;
/*     */ import java.util.concurrent.ThreadPoolExecutor;
/*     */ import java.util.concurrent.ThreadPoolExecutor.AbortPolicy;
/*     */ import java.util.concurrent.ThreadPoolExecutor.CallerRunsPolicy;
/*     */ import java.util.concurrent.ThreadPoolExecutor.DiscardOldestPolicy;
/*     */ import java.util.concurrent.ThreadPoolExecutor.DiscardPolicy;
/*     */ import java.util.concurrent.TimeUnit;
/*     */ import java.util.concurrent.atomic.AtomicBoolean;
/*     */ import java.util.concurrent.atomic.AtomicInteger;
/*     */ import org.jboss.logging.Logger;
/*     */ import org.jboss.util.collection.WeakValueHashMap;
/*     */ 
/*     */ public class BasicThreadPool
/*     */   implements ThreadPool, BasicThreadPoolMBean
/*     */ {
/*  53 */   private static final ThreadGroup JBOSS_THREAD_GROUP = new ThreadGroup("JBoss Pooled Threads");
/*     */ 
/*  56 */   private static final Map threadGroups = Collections.synchronizedMap(new WeakValueHashMap());
/*     */ 
/*  59 */   private static final AtomicInteger lastPoolNumber = new AtomicInteger(0);
/*     */ 
/*  61 */   private static Logger log = Logger.getLogger(BasicThreadPool.class);
/*     */   private String name;
/*     */   private int poolNumber;
/*  72 */   private BlockingMode blockingMode = BlockingMode.ABORT;
/*     */   private ThreadPoolExecutor executor;
/*     */   private LinkedBlockingQueue queue;
/*     */   private ThreadGroup threadGroup;
/*  84 */   private AtomicInteger lastThreadNumber = new AtomicInteger(0);
/*     */ 
/*  87 */   private AtomicBoolean stopped = new AtomicBoolean(false);
/*     */ 
/*  89 */   private PriorityQueue<TimeoutInfo> tasksWithTimeouts = new PriorityQueue(13);
/*     */   private TimeoutMonitor timeoutTask;
/*     */   private boolean trace;
/*     */ 
/*     */   public BasicThreadPool()
/*     */   {
/* 104 */     this("ThreadPool");
/*     */   }
/*     */ 
/*     */   public BasicThreadPool(String name)
/*     */   {
/* 115 */     this(name, JBOSS_THREAD_GROUP);
/*     */   }
/*     */ 
/*     */   public BasicThreadPool(String name, ThreadGroup threadGroup)
/*     */   {
/* 127 */     this.trace = log.isTraceEnabled();
/* 128 */     ThreadFactory factory = new ThreadPoolThreadFactory(null);
/*     */ 
/* 130 */     this.queue = new LinkedBlockingQueue(1024);
/*     */ 
/* 133 */     this.executor = new ThreadPoolExecutor(4, 4, 60L, TimeUnit.SECONDS, this.queue);
/* 134 */     this.executor.setThreadFactory(factory);
/* 135 */     this.executor.setRejectedExecutionHandler(new ThreadPoolExecutor.AbortPolicy());
/*     */ 
/* 137 */     this.poolNumber = lastPoolNumber.incrementAndGet();
/* 138 */     setName(name);
/* 139 */     this.threadGroup = threadGroup;
/*     */   }
/*     */ 
/*     */   public void stop(boolean immediate)
/*     */   {
/* 148 */     log.debug("stop, immediate=" + immediate);
/* 149 */     this.stopped.set(true);
/* 150 */     if (immediate)
/* 151 */       this.executor.shutdownNow();
/*     */     else
/* 153 */       this.executor.shutdown();
/*     */   }
/*     */ 
/*     */   public void waitForTasks() throws InterruptedException
/*     */   {
/* 158 */     this.executor.awaitTermination(9223372036854775807L, TimeUnit.MILLISECONDS);
/*     */   }
/*     */ 
/*     */   public void waitForTasks(long maxWaitTime) throws InterruptedException {
/* 162 */     this.executor.awaitTermination(maxWaitTime, TimeUnit.MILLISECONDS);
/*     */   }
/*     */ 
/*     */   public void runTaskWrapper(TaskWrapper wrapper)
/*     */   {
/* 167 */     if (this.trace)
/* 168 */       log.trace("runTaskWrapper, wrapper=" + wrapper);
/* 169 */     if (this.stopped.get())
/*     */     {
/* 171 */       wrapper.rejectTask(new ThreadPoolStoppedException("Thread pool has been stopped"));
/* 172 */       return;
/*     */     }
/*     */ 
/* 175 */     wrapper.acceptTask();
/*     */ 
/* 177 */     long completionTimeout = wrapper.getTaskCompletionTimeout();
/* 178 */     TimeoutInfo info = null;
/* 179 */     if (completionTimeout > 0L)
/*     */     {
/* 181 */       checkTimeoutMonitor();
/*     */ 
/* 183 */       info = new TimeoutInfo(wrapper, completionTimeout);
/* 184 */       this.tasksWithTimeouts.add(info);
/*     */     }
/* 186 */     int waitType = wrapper.getTaskWaitType();
/* 187 */     switch (waitType)
/*     */     {
/*     */     case 2:
/* 191 */       executeOnThread(wrapper);
/* 192 */       break;
/*     */     default:
/* 196 */       execute(wrapper);
/*     */     }
/*     */ 
/* 199 */     waitForTask(wrapper);
/*     */   }
/*     */ 
/*     */   public void runTask(Task task)
/*     */   {
/* 204 */     BasicTaskWrapper wrapper = new BasicTaskWrapper(task);
/* 205 */     runTaskWrapper(wrapper);
/*     */   }
/*     */ 
/*     */   public void run(Runnable runnable)
/*     */   {
/* 210 */     run(runnable, 0L, 0L);
/*     */   }
/*     */ 
/*     */   public void run(Runnable runnable, long startTimeout, long completeTimeout)
/*     */   {
/* 215 */     RunnableTaskWrapper wrapper = new RunnableTaskWrapper(runnable, startTimeout, completeTimeout);
/* 216 */     runTaskWrapper(wrapper);
/*     */   }
/*     */ 
/*     */   public ThreadGroup getThreadGroup()
/*     */   {
/* 221 */     return this.threadGroup;
/*     */   }
/*     */ 
/*     */   public String getName()
/*     */   {
/* 228 */     return this.name;
/*     */   }
/*     */ 
/*     */   public void setName(String name)
/*     */   {
/* 233 */     this.name = name;
/*     */   }
/*     */ 
/*     */   public int getPoolNumber()
/*     */   {
/* 238 */     return this.poolNumber;
/*     */   }
/*     */ 
/*     */   public String getThreadGroupName()
/*     */   {
/* 243 */     return this.threadGroup.getName();
/*     */   }
/*     */ 
/*     */   public void setThreadGroupName(String threadGroupName)
/*     */   {
/*     */     ThreadGroup group;
/* 249 */     synchronized (threadGroups)
/*     */     {
/* 251 */       group = (ThreadGroup)threadGroups.get(threadGroupName);
/* 252 */       if (group == null)
/*     */       {
/* 254 */         group = new ThreadGroup(JBOSS_THREAD_GROUP, threadGroupName);
/* 255 */         threadGroups.put(threadGroupName, group);
/*     */       }
/*     */     }
/* 258 */     this.threadGroup = group;
/*     */   }
/*     */ 
/*     */   public int getQueueSize()
/*     */   {
/* 263 */     return this.queue.size();
/*     */   }
/*     */ 
/*     */   public int getMaximumQueueSize()
/*     */   {
/* 268 */     int maxSize = this.queue.remainingCapacity() + this.queue.size();
/* 269 */     return maxSize;
/*     */   }
/*     */ 
/*     */   public void setMaximumQueueSize(int size)
/*     */   {
/* 282 */     ArrayList tmp = new ArrayList();
/* 283 */     this.queue.drainTo(tmp);
/* 284 */     this.queue = new LinkedBlockingQueue(size);
/* 285 */     this.queue.addAll(tmp);
/*     */ 
/* 287 */     ThreadFactory tf = this.executor.getThreadFactory();
/* 288 */     RejectedExecutionHandler handler = this.executor.getRejectedExecutionHandler();
/* 289 */     long keepAlive = this.executor.getKeepAliveTime(TimeUnit.SECONDS);
/* 290 */     int cs = this.executor.getCorePoolSize();
/* 291 */     int mcs = this.executor.getMaximumPoolSize();
/* 292 */     this.executor = new ThreadPoolExecutor(cs, mcs, keepAlive, TimeUnit.SECONDS, this.queue);
/* 293 */     this.executor.setThreadFactory(tf);
/* 294 */     this.executor.setRejectedExecutionHandler(handler);
/*     */   }
/*     */ 
/*     */   public int getPoolSize()
/*     */   {
/* 299 */     return this.executor.getPoolSize();
/*     */   }
/*     */ 
/*     */   public int getMinimumPoolSize()
/*     */   {
/* 304 */     return this.executor.getCorePoolSize();
/*     */   }
/*     */ 
/*     */   public void setMinimumPoolSize(int size)
/*     */   {
/* 309 */     synchronized (this.executor)
/*     */     {
/* 312 */       if (this.executor.getMaximumPoolSize() < size)
/*     */       {
/* 314 */         this.executor.setCorePoolSize(size);
/* 315 */         this.executor.setMaximumPoolSize(size);
/*     */       }
/*     */     }
/*     */   }
/*     */ 
/*     */   public int getMaximumPoolSize()
/*     */   {
/* 322 */     return this.executor.getMaximumPoolSize();
/*     */   }
/*     */ 
/*     */   public void setMaximumPoolSize(int size)
/*     */   {
/* 327 */     synchronized (this.executor)
/*     */     {
/* 329 */       this.executor.setCorePoolSize(size);
/* 330 */       this.executor.setMaximumPoolSize(size);
/*     */     }
/*     */   }
/*     */ 
/*     */   public long getKeepAliveTime()
/*     */   {
/* 336 */     return this.executor.getKeepAliveTime(TimeUnit.MILLISECONDS);
/*     */   }
/*     */ 
/*     */   public void setKeepAliveTime(long time)
/*     */   {
/* 341 */     this.executor.setKeepAliveTime(time, TimeUnit.MILLISECONDS);
/*     */   }
/*     */ 
/*     */   public BlockingMode getBlockingMode()
/*     */   {
/* 346 */     return this.blockingMode;
/*     */   }
/*     */ 
/*     */   public void setBlockingMode(BlockingMode mode)
/*     */   {
/* 351 */     this.blockingMode = mode;
/*     */ 
/* 353 */     if (this.blockingMode == BlockingMode.RUN)
/*     */     {
/* 355 */       this.executor.setRejectedExecutionHandler(new ThreadPoolExecutor.CallerRunsPolicy());
/*     */     }
/* 357 */     else if (this.blockingMode == BlockingMode.WAIT)
/*     */     {
/* 359 */       this.executor.setRejectedExecutionHandler(new ThreadPoolExecutor.CallerRunsPolicy());
/*     */     }
/* 361 */     else if (this.blockingMode == BlockingMode.DISCARD)
/*     */     {
/* 363 */       this.executor.setRejectedExecutionHandler(new ThreadPoolExecutor.DiscardPolicy());
/*     */     }
/* 365 */     else if (this.blockingMode == BlockingMode.DISCARD_OLDEST)
/*     */     {
/* 367 */       this.executor.setRejectedExecutionHandler(new ThreadPoolExecutor.DiscardOldestPolicy());
/*     */     }
/* 369 */     else if (this.blockingMode == BlockingMode.ABORT)
/*     */     {
/* 371 */       this.executor.setRejectedExecutionHandler(new ThreadPoolExecutor.AbortPolicy());
/*     */     }
/*     */     else
/*     */     {
/* 375 */       throw new IllegalArgumentException("Failed to recognize mode: " + mode);
/*     */     }
/*     */   }
/*     */ 
/*     */   public void setBlockingMode(String name)
/*     */   {
/* 385 */     this.blockingMode = BlockingMode.toBlockingMode(name);
/* 386 */     if (this.blockingMode == null)
/* 387 */       this.blockingMode = BlockingMode.ABORT;
/*     */   }
/*     */ 
/*     */   public void setBlockingModeString(String name)
/*     */   {
/* 398 */     this.blockingMode = BlockingMode.toBlockingMode(name);
/* 399 */     if (this.blockingMode == null)
/* 400 */       this.blockingMode = BlockingMode.ABORT;
/*     */   }
/*     */ 
/*     */   public ThreadPool getInstance()
/*     */   {
/* 405 */     return this;
/*     */   }
/*     */ 
/*     */   public void stop()
/*     */   {
/* 410 */     stop(false);
/*     */   }
/*     */ 
/*     */   public String toString()
/*     */   {
/* 417 */     return this.name + '(' + this.poolNumber + ')';
/*     */   }
/*     */ 
/*     */   protected void executeOnThread(TaskWrapper wrapper)
/*     */   {
/* 431 */     if (this.trace)
/* 432 */       log.trace("executeOnThread, wrapper=" + wrapper);
/* 433 */     wrapper.run();
/*     */   }
/*     */ 
/*     */   protected void execute(TaskWrapper wrapper)
/*     */   {
/* 443 */     if (this.trace)
/* 444 */       log.trace("execute, wrapper=" + wrapper);
/*     */     try
/*     */     {
/* 447 */       this.executor.execute(wrapper);
/*     */     }
/*     */     catch (Throwable t)
/*     */     {
/* 451 */       wrapper.rejectTask(new ThreadPoolFullException(t.toString()));
/*     */     }
/*     */   }
/*     */ 
/*     */   protected void waitForTask(TaskWrapper wrapper)
/*     */   {
/* 462 */     wrapper.waitForTask();
/*     */   }
/*     */ 
/*     */   protected synchronized void checkTimeoutMonitor()
/*     */   {
/* 470 */     if (this.timeoutTask == null)
/* 471 */       this.timeoutTask = new TimeoutMonitor(this.name, log);
/*     */   }
/*     */ 
/*     */   protected TimeoutInfo getNextTimeout() {
/* 475 */     TimeoutInfo info = null;
/* 476 */     if (!this.tasksWithTimeouts.isEmpty())
/*     */     {
/* 478 */       info = (TimeoutInfo)this.tasksWithTimeouts.remove();
/*     */     }
/* 480 */     return info;
/*     */   }
/*     */ 
/*     */   private class TimeoutMonitor
/*     */     implements Runnable
/*     */   {
/*     */     final Logger log;
/*     */ 
/*     */     TimeoutMonitor(String name, Logger log)
/*     */     {
/* 570 */       this.log = log;
/* 571 */       Thread t = new Thread(this, name + " TimeoutMonitor");
/* 572 */       t.setDaemon(true);
/* 573 */       t.start();
/*     */     }
/*     */ 
/*     */     public void run()
/*     */     {
/* 591 */       boolean isStopped = BasicThreadPool.this.stopped.get();
/* 592 */       while (!isStopped)
/*     */       {
/* 594 */         boolean trace = this.log.isTraceEnabled();
/*     */         try
/*     */         {
/* 597 */           BasicThreadPool.TimeoutInfo info = BasicThreadPool.this.getNextTimeout();
/* 598 */           if (info != null)
/*     */           {
/* 600 */             long now = System.currentTimeMillis();
/* 601 */             long timeToTimeout = info.getTaskCompletionTimeout(now);
/* 602 */             if (timeToTimeout > 0L)
/*     */             {
/* 604 */               if (trace)
/*     */               {
/* 606 */                 this.log.trace("Will check wrapper=" + info.getTaskWrapper() + " after " + timeToTimeout);
/*     */               }
/*     */ 
/* 609 */               Thread.sleep(timeToTimeout);
/*     */             }
/*     */ 
/* 612 */             TaskWrapper wrapper = info.getTaskWrapper();
/* 613 */             if (!wrapper.isComplete())
/*     */             {
/* 615 */               if (trace)
/* 616 */                 this.log.trace("Failed completion check for wrapper=" + wrapper);
/* 617 */               if (info.stopTask() == true)
/*     */               {
/* 620 */                 info.setTimeout(1000L);
/* 621 */                 BasicThreadPool.this.tasksWithTimeouts.add(info);
/* 622 */                 if (trace)
/* 623 */                   this.log.trace("Rescheduled completion check for wrapper=" + wrapper);
/*     */               }
/*     */             }
/*     */           }
/*     */           else
/*     */           {
/* 629 */             Thread.sleep(1000L);
/*     */           }
/*     */         }
/*     */         catch (InterruptedException e)
/*     */         {
/* 634 */           this.log.debug("Timeout monitor has been interrupted", e);
/*     */         }
/*     */         catch (Throwable e)
/*     */         {
/* 638 */           this.log.debug("Timeout monitor saw unexpected error", e);
/*     */         }
/* 640 */         isStopped = BasicThreadPool.this.stopped.get();
/*     */       }
/*     */     }
/*     */   }
/*     */ 
/*     */   private static class TimeoutInfo
/*     */     implements Comparable
/*     */   {
/*     */     long start;
/*     */     long timeoutMS;
/*     */     TaskWrapper wrapper;
/*     */     boolean firstStop;
/*     */ 
/*     */     TimeoutInfo(TaskWrapper wrapper, long timeout)
/*     */     {
/* 511 */       this.start = System.currentTimeMillis();
/* 512 */       this.timeoutMS = (this.start + timeout);
/* 513 */       this.wrapper = wrapper;
/*     */     }
/*     */ 
/*     */     public void setTimeout(long timeout) {
/* 517 */       this.start = System.currentTimeMillis();
/* 518 */       this.timeoutMS = (this.start + timeout);
/*     */     }
/*     */ 
/*     */     public int compareTo(Object o)
/*     */     {
/* 527 */       TimeoutInfo ti = (TimeoutInfo)o;
/* 528 */       long to0 = this.timeoutMS;
/* 529 */       long to1 = ti.timeoutMS;
/* 530 */       int diff = (int)(to0 - to1);
/* 531 */       return diff;
/*     */     }
/*     */ 
/*     */     TaskWrapper getTaskWrapper() {
/* 535 */       return this.wrapper;
/*     */     }
/*     */ 
/*     */     public long getTaskCompletionTimeout() {
/* 539 */       return this.wrapper.getTaskCompletionTimeout();
/*     */     }
/*     */ 
/*     */     public long getTaskCompletionTimeout(long now)
/*     */     {
/* 547 */       return this.timeoutMS - now;
/*     */     }
/*     */ 
/*     */     public boolean stopTask()
/*     */     {
/* 555 */       this.wrapper.stopTask();
/* 556 */       boolean wasFirstStop = !this.firstStop;
/* 557 */       this.firstStop = true;
/* 558 */       return wasFirstStop;
/*     */     }
/*     */   }
/*     */ 
/*     */   private class ThreadPoolThreadFactory
/*     */     implements ThreadFactory
/*     */   {
/*     */     private ThreadPoolThreadFactory()
/*     */     {
/*     */     }
/*     */ 
/*     */     public Thread newThread(Runnable runnable)
/*     */     {
/* 494 */       String threadName = BasicThreadPool.this.toString() + "-" + BasicThreadPool.this.lastThreadNumber.incrementAndGet();
/* 495 */       Thread thread = new Thread(BasicThreadPool.this.threadGroup, runnable, threadName);
/* 496 */       thread.setDaemon(true);
/* 497 */       return thread;
/*     */     }
/*     */   }
/*     */ }

/* Location:           /home/mnovotny/projects/EMBEDDED_JBOSS_BETA3_COMMUNITY/embedded/output/lib/embedded-jboss/lib/jboss-embedded-all.jar
 * Qualified Name:     org.jboss.util.threadpool.BasicThreadPool
 * JD-Core Version:    0.6.0
 */