/*     */ package org.jboss.remoting.transport.multiplex;
/*     */ 
/*     */ import java.io.BufferedInputStream;
/*     */ import java.io.EOFException;
/*     */ import java.io.IOException;
/*     */ import java.io.InputStream;
/*     */ import java.io.OutputStream;
/*     */ import java.net.Socket;
/*     */ import java.nio.ByteBuffer;
/*     */ import java.nio.channels.ClosedSelectorException;
/*     */ import java.nio.channels.SelectableChannel;
/*     */ import java.nio.channels.SelectionKey;
/*     */ import java.nio.channels.Selector;
/*     */ import java.nio.channels.SocketChannel;
/*     */ import java.util.Collections;
/*     */ import java.util.HashMap;
/*     */ import java.util.HashSet;
/*     */ import java.util.Iterator;
/*     */ import java.util.Map;
/*     */ import java.util.Set;
/*     */ import org.jboss.logging.Logger;
/*     */ import org.jboss.remoting.transport.multiplex.utility.StoppableThread;
/*     */ 
/*     */ public class InputMultiplexor
/*     */ {
/*  70 */   protected static final Logger log = Logger.getLogger(InputMultiplexor.class);
/*     */   private static final int HEADER_LENGTH = 7;
/*     */   private int bufferSize;
/*     */   private int maxErrors;
/*     */ 
/*     */   public InputMultiplexor(Map configuration)
/*     */   {
/*  79 */     this.bufferSize = Multiplex.getOneParameter(configuration, "bufferSize", "multiplex.inputBufferSize", 4096);
/*     */ 
/*  85 */     this.maxErrors = Multiplex.getOneParameter(configuration, "maxErrors", "multiplex.inputMaxErrors", 3);
/*     */   }
/*     */ 
/*     */   public MultiGroupInputThread getaMultiGroupInputThread()
/*     */     throws IOException
/*     */   {
/* 100 */     return new MultiGroupInputThread();
/*     */   }
/*     */ 
/*     */   public SingleGroupInputThread getaSingleGroupInputThread(MultiplexingManager manager, Socket socket, OutputStream os)
/*     */     throws IOException
/*     */   {
/* 110 */     return new SingleGroupInputThread(manager, socket, os);
/*     */   }
/*     */ 
/*     */   private static class CorruptedStreamException extends IOException
/*     */   {
/*     */     CorruptedStreamException(String message)
/*     */     {
/* 657 */       super();
/*     */     }
/*     */   }
/*     */ 
/*     */   class SingleGroupInputThread extends StoppableThread
/*     */   {
/*     */     private InputStream is;
/*     */     private OutputStream currentOutputStream;
/* 488 */     private byte[] dataBytes = new byte[InputMultiplexor.this.bufferSize];
/*     */     private MultiplexingManager manager;
/* 490 */     private int dataInCount = 0;
/*     */     private int errorCount;
/*     */     private boolean eof;
/* 495 */     private byte[] headerBytes = new byte[7];
/*     */     private int headerCount;
/*     */     private byte version;
/*     */     private int destination;
/*     */     private short size;
/*     */     private boolean trace;
/*     */     private boolean debug;
/*     */     private boolean info;
/*     */ 
/*     */     public SingleGroupInputThread(MultiplexingManager manager, Socket socket, OutputStream os)
/*     */       throws IOException
/*     */     {
/* 509 */       this.is = new BufferedInputStream(socket.getInputStream());
/* 510 */       this.manager = manager;
/* 511 */       this.currentOutputStream = os;
/*     */ 
/* 513 */       this.trace = InputMultiplexor.log.isTraceEnabled();
/* 514 */       this.debug = InputMultiplexor.log.isDebugEnabled();
/* 515 */       this.info = InputMultiplexor.log.isInfoEnabled();
/*     */     }
/*     */ 
/*     */     public void shutdown()
/*     */     {
/* 522 */       super.shutdown();
/* 523 */       InputMultiplexor.log.info("interrupting input thread");
/* 524 */       interrupt();
/*     */     }
/*     */ 
/*     */     protected void doInit()
/*     */     {
/* 533 */       InputMultiplexor.log.debug("SingleGroupInputThread thread starting");
/*     */     }
/*     */ 
/*     */     protected void doRun()
/*     */     {
/*     */       try
/*     */       {
/* 545 */         if (!completeHeader())
/*     */         {
/* 547 */           this.eof = true;
/* 548 */           jsr 281;
/*     */         }
/*     */ 
/* 551 */         SocketId socketId = new SocketId(this.destination);
/* 552 */         this.currentOutputStream = this.manager.getOutputStreamByLocalSocket(socketId);
/* 553 */         if (this.currentOutputStream == null)
/*     */         {
/* 558 */           InputMultiplexor.log.info("unknown socket id: " + this.destination);
/* 559 */           this.currentOutputStream = this.manager.getConnectedOutputStream(socketId);
/*     */         }
/*     */ 
/* 562 */         int bytesRead = 0;
/* 563 */         while (bytesRead < this.size)
/*     */         {
/* 565 */           int n = this.is.read(this.dataBytes, 0, this.size - bytesRead);
/* 566 */           if (n < 0)
/*     */           {
/* 568 */             this.eof = true;
/* 569 */             jsr 168;
/*     */           }
/*     */ 
/* 572 */           this.currentOutputStream.write(this.dataBytes, 0, n);
/* 573 */           bytesRead += n;
/*     */ 
/* 575 */           if (this.trace)
/*     */           {
/* 577 */             for (int i = 0; i < n; i++)
/* 578 */               InputMultiplexor.log.trace("" + this.dataBytes[i]);
/*     */           }
/*     */         }
/*     */       }
/*     */       catch (EOFException e)
/*     */       {
/* 584 */         this.eof = true;
/* 585 */         InputMultiplexor.log.info("end of file");
/*     */       }
/*     */       catch (IOException e)
/*     */       {
/* 589 */         if (++this.errorCount > InputMultiplexor.this.maxErrors)
/*     */         {
/* 591 */           this.manager.setReadException(e);
/* 592 */           super.shutdown();
/* 593 */           InputMultiplexor.log.error(e);
/*     */         }
/*     */         else {
/* 596 */           InputMultiplexor.log.warn(e);
/*     */         }
/*     */       }
/*     */       finally {
/* 600 */         if (this.eof)
/*     */         {
/* 602 */           super.shutdown();
/* 603 */           this.manager.setEOF();
/*     */         }
/*     */       }
/*     */     }
/*     */ 
/*     */     private boolean completeHeader()
/*     */       throws IOException
/*     */     {
/* 611 */       while (this.headerCount < 7)
/*     */       {
/* 613 */         int n = this.is.read(this.headerBytes, this.headerCount, 7 - this.headerCount);
/*     */ 
/* 616 */         if (n < 0) {
/* 617 */           return false;
/*     */         }
/* 619 */         this.headerCount += n;
/*     */       }
/*     */ 
/* 623 */       this.headerCount = 0;
/*     */ 
/* 625 */       this.version = this.headerBytes[0];
/* 626 */       this.destination = (this.headerBytes[1] << 24 | 0xFF0000 & this.headerBytes[2] << 16 | 0xFF00 & this.headerBytes[3] << 8 | 0xFF & this.headerBytes[4]);
/*     */ 
/* 628 */       this.size = (short)(0xFF00 & this.headerBytes[5] << 8 | 0xFF & this.headerBytes[6]);
/*     */ 
/* 630 */       if (this.trace)
/*     */       {
/* 632 */         InputMultiplexor.log.trace("version:     " + this.version);
/* 633 */         InputMultiplexor.log.trace("destination: " + this.destination);
/* 634 */         InputMultiplexor.log.trace("size:        " + this.size);
/*     */       }
/*     */ 
/* 637 */       if ((this.size < 0) || (InputMultiplexor.this.bufferSize < this.size)) {
/* 638 */         throw new InputMultiplexor.CorruptedStreamException("invalid chunk size read on: " + this.manager + ": " + this.size);
/*     */       }
/* 640 */       if (this.version != 0) {
/* 641 */         throw new InputMultiplexor.CorruptedStreamException("invalid version read on: " + this.manager + ": " + this.version);
/*     */       }
/* 643 */       return true;
/*     */     }
/*     */ 
/*     */     protected void doShutDown()
/*     */     {
/* 649 */       InputMultiplexor.log.debug("input thread: data bytes read: " + this.dataInCount);
/* 650 */       InputMultiplexor.log.debug("input thread shutting down");
/*     */     }
/*     */   }
/*     */ 
/*     */   public class MultiGroupInputThread extends StoppableThread
/*     */   {
/*     */     private static final String errMsg1 = "An existing connection was forcibly closed by the remote host";
/*     */     private static final String errMsg2 = "An established connection was aborted by the software in your host machine";
/*     */     private Map managerProcessorMap;
/* 120 */     private Set socketGroupsToBeRegistered = new HashSet();
/* 121 */     private Set tempSocketGroupSet = new HashSet();
/*     */     private boolean socketGroupsAreWaiting;
/*     */     private Selector selector;
/*     */     private ByteBuffer buffer;
/*     */     private byte[] data;
/*     */     private boolean trace;
/*     */     private boolean debug;
/*     */     private boolean info;
/*     */ 
/*     */     public MultiGroupInputThread()
/*     */       throws IOException
/*     */     {
/* 134 */       this.managerProcessorMap = Collections.synchronizedMap(new HashMap());
/* 135 */       this.selector = Selector.open();
/* 136 */       this.buffer = ByteBuffer.allocate(InputMultiplexor.this.bufferSize);
/* 137 */       this.data = new byte[InputMultiplexor.this.bufferSize];
/*     */ 
/* 139 */       this.trace = InputMultiplexor.log.isTraceEnabled();
/* 140 */       this.debug = InputMultiplexor.log.isDebugEnabled();
/* 141 */       this.info = InputMultiplexor.log.isInfoEnabled();
/*     */     }
/*     */ 
/*     */     public void registerSocketGroup(MultiplexingManager manager)
/*     */       throws IOException
/*     */     {
/* 153 */       if (this.debug) InputMultiplexor.log.debug(" accepting socket group for registration: " + manager);
/*     */ 
/* 155 */       synchronized (this.socketGroupsToBeRegistered)
/*     */       {
/* 157 */         this.socketGroupsToBeRegistered.add(manager);
/* 158 */         this.socketGroupsAreWaiting = true;
/*     */       }
/*     */     }
/*     */ 
/*     */     protected void doRegistration()
/*     */     {
/* 165 */       this.tempSocketGroupSet.clear();
/* 166 */       synchronized (this.socketGroupsToBeRegistered)
/*     */       {
/* 168 */         this.tempSocketGroupSet.addAll(this.socketGroupsToBeRegistered);
/* 169 */         this.socketGroupsToBeRegistered.clear();
/* 170 */         this.socketGroupsAreWaiting = false;
/*     */       }
/*     */ 
/* 173 */       Iterator it = this.tempSocketGroupSet.iterator();
/* 174 */       while (it.hasNext())
/*     */       {
/* 176 */         MultiplexingManager manager = (MultiplexingManager)it.next();
/* 177 */         GroupProcessor groupProcessor = new GroupProcessor(manager);
/* 178 */         SelectableChannel channel = manager.getSocket().getChannel();
/*     */         try
/*     */         {
/* 182 */           SelectionKey key = channel.register(this.selector, 1, groupProcessor);
/* 183 */           groupProcessor.setKey(key);
/* 184 */           this.managerProcessorMap.put(manager, groupProcessor);
/*     */         }
/*     */         catch (IOException e)
/*     */         {
/* 189 */           InputMultiplexor.log.warn(e);
/*     */         }
/*     */       }
/*     */     }
/*     */ 
/*     */     public void unregisterSocketGroup(MultiplexingManager manager)
/*     */     {
/* 202 */       GroupProcessor groupProcessor = (GroupProcessor)this.managerProcessorMap.get(manager);
/* 203 */       if (groupProcessor == null)
/*     */       {
/* 205 */         InputMultiplexor.log.debug("attempting to unregister unknown MultiplexingManager: " + manager);
/* 206 */         return;
/*     */       }
/*     */ 
/* 209 */       SelectionKey key = groupProcessor.getKey();
/* 210 */       key.cancel();
/* 211 */       this.managerProcessorMap.remove(manager);
/* 212 */       if (this.debug) InputMultiplexor.log.debug("unregistered socket group:" + manager);
/*     */     }
/*     */ 
/*     */     public void shutdown()
/*     */     {
/* 219 */       super.shutdown();
/*     */       try
/*     */       {
/* 222 */         this.selector.close();
/*     */       }
/*     */       catch (IOException e)
/*     */       {
/* 226 */         InputMultiplexor.log.error("unable to close selector", e);
/*     */       }
/* 228 */       interrupt();
/*     */     }
/*     */ 
/*     */     protected void doInit()
/*     */     {
/* 234 */       InputMultiplexor.log.debug("MultiGroupInputThread thread starting");
/*     */     }
/*     */ 
/*     */     protected void doRun()
/*     */     {
/* 240 */       InputMultiplexor.log.debug("entering doRun()");
/* 241 */       Set keys = null;
/*     */       try
/*     */       {
/*     */         while (true)
/*     */         {
/* 247 */           if (!this.running) {
/* 248 */             return;
/*     */           }
/* 250 */           if (this.socketGroupsAreWaiting) {
/* 251 */             doRegistration();
/*     */           }
/* 253 */           this.selector.select(200L);
/* 254 */           keys = this.selector.selectedKeys();
/*     */ 
/* 256 */           if (!keys.isEmpty())
/* 257 */             break;
/*     */         }
/*     */       }
/*     */       catch (IOException e)
/*     */       {
/* 262 */         InputMultiplexor.log.info(e);
/*     */       }
/*     */       catch (ClosedSelectorException e)
/*     */       {
/* 266 */         InputMultiplexor.log.info("Selector is closed: shutting down input thread");
/* 267 */         super.shutdown();
/* 268 */         return;
/*     */       }
/*     */ 
/* 271 */       if (this.trace)
/*     */       {
/* 273 */         InputMultiplexor.log.trace("keys: " + this.selector.keys().size());
/* 274 */         InputMultiplexor.log.trace("selected keys: " + keys.size());
/*     */       }
/*     */ 
/* 277 */       Iterator it = keys.iterator();
/* 278 */       while (it.hasNext())
/*     */       {
/* 280 */         SelectionKey key = (SelectionKey)it.next();
/* 281 */         it.remove();
/* 282 */         GroupProcessor groupProcessor = (GroupProcessor)key.attachment();
/*     */ 
/* 284 */         if (groupProcessor == null)
/*     */         {
/* 286 */           if (key.isValid()) {
/* 287 */             InputMultiplexor.log.error("valid SelectionKey has no attachment: " + key); continue;
/*     */           }
/*     */ 
/*     */         }
/*     */ 
/* 292 */         groupProcessor.processChannel(key);
/*     */       }
/*     */     }
/*     */ 
/*     */     protected void doShutDown()
/*     */     {
/* 299 */       InputMultiplexor.log.debug("MultiGroupInputThread shutting down"); } 
/* 306 */     class GroupProcessor { private byte[] b = new byte[7];
/*     */       private int headerCount;
/*     */       private byte version;
/*     */       private int destination;
/*     */       private short size;
/*     */       private MultiplexingManager manager;
/*     */       private OutputStream outputStream;
/*     */       private SelectionKey key;
/*     */       private int errorCount;
/*     */ 
/* 320 */       public GroupProcessor(MultiplexingManager manager) { this.manager = manager;
/*     */       }
/*     */ 
/*     */       public void processChannel(SelectionKey key)
/*     */       {
/* 325 */         InputMultiplexor.log.debug("processChannel()");
/* 326 */         SocketChannel channel = (SocketChannel)key.channel();
/* 327 */         InputMultiplexor.MultiGroupInputThread.this.buffer.clear();
/*     */         try
/*     */         {
/* 331 */           if (channel.read(InputMultiplexor.MultiGroupInputThread.this.buffer) < 0) {
/* 332 */             throw new EOFException();
/*     */           }
/* 334 */           InputMultiplexor.MultiGroupInputThread.this.buffer.flip();
/*     */ 
/* 336 */           if (InputMultiplexor.MultiGroupInputThread.this.debug) {
/* 337 */             InputMultiplexor.log.debug("read: " + InputMultiplexor.MultiGroupInputThread.this.buffer.remaining());
/*     */           }
/* 339 */           while (InputMultiplexor.MultiGroupInputThread.this.buffer.hasRemaining())
/*     */           {
/* 341 */             if ((this.headerCount < 7) || (this.size == 0))
/*     */             {
/* 344 */               completeHeader(InputMultiplexor.MultiGroupInputThread.this.buffer);
/*     */ 
/* 346 */               if (this.headerCount < 7) {
/* 347 */                 return;
/*     */               }
/* 349 */               SocketId socketId = new SocketId(this.destination);
/* 350 */               this.outputStream = this.manager.getOutputStreamByLocalSocket(socketId);
/* 351 */               if (this.outputStream == null)
/*     */               {
/* 356 */                 InputMultiplexor.log.info("unknown socket id: " + this.destination);
/* 357 */                 this.outputStream = this.manager.getConnectedOutputStream(socketId);
/*     */               }
/*     */ 
/* 360 */               if (!InputMultiplexor.MultiGroupInputThread.this.buffer.hasRemaining()) {
/* 361 */                 return;
/*     */               }
/*     */             }
/* 364 */             int n = Math.min(this.size, InputMultiplexor.MultiGroupInputThread.this.buffer.remaining());
/* 365 */             InputMultiplexor.MultiGroupInputThread.this.buffer.get(InputMultiplexor.MultiGroupInputThread.this.data, 0, n);
/* 366 */             this.outputStream.write(InputMultiplexor.MultiGroupInputThread.this.data, 0, n);
/*     */ 
/* 368 */             if (InputMultiplexor.MultiGroupInputThread.this.trace)
/*     */             {
/* 370 */               InputMultiplexor.log.trace("received " + n + " bytes for socket: " + this.destination);
/* 371 */               for (int i = 0; i < n; i++) {
/* 372 */                 InputMultiplexor.log.trace("" + (0xFF & InputMultiplexor.MultiGroupInputThread.this.data[i]));
/*     */               }
/*     */             }
/* 375 */             this.size = (short)(this.size - n);
/* 376 */             if (this.size == 0)
/* 377 */               this.headerCount = 0;
/*     */           }
/*     */         }
/*     */         catch (IOException e)
/*     */         {
/* 382 */           handleChannelException(e, key, channel);
/*     */         }
/*     */         catch (Throwable t)
/*     */         {
/* 386 */           InputMultiplexor.log.error("doRun()");
/* 387 */           InputMultiplexor.log.error(t);
/*     */         }
/*     */       }
/*     */ 
/*     */       public SelectionKey getKey()
/*     */       {
/* 393 */         return this.key;
/*     */       }
/*     */ 
/*     */       public void setKey(SelectionKey key)
/*     */       {
/* 398 */         this.key = key;
/*     */       }
/*     */ 
/*     */       private void completeHeader(ByteBuffer bb) throws IOException
/*     */       {
/* 403 */         int n = Math.min(bb.remaining(), 7 - this.headerCount);
/* 404 */         bb.get(this.b, this.headerCount, n);
/* 405 */         this.headerCount += n;
/*     */ 
/* 407 */         if (this.headerCount == 7)
/*     */         {
/* 409 */           this.version = this.b[0];
/* 410 */           this.destination = (this.b[1] << 24 | 0xFF0000 & this.b[2] << 16 | 0xFF00 & this.b[3] << 8 | 0xFF & this.b[4]);
/*     */ 
/* 412 */           this.size = (short)(0xFF00 & this.b[5] << 8 | 0xFF & this.b[6]);
/*     */ 
/* 415 */           if ((this.size < 0) || (InputMultiplexor.this.bufferSize < this.size)) {
/* 416 */             throw new InputMultiplexor.CorruptedStreamException("invalid chunk size read on: " + this.manager + ": " + this.size);
/*     */           }
/* 418 */           if (this.version != 0)
/* 419 */             throw new InputMultiplexor.CorruptedStreamException("invalid version read on: " + this.manager + ": " + this.version);
/*     */         }
/*     */       }
/*     */ 
/*     */       private void handleChannelException(IOException e, SelectionKey key, SocketChannel channel)
/*     */       {
/* 425 */         InputMultiplexor.log.error("handleChannelException()");
/* 426 */         InputMultiplexor.log.error(e);
/*     */         try
/*     */         {
/* 429 */           if (!channel.isOpen())
/*     */           {
/* 431 */             key.cancel();
/* 432 */             return;
/*     */           }
/*     */ 
/* 435 */           if ((e instanceof EOFException))
/*     */           {
/* 437 */             key.cancel();
/* 438 */             this.manager.setEOF();
/* 439 */             InputMultiplexor.log.debug(e);
/* 440 */             return;
/*     */           }
/*     */ 
/* 443 */           if (++this.errorCount > InputMultiplexor.this.maxErrors)
/*     */           {
/* 445 */             this.manager.setReadException(e);
/* 446 */             channel.close();
/* 447 */             key.cancel();
/* 448 */             InputMultiplexor.log.error(e);
/* 449 */             InputMultiplexor.log.error("error count exceeds max errors: " + this.errorCount);
/* 450 */             return;
/*     */           }
/*     */ 
/* 453 */           Socket socket = channel.socket();
/* 454 */           String message = e.getMessage();
/*     */ 
/* 456 */           if ((socket.isClosed()) || (socket.isInputShutdown()) || ("An existing connection was forcibly closed by the remote host".equals(message)) || ("An established connection was aborted by the software in your host machine".equals(message)) || ((e instanceof InputMultiplexor.CorruptedStreamException)))
/*     */           {
/* 460 */             this.manager.setReadException(e);
/* 461 */             channel.close();
/* 462 */             key.cancel();
/* 463 */             InputMultiplexor.log.info(e);
/* 464 */             return;
/*     */           }
/*     */ 
/* 468 */           InputMultiplexor.log.warn(e);
/*     */         }
/*     */         catch (IOException e2)
/*     */         {
/* 472 */           InputMultiplexor.log.error("problem closing channel: " + this.manager, e2);
/*     */         }
/*     */       }
/*     */ 
/*     */       public int getDestination() {
/* 476 */         return this.destination; } 
/* 477 */       public short getSize() { return this.size; } 
/* 478 */       public byte getVersion() { return this.version; } 
/* 479 */       public OutputStream getOutputStream() { return this.outputStream;
/*     */       }
/*     */     }
/*     */   }
/*     */ }

/* Location:           /home/mnovotny/projects/EMBEDDED_JBOSS_BETA3_COMMUNITY/embedded/output/lib/embedded-jboss/lib/jboss-embedded-all.jar
 * Qualified Name:     org.jboss.remoting.transport.multiplex.InputMultiplexor
 * JD-Core Version:    0.6.0
 */