/*     */ package org.jboss.messaging.core.impl.tx;
/*     */ 
/*     */ import java.util.ArrayList;
/*     */ import java.util.HashMap;
/*     */ import java.util.Iterator;
/*     */ import java.util.List;
/*     */ import java.util.Map;
/*     */ import javax.transaction.xa.Xid;
/*     */ import org.jboss.logging.Logger;
/*     */ 
/*     */ public class Transaction
/*     */ {
/*  49 */   private static final Logger log = Logger.getLogger(Transaction.class);
/*     */ 
/*  53 */   private boolean trace = log.isTraceEnabled();
/*     */   private long id;
/*     */   private int state;
/*     */   private Xid xid;
/*     */   private List callbacks;
/*     */   private Map callbackMap;
/*     */   private boolean recoveredFromStorage;
/*     */   protected TransactionRepository repository;
/*     */   protected TxCallback firstCallback;
/*     */   public static final int STATE_ACTIVE = 0;
/*     */   public static final int STATE_PREPARED = 1;
/*     */   public static final int STATE_COMMITTED = 2;
/*     */   public static final int STATE_ROLLEDBACK = 3;
/*     */   public static final int STATE_ROLLBACK_ONLY = 4;
/*     */ 
/*     */   public static String stateToString(int state)
/*     */   {
/*  91 */     if (state == 0)
/*     */     {
/*  93 */       return "ACTIVE";
/*     */     }
/*  95 */     if (state == 1)
/*     */     {
/*  97 */       return "PREPARED";
/*     */     }
/*  99 */     if (state == 2)
/*     */     {
/* 101 */       return "COMMITTED";
/*     */     }
/* 103 */     if (state == 3)
/*     */     {
/* 105 */       return "ROLLEDBACK";
/*     */     }
/* 107 */     if (state == 4)
/*     */     {
/* 109 */       return "ROLLBACK_ONLY";
/*     */     }
/*     */ 
/* 113 */     return "UNKNOWN";
/*     */   }
/*     */ 
/*     */   Transaction(long id)
/*     */   {
/* 121 */     this.id = id;
/* 122 */     this.state = 0;
/* 123 */     this.callbacks = new ArrayList();
/* 124 */     this.callbackMap = new HashMap();
/*     */   }
/*     */ 
/*     */   Transaction(long id, Xid xid, TransactionRepository tr)
/*     */   {
/* 129 */     this(id);
/* 130 */     this.xid = xid;
/* 131 */     this.repository = tr;
/*     */   }
/*     */ 
/*     */   public int getState()
/*     */   {
/* 138 */     return this.state;
/*     */   }
/*     */ 
/*     */   public Xid getXid()
/*     */   {
/* 143 */     return this.xid;
/*     */   }
/*     */ 
/*     */   public void addCallback(TxCallback callback, Object key)
/*     */   {
/* 148 */     this.callbacks.add(callback);
/*     */ 
/* 150 */     this.callbackMap.put(key, callback);
/*     */   }
/*     */ 
/*     */   public void addFirstCallback(TxCallback callback, Object key)
/*     */   {
/* 155 */     if (this.firstCallback != null)
/*     */     {
/* 157 */       throw new IllegalStateException("There is already a first callback");
/*     */     }
/*     */ 
/* 160 */     this.firstCallback = callback;
/*     */ 
/* 162 */     this.callbackMap.put(key, callback);
/*     */   }
/*     */ 
/*     */   public TxCallback getCallback(Object key)
/*     */   {
/* 167 */     return (TxCallback)this.callbackMap.get(key);
/*     */   }
/*     */ 
/*     */   public synchronized void commit() throws Exception
/*     */   {
/* 172 */     if (this.state == 4)
/*     */     {
/* 174 */       throw new TransactionException("Transaction marked rollback only, cannot commit");
/*     */     }
/* 176 */     if (this.state == 2)
/*     */     {
/* 178 */       throw new TransactionException("Transaction already committed, cannot commit");
/*     */     }
/* 180 */     if (this.state == 3)
/*     */     {
/* 182 */       throw new TransactionException("Transaction already rolled back, cannot commit");
/*     */     }
/*     */ 
/* 185 */     if (this.recoveredFromStorage)
/*     */     {
/* 190 */       loadState();
/*     */     }
/*     */ 
/* 193 */     if (this.trace) log.trace(this + " executing before commit hooks");
/*     */ 
/* 195 */     boolean onePhase = this.state != 1;
/*     */ 
/* 197 */     if (this.firstCallback != null)
/*     */     {
/* 199 */       this.firstCallback.beforeCommit(onePhase);
/*     */     }
/*     */ 
/* 202 */     Iterator iter = this.callbacks.iterator();
/*     */ 
/* 204 */     while (iter.hasNext())
/*     */     {
/* 206 */       TxCallback callback = (TxCallback)iter.next();
/*     */ 
/* 208 */       callback.beforeCommit(onePhase);
/*     */     }
/*     */ 
/* 211 */     this.state = 2;
/*     */ 
/* 213 */     if (this.trace) log.trace(this + " committed");
/*     */ 
/* 215 */     iter = this.callbacks.iterator();
/*     */ 
/* 217 */     if (this.trace) log.trace(this + " executing after commit hooks");
/*     */ 
/* 219 */     if (this.firstCallback != null)
/*     */     {
/* 221 */       this.firstCallback.afterCommit(onePhase);
/*     */     }
/*     */ 
/* 224 */     while (iter.hasNext())
/*     */     {
/* 226 */       TxCallback callback = (TxCallback)iter.next();
/*     */ 
/* 228 */       callback.afterCommit(onePhase);
/*     */     }
/*     */ 
/* 231 */     this.callbacks = null;
/*     */ 
/* 233 */     this.callbackMap = null;
/*     */ 
/* 235 */     this.firstCallback = null;
/*     */ 
/* 237 */     if (this.repository != null)
/*     */     {
/* 239 */       this.repository.deleteTransaction(this);
/*     */     }
/*     */ 
/* 242 */     if (this.trace) log.trace(this + " commit process complete"); 
/*     */   }
/*     */ 
/*     */   public synchronized void prepare()
/*     */     throws Exception
/*     */   {
/* 247 */     if (this.state != 0)
/*     */     {
/* 249 */       throw new TransactionException("Transaction not active, cannot prepare");
/*     */     }
/*     */ 
/* 252 */     if (this.trace) log.trace(this + " executing before prepare hooks");
/*     */ 
/* 254 */     if (this.firstCallback != null)
/*     */     {
/* 256 */       this.firstCallback.beforePrepare();
/*     */     }
/*     */ 
/* 259 */     Iterator iter = this.callbacks.iterator();
/*     */ 
/* 261 */     while (iter.hasNext())
/*     */     {
/* 263 */       TxCallback callback = (TxCallback)iter.next();
/*     */ 
/* 265 */       callback.beforePrepare();
/*     */     }
/*     */ 
/* 268 */     this.state = 1;
/*     */ 
/* 270 */     if (this.trace) log.trace(this + " prepared");
/*     */ 
/* 272 */     if (this.firstCallback != null)
/*     */     {
/* 274 */       this.firstCallback.afterPrepare();
/*     */     }
/*     */ 
/* 277 */     iter = this.callbacks.iterator();
/*     */ 
/* 279 */     if (this.trace) log.trace(this + " executing after prepare hooks");
/*     */ 
/* 281 */     while (iter.hasNext())
/*     */     {
/* 283 */       TxCallback callback = (TxCallback)iter.next();
/*     */ 
/* 285 */       callback.afterPrepare();
/*     */     }
/*     */ 
/* 288 */     if (this.trace) log.trace(this + " prepare process complete"); 
/*     */   }
/*     */ 
/*     */   public synchronized void rollback()
/*     */     throws Exception
/*     */   {
/* 293 */     if (this.state == 2)
/*     */     {
/* 295 */       throw new TransactionException("Transaction already committed, cannot rollback");
/*     */     }
/* 297 */     if (this.state == 3)
/*     */     {
/* 299 */       throw new TransactionException("Transaction already rolled back, cannot rollback");
/*     */     }
/*     */ 
/* 302 */     if (this.recoveredFromStorage)
/*     */     {
/* 307 */       loadState();
/*     */     }
/*     */ 
/* 310 */     if (this.trace) log.trace(this + " executing before rollback hooks");
/*     */ 
/* 312 */     boolean onePhase = this.state != 1;
/*     */ 
/* 314 */     if (this.firstCallback != null)
/*     */     {
/* 316 */       this.firstCallback.beforeRollback(onePhase);
/*     */     }
/*     */ 
/* 319 */     for (Iterator i = this.callbacks.iterator(); i.hasNext(); )
/*     */     {
/* 321 */       TxCallback callback = (TxCallback)i.next();
/* 322 */       callback.beforeRollback(onePhase);
/*     */     }
/*     */ 
/* 325 */     this.state = 3;
/*     */ 
/* 327 */     if (this.trace) log.trace(this + " rolled back");
/*     */ 
/* 329 */     if (this.trace) log.trace(this + " executing after rollback hooks");
/*     */ 
/* 331 */     if (this.firstCallback != null)
/*     */     {
/* 333 */       this.firstCallback.afterRollback(onePhase);
/*     */     }
/*     */ 
/* 336 */     for (Iterator i = this.callbacks.iterator(); i.hasNext(); )
/*     */     {
/* 338 */       TxCallback callback = (TxCallback)i.next();
/* 339 */       callback.afterRollback(onePhase);
/*     */     }
/*     */ 
/* 342 */     this.callbacks = null;
/* 343 */     this.callbackMap = null;
/*     */ 
/* 345 */     if (this.repository != null)
/*     */     {
/* 347 */       this.repository.deleteTransaction(this);
/*     */     }
/*     */ 
/* 350 */     if (this.trace) log.trace(this + " rollback process complete"); 
/*     */   }
/*     */ 
/*     */   public void loadState()
/*     */     throws Exception
/*     */   {
/* 355 */     this.repository.handleReferences(this);
/*     */ 
/* 357 */     this.repository.handleAcks(this);
/*     */ 
/* 359 */     this.recoveredFromStorage = false;
/*     */   }
/*     */ 
/*     */   public synchronized void setRollbackOnly() throws Exception
/*     */   {
/* 364 */     if (this.trace) log.trace("setting ROLLBACK_ONLY on " + this);
/*     */ 
/* 366 */     this.state = 4;
/*     */   }
/*     */ 
/*     */   public long getId()
/*     */   {
/* 371 */     return this.id;
/*     */   }
/*     */ 
/*     */   public boolean isRecoveredFromStorage()
/*     */   {
/* 376 */     return this.recoveredFromStorage;
/*     */   }
/*     */ 
/*     */   public void setRecoveredFromStorage(boolean recoveredFromStorage)
/*     */   {
/* 381 */     this.recoveredFromStorage = recoveredFromStorage;
/*     */   }
/*     */ 
/*     */   public void setState(int state)
/*     */   {
/* 386 */     this.state = state;
/*     */   }
/*     */ 
/*     */   public String toString()
/*     */   {
/* 391 */     StringBuffer sb = new StringBuffer("TX(");
/* 392 */     sb.append(this.id);
/* 393 */     sb.append("):");
/* 394 */     sb.append(stateToString(this.state));
/* 395 */     return sb.toString();
/*     */   }
/*     */ }

/* Location:           /home/mnovotny/projects/EMBEDDED_JBOSS_BETA3_COMMUNITY/embedded/output/lib/embedded-jboss/lib/jboss-embedded-all.jar
 * Qualified Name:     org.jboss.messaging.core.impl.tx.Transaction
 * JD-Core Version:    0.6.0
 */