/*     */ package org.jboss.messaging.core.impl;
/*     */ 
/*     */ import java.util.ArrayList;
/*     */ import java.util.HashMap;
/*     */ import java.util.Iterator;
/*     */ import java.util.List;
/*     */ import java.util.Map;
/*     */ import org.jboss.logging.Logger;
/*     */ import org.jboss.messaging.core.contract.Message;
/*     */ import org.jboss.messaging.core.contract.MessageReference;
/*     */ import org.jboss.messaging.core.contract.MessageStore;
/*     */ import org.jboss.messaging.core.contract.PersistenceManager;
/*     */ import org.jboss.messaging.core.contract.PersistenceManager.InitialLoadInfo;
/*     */ import org.jboss.messaging.core.contract.PersistenceManager.ReferenceInfo;
/*     */ import org.jboss.messaging.util.prioritylinkedlist.PriorityLinkedList;
/*     */ 
/*     */ public abstract class PagingChannelSupport extends ChannelSupport
/*     */ {
/*  52 */   private static final Logger log = Logger.getLogger(PagingChannelSupport.class);
/*     */ 
/*  54 */   private boolean trace = log.isTraceEnabled();
/*     */   protected List downCache;
/*  61 */   protected int fullSize = 200000;
/*     */ 
/*  66 */   protected int pageSize = 2000;
/*     */ 
/*  71 */   protected int downCacheSize = 2000;
/*     */   protected boolean paging;
/*     */   protected long firstPagingOrder;
/*     */   protected long nextPagingOrder;
/*     */   protected MessageStore ms;
/*     */ 
/*     */   public PagingChannelSupport(long channelID, MessageStore ms, PersistenceManager pm, boolean recoverable, int maxSize)
/*     */   {
/*  96 */     super(channelID, pm, recoverable, maxSize);
/*     */ 
/*  98 */     this.downCache = new ArrayList(this.downCacheSize);
/*     */ 
/* 100 */     this.ms = ms;
/*     */   }
/*     */ 
/*     */   public PagingChannelSupport(long channelID, MessageStore ms, PersistenceManager pm, boolean recoverable, int maxSize, int fullSize, int pageSize, int downCacheSize)
/*     */   {
/* 110 */     super(channelID, pm, recoverable, maxSize);
/*     */ 
/* 112 */     if (pageSize >= fullSize)
/*     */     {
/* 114 */       throw new IllegalArgumentException("pageSize must be less than full size " + pageSize + ", " + fullSize);
/*     */     }
/* 116 */     if (downCacheSize > pageSize)
/*     */     {
/* 118 */       throw new IllegalArgumentException("pageSize cannot be smaller than downCacheSize");
/*     */     }
/* 120 */     if (pageSize <= 0)
/*     */     {
/* 122 */       throw new IllegalArgumentException("pageSize must be greater than zero");
/*     */     }
/* 124 */     if (downCacheSize <= 0)
/*     */     {
/* 126 */       throw new IllegalArgumentException("downCacheSize must be greater than zero");
/*     */     }
/*     */ 
/* 129 */     this.downCache = new ArrayList(downCacheSize);
/*     */ 
/* 131 */     this.fullSize = fullSize;
/*     */ 
/* 133 */     this.pageSize = pageSize;
/*     */ 
/* 135 */     this.downCacheSize = downCacheSize;
/*     */ 
/* 137 */     this.ms = ms;
/*     */   }
/*     */ 
/*     */   public int getMessageCount()
/*     */   {
/* 146 */     int count = super.getMessageCount();
/*     */ 
/* 150 */     synchronized (this.lock)
/*     */     {
/* 152 */       count = (int)(count + (this.nextPagingOrder - this.firstPagingOrder));
/*     */     }
/*     */ 
/* 155 */     return count;
/*     */   }
/*     */ 
/*     */   public int downCacheCount()
/*     */   {
/* 163 */     synchronized (this.lock)
/*     */     {
/* 165 */       return this.downCache.size();
/*     */     }
/*     */   }
/*     */ 
/*     */   public boolean isPaging()
/*     */   {
/* 172 */     synchronized (this.lock)
/*     */     {
/* 174 */       return this.paging;
/*     */     }
/*     */   }
/*     */ 
/*     */   public void setPagingParams(int fullSize, int pageSize, int downCacheSize)
/*     */   {
/* 180 */     synchronized (this.lock)
/*     */     {
/* 182 */       if (this.active)
/*     */       {
/* 184 */         throw new IllegalStateException("Cannot set paging params when active");
/*     */       }
/*     */ 
/* 187 */       this.fullSize = fullSize;
/*     */ 
/* 189 */       this.pageSize = pageSize;
/*     */ 
/* 191 */       this.downCacheSize = downCacheSize;
/*     */     }
/*     */   }
/*     */ 
/*     */   public void load() throws Exception
/*     */   {
/* 197 */     synchronized (this.lock)
/*     */     {
/* 199 */       if (this.active)
/*     */       {
/* 201 */         throw new IllegalStateException("Cannot load channel when active");
/*     */       }
/*     */ 
/* 204 */       if (this.trace) log.trace(this + " loading channel state"); 
/*     */ unload();
/*     */ 
/* 209 */       PersistenceManager.InitialLoadInfo ili = this.pm.loadFromStart(this.channelID, this.fullSize);
/*     */ 
/* 211 */       doLoad(ili);
/*     */ 
/* 215 */       while (checkLoad());
/*     */     }
/*     */   }
/*     */ 
/*     */   public void unload() throws Exception
/*     */   {
/* 221 */     synchronized (this.lock)
/*     */     {
/* 223 */       if (this.active)
/*     */       {
/* 225 */         throw new IllegalStateException("Cannot unload channel when active");
/*     */       }
/*     */ 
/* 228 */       this.messageRefs.clear();
/*     */ 
/* 230 */       this.downCache.clear();
/*     */ 
/* 232 */       this.paging = false;
/*     */ 
/* 234 */       this.firstPagingOrder = (this.nextPagingOrder = 0L);
/*     */ 
/* 236 */       clearAllScheduledDeliveries();
/*     */     }
/*     */   }
/*     */ 
/*     */   protected void loadPagedReferences(int number)
/*     */     throws Exception
/*     */   {
/* 245 */     if (this.trace) log.trace(this + " Loading " + number + " paged references from storage");
/*     */ 
/* 248 */     flushDownCache();
/*     */ 
/* 250 */     List refInfos = this.pm.getPagedReferenceInfos(this.channelID, this.firstPagingOrder, number);
/*     */ 
/* 252 */     Map refMap = processReferences(refInfos);
/*     */ 
/* 254 */     boolean loadedReliable = false;
/*     */ 
/* 256 */     List toRemove = new ArrayList();
/*     */ 
/* 258 */     int unreliableNumber = 0;
/*     */ 
/* 260 */     Iterator iter = refInfos.iterator();
/* 261 */     while (iter.hasNext())
/*     */     {
/* 263 */       PersistenceManager.ReferenceInfo info = (PersistenceManager.ReferenceInfo)iter.next();
/*     */ 
/* 265 */       MessageReference ref = addFromRefInfo(info, refMap);
/*     */ 
/* 267 */       if ((this.recoverable) && (ref.getMessage().isReliable()))
/*     */       {
/* 269 */         loadedReliable = true;
/*     */       }
/*     */       else
/*     */       {
/* 275 */         toRemove.add(ref);
/*     */ 
/* 277 */         unreliableNumber++;
/*     */       }
/*     */     }
/*     */ 
/* 281 */     if (!toRemove.isEmpty())
/*     */     {
/* 284 */       if (this.trace) log.trace(this + " removing depaged refs " + toRemove.size());
/*     */ 
/* 286 */       this.pm.removeDepagedReferences(this.channelID, toRemove);
/*     */     }
/*     */ 
/* 289 */     if (loadedReliable)
/*     */     {
/* 295 */       if (this.trace) log.trace("Updating refs not paged");
/*     */ 
/* 297 */       this.pm.updateReferencesNotPagedInRange(this.channelID, this.firstPagingOrder, this.firstPagingOrder + number - 1L, number - unreliableNumber);
/*     */     }
/*     */ 
/* 300 */     this.firstPagingOrder += number;
/*     */ 
/* 302 */     if (this.trace) log.trace(this + " set firstPagingOrder to " + this.firstPagingOrder);
/*     */ 
/* 304 */     if (this.firstPagingOrder == this.nextPagingOrder)
/*     */     {
/* 307 */       this.firstPagingOrder = (this.nextPagingOrder = 0L);
/*     */ 
/* 309 */       if (this.messageRefs.size() != this.fullSize)
/*     */       {
/* 311 */         this.paging = false;
/*     */       }
/*     */     }
/*     */   }
/*     */ 
/*     */   protected void cancelInternal(MessageReference ref) throws Exception
/*     */   {
/* 318 */     synchronized (this.lock)
/*     */     {
/* 320 */       super.cancelInternal(ref);
/*     */ 
/* 322 */       if (this.paging)
/*     */       {
/* 326 */         if (this.messageRefs.size() == this.fullSize + 1)
/*     */         {
/* 328 */           MessageReference refCancel = (MessageReference)this.messageRefs.removeLast();
/*     */ 
/* 330 */           addToDownCache(refCancel, true);
/*     */         }
/*     */       }
/*     */ 
/* 334 */       if (this.trace) log.trace(this + " added " + ref + " back into state"); 
/*     */     }
/*     */   }
/*     */ 
/*     */   protected MessageReference removeFirstInMemory()
/*     */     throws Exception
/*     */   {
/* 340 */     MessageReference result = super.removeFirstInMemory();
/*     */ 
/* 342 */     checkLoad();
/*     */ 
/* 344 */     return result;
/*     */   }
/*     */ 
/*     */   private boolean checkLoad() throws Exception
/*     */   {
/* 349 */     long refNum = this.nextPagingOrder - this.firstPagingOrder;
/*     */ 
/* 351 */     if (refNum > 0L)
/*     */     {
/* 353 */       int numberLoadable = (int)Math.min(refNum, this.pageSize);
/*     */ 
/* 355 */       if (this.messageRefs.size() <= this.fullSize - numberLoadable)
/*     */       {
/* 358 */         loadPagedReferences(numberLoadable);
/*     */ 
/* 360 */         return true;
/*     */       }
/*     */ 
/* 364 */       return false;
/*     */     }
/*     */ 
/* 369 */     this.paging = false;
/*     */ 
/* 371 */     return false;
/*     */   }
/*     */ 
/*     */   protected void addReferenceInMemory(MessageReference ref)
/*     */     throws Exception
/*     */   {
/* 377 */     if (this.paging)
/*     */     {
/* 379 */       addToDownCache(ref, false);
/*     */     }
/*     */     else
/*     */     {
/* 383 */       super.addReferenceInMemory(ref);
/*     */ 
/* 385 */       if (this.messageRefs.size() == this.fullSize)
/*     */       {
/* 388 */         if (this.trace) log.trace(this + " going into paging mode");
/*     */ 
/* 390 */         this.paging = true;
/*     */       }
/*     */     }
/*     */   }
/*     */ 
/*     */   protected void addToDownCache(MessageReference ref, boolean cancelling)
/*     */     throws Exception
/*     */   {
/* 411 */     if (cancelling)
/*     */     {
/* 413 */       ref.setPagingOrder(this.firstPagingOrder - 1L);
/*     */ 
/* 415 */       this.firstPagingOrder -= 1L;
/*     */     }
/*     */     else
/*     */     {
/* 419 */       ref.setPagingOrder(this.nextPagingOrder);
/*     */ 
/* 421 */       this.nextPagingOrder += 1L;
/*     */     }
/*     */ 
/* 424 */     this.downCache.add(ref);
/*     */ 
/* 426 */     if (this.trace) log.trace(ref + " sent to downcache");
/*     */ 
/* 428 */     if (this.downCache.size() == this.downCacheSize)
/*     */     {
/* 430 */       if (this.trace) log.trace(this + "'s downcache is full (" + this.downCache.size() + " messages)");
/*     */ 
/* 432 */       flushDownCache();
/*     */     }
/*     */   }
/*     */ 
/*     */   protected void flushDownCache() throws Exception
/*     */   {
/* 438 */     if (this.trace) log.trace(this + " flushing " + this.downCache.size() + " refs from downcache");
/*     */ 
/* 443 */     List toUpdate = new ArrayList();
/*     */ 
/* 445 */     List toAdd = new ArrayList();
/*     */ 
/* 447 */     Iterator iter = this.downCache.iterator();
/*     */ 
/* 449 */     while (iter.hasNext())
/*     */     {
/* 451 */       MessageReference ref = (MessageReference)iter.next();
/*     */ 
/* 453 */       if ((ref.getMessage().isReliable()) && (this.recoverable))
/*     */       {
/* 455 */         toUpdate.add(ref);
/*     */       }
/*     */       else
/*     */       {
/* 459 */         toAdd.add(ref);
/*     */       }
/*     */     }
/*     */ 
/* 463 */     if (!toAdd.isEmpty())
/*     */     {
/* 465 */       this.pm.pageReferences(this.channelID, toAdd, true);
/*     */     }
/*     */ 
/* 468 */     if (!toUpdate.isEmpty())
/*     */     {
/* 470 */       this.pm.updatePageOrder(this.channelID, toUpdate);
/*     */     }
/*     */ 
/* 473 */     this.downCache.clear();
/*     */ 
/* 475 */     if (this.trace) log.trace(this + " cleared downcache");
/*     */   }
/*     */ 
/*     */   protected void doLoad(PersistenceManager.InitialLoadInfo ili)
/*     */     throws Exception
/*     */   {
/* 482 */     if (ili.getMaxPageOrdering() != null)
/*     */     {
/* 484 */       this.firstPagingOrder = ili.getMinPageOrdering().longValue();
/*     */ 
/* 486 */       this.nextPagingOrder = (ili.getMaxPageOrdering().longValue() + 1L);
/*     */ 
/* 488 */       this.paging = true;
/*     */     }
/*     */     else
/*     */     {
/* 492 */       this.firstPagingOrder = (this.nextPagingOrder = 0L);
/*     */ 
/* 494 */       this.paging = false;
/*     */     }
/*     */ 
/* 497 */     Map refMap = processReferences(ili.getRefInfos());
/*     */ 
/* 499 */     Iterator iter = ili.getRefInfos().iterator();
/* 500 */     while (iter.hasNext())
/*     */     {
/* 502 */       PersistenceManager.ReferenceInfo info = (PersistenceManager.ReferenceInfo)iter.next();
/*     */ 
/* 504 */       addFromRefInfo(info, refMap);
/*     */     }
/*     */   }
/*     */ 
/*     */   protected MessageReference addFromRefInfo(PersistenceManager.ReferenceInfo info, Map refMap)
/*     */   {
/* 512 */     long msgId = info.getMessageId();
/*     */ 
/* 514 */     MessageReference ref = (MessageReference)refMap.get(new Long(msgId));
/*     */ 
/* 516 */     ref.setDeliveryCount(info.getDeliveryCount());
/*     */ 
/* 518 */     ref.setPagingOrder(-1L);
/*     */ 
/* 520 */     ref.setScheduledDeliveryTime(info.getScheduledDelivery());
/*     */ 
/* 523 */     if (!checkAndSchedule(ref))
/*     */     {
/* 525 */       this.messageRefs.addLast(ref, ref.getMessage().getPriority());
/*     */     }
/*     */ 
/* 528 */     return ref;
/*     */   }
/*     */ 
/*     */   protected Map processReferences(List refInfos)
/*     */     throws Exception
/*     */   {
/* 534 */     Map refMap = new HashMap(refInfos.size());
/*     */ 
/* 536 */     List msgIdsToLoad = new ArrayList(refInfos.size());
/*     */ 
/* 538 */     Iterator iter = refInfos.iterator();
/*     */ 
/* 541 */     while (iter.hasNext())
/*     */     {
/* 543 */       PersistenceManager.ReferenceInfo info = (PersistenceManager.ReferenceInfo)iter.next();
/*     */ 
/* 545 */       long msgId = info.getMessageId();
/*     */ 
/* 547 */       MessageReference ref = this.ms.reference(msgId);
/*     */ 
/* 549 */       if (ref != null)
/*     */       {
/* 551 */         refMap.put(Long.valueOf(msgId), ref);
/*     */       }
/*     */       else
/*     */       {
/* 556 */         msgIdsToLoad.add(Long.valueOf(msgId));
/*     */       }
/*     */ 
/*     */     }
/*     */ 
/* 561 */     List messages = null;
/* 562 */     if (!msgIdsToLoad.isEmpty())
/*     */     {
/* 564 */       messages = this.pm.getMessages(msgIdsToLoad);
/*     */ 
/* 566 */       if (messages.size() != msgIdsToLoad.size())
/*     */       {
/* 570 */         throw new IllegalStateException("Did not load correct number of messages, wanted:" + msgIdsToLoad.size() + " but got:" + messages.size());
/*     */       }
/*     */ 
/* 576 */       iter = messages.iterator();
/*     */ 
/* 578 */       while (iter.hasNext())
/*     */       {
/* 580 */         Message m = (Message)iter.next();
/*     */ 
/* 582 */         MessageReference ref = this.ms.reference(m);
/*     */ 
/* 584 */         refMap.put(new Long(m.getMessageID()), ref);
/*     */       }
/*     */     }
/*     */ 
/* 588 */     return refMap;
/*     */   }
/*     */ }

/* Location:           /home/mnovotny/projects/EMBEDDED_JBOSS_BETA3_COMMUNITY/embedded/output/lib/embedded-jboss/lib/jboss-embedded-all.jar
 * Qualified Name:     org.jboss.messaging.core.impl.PagingChannelSupport
 * JD-Core Version:    0.6.0
 */