/*     */ package org.jboss.ejb3.cache.impl;
/*     */ 
/*     */ import java.io.Serializable;
/*     */ import java.util.Collection;
/*     */ import java.util.HashMap;
/*     */ import java.util.Iterator;
/*     */ import java.util.Map;
/*     */ import javax.ejb.NoSuchEJBException;
/*     */ import org.jboss.ejb3.cache.Identifiable;
/*     */ import org.jboss.ejb3.cache.ObjectStore;
/*     */ import org.jboss.ejb3.cache.PassivatingCache;
/*     */ import org.jboss.ejb3.cache.PassivationManager;
/*     */ import org.jboss.ejb3.cache.StatefulObjectFactory;
/*     */ import org.jboss.logging.Logger;
/*     */ 
/*     */ public class SimplePassivatingCache<T extends Identifiable,  extends Serializable>
/*     */   implements PassivatingCache<T>
/*     */ {
/*     */   private static final Logger log;
/*     */   private StatefulObjectFactory<T> factory;
/*     */   private PassivationManager<T> passivationManager;
/*     */   private ObjectStore<T> store;
/*     */   private Map<Object, SimplePassivatingCache<T>.Entry> cache;
/*  54 */   private int sessionTimeout = -1;
/*     */   private String name;
/*     */   private Thread sessionTimeoutTask;
/*     */ 
/*     */   public SimplePassivatingCache(StatefulObjectFactory<T> factory, PassivationManager<T> passivationManager, ObjectStore<T> store)
/*     */   {
/* 133 */     assert (factory != null) : "factory is null";
/* 134 */     assert (passivationManager != null) : "passivationManager is null";
/* 135 */     assert (store != null) : "store is null";
/*     */ 
/* 137 */     this.factory = factory;
/* 138 */     this.passivationManager = passivationManager;
/* 139 */     this.store = store;
/* 140 */     this.cache = new HashMap();
/*     */   }
/*     */ 
/*     */   protected SimplePassivatingCache<T>.Entry activate(Object key)
/*     */   {
/* 153 */     Identifiable obj = this.store.load(key);
/* 154 */     if (obj == null) {
/* 155 */       return null;
/*     */     }
/* 157 */     this.passivationManager.postActivate((Serializable)obj);
/*     */ 
/* 159 */     Entry entry = new Entry(obj);
/* 160 */     this.cache.put(key, entry);
/* 161 */     return entry;
/*     */   }
/*     */ 
/*     */   public T create(Class<?>[] initTypes, Object[] initValues)
/*     */   {
/* 166 */     Identifiable obj = (Identifiable)this.factory.create(initTypes, initValues);
/* 167 */     Entry entry = new Entry(obj);
/* 168 */     synchronized (this.cache)
/*     */     {
/* 170 */       this.cache.put(obj.getId(), entry);
/*     */     }
/* 172 */     return obj;
/*     */   }
/*     */ 
/*     */   public T get(Object key) throws NoSuchEJBException
/*     */   {
/* 177 */     synchronized (this.cache)
/*     */     {
/* 179 */       Entry entry = (Entry)this.cache.get(key);
/* 180 */       if (entry == null)
/*     */       {
/* 182 */         entry = activate(key);
/* 183 */         if (entry != null)
/* 184 */           return entry.obj;
/*     */       }
/* 186 */       if (entry == null)
/* 187 */         throw new NoSuchEJBException(String.valueOf(key));
/* 188 */       if (entry.state != EntryState.READY)
/* 189 */         throw new IllegalStateException("entry " + key + " is not ready");
/* 190 */       entry.state = EntryState.IN_USE;
/* 191 */       entry.lastUsed = System.currentTimeMillis();
/* 192 */       return entry.obj;
/*     */     }
/*     */   }
/*     */ 
/*     */   public void passivate(Object key)
/*     */   {
/* 198 */     log.trace("passivate " + key);
/* 199 */     synchronized (this.cache)
/*     */     {
/* 201 */       Entry entry = (Entry)this.cache.get(key);
/*     */ 
/* 203 */       if (entry == null) {
/* 204 */         throw new IllegalArgumentException("entry " + key + " not found in cache " + this);
/*     */       }
/* 206 */       if (entry.state == EntryState.IN_USE) {
/* 207 */         throw new IllegalStateException("entry " + entry + " is in use");
/*     */       }
/* 209 */       this.passivationManager.prePassivate((Serializable)entry.obj);
/*     */ 
/* 211 */       this.store.store(entry.obj);
/*     */ 
/* 213 */       this.cache.remove(key);
/*     */     }
/*     */   }
/*     */ 
/*     */   public T peek(Object key) throws NoSuchEJBException
/*     */   {
/* 219 */     synchronized (this.cache)
/*     */     {
/* 221 */       Entry entry = (Entry)this.cache.get(key);
/* 222 */       if (entry == null)
/*     */       {
/* 224 */         entry = activate(key);
/*     */ 
/* 227 */         if (entry != null)
/* 228 */           entry.state = EntryState.READY;
/*     */       }
/* 230 */       if (entry == null)
/* 231 */         throw new NoSuchEJBException(String.valueOf(key));
/* 232 */       return entry.obj;
/*     */     }
/*     */   }
/*     */ 
/*     */   public void release(T obj)
/*     */   {
/* 238 */     releaseByKey(obj.getId());
/*     */   }
/*     */ 
/*     */   protected void releaseByKey(Object key)
/*     */   {
/* 243 */     synchronized (this.cache)
/*     */     {
/* 245 */       Entry entry = (Entry)this.cache.get(key);
/* 246 */       if (entry == null)
/* 247 */         throw new IllegalStateException("object " + key + " not from this cache");
/* 248 */       if (entry.state != EntryState.IN_USE)
/* 249 */         throw new IllegalStateException("entry " + entry + " is not in use");
/* 250 */       entry.state = EntryState.READY;
/* 251 */       entry.lastUsed = System.currentTimeMillis();
/*     */     }
/*     */   }
/*     */ 
/*     */   public void remove(Object key)
/*     */   {
/*     */     Entry entry;
/* 258 */     synchronized (this.cache)
/*     */     {
/* 260 */       entry = (Entry)this.cache.remove(key);
/* 261 */       if (entry == null)
/*     */       {
/* 263 */         entry = activate(key);
/* 264 */         if (entry == null) {
/* 265 */           throw new NoSuchEJBException(String.valueOf(key));
/*     */         }
/* 267 */         entry.state = EntryState.READY;
/*     */       }
/* 269 */       if (entry.state != EntryState.READY)
/* 270 */         throw new IllegalStateException("entry " + entry + " is not ready");
/*     */     }
/* 272 */     if (entry != null)
/* 273 */       this.factory.destroy(entry.obj);
/*     */   }
/*     */ 
/*     */   public void setName(String name)
/*     */   {
/* 278 */     this.name = name;
/*     */   }
/*     */ 
/*     */   public void setSessionTimeout(int sessionTimeout)
/*     */   {
/* 283 */     assert (sessionTimeout >= 0) : "sessionTimeout must be >= 0";
/* 284 */     this.sessionTimeout = sessionTimeout;
/*     */   }
/*     */ 
/*     */   public void start()
/*     */   {
/* 289 */     assert (this.name != null) : "name has not been set";
/* 290 */     assert (this.sessionTimeout != -1) : "sessionTimeout has not been set";
/*     */ 
/* 292 */     if (this.sessionTimeout > 0)
/*     */     {
/* 294 */       this.sessionTimeoutTask = new SessionTimeoutThread("Passivation Thread - " + this.name);
/* 295 */       this.sessionTimeoutTask.start();
/*     */     }
/*     */   }
/*     */ 
/*     */   public void stop()
/*     */   {
/* 301 */     if (this.sessionTimeoutTask != null)
/*     */     {
/* 303 */       this.sessionTimeoutTask.interrupt();
/*     */       try
/*     */       {
/* 306 */         this.sessionTimeoutTask.join(5000L);
/*     */       }
/*     */       catch (InterruptedException e)
/*     */       {
/*     */       }
/*     */ 
/* 312 */       if (this.sessionTimeoutTask.isAlive())
/* 313 */         log.warn("Failed to stop " + this.sessionTimeoutTask);
/*     */     }
/*     */   }
/*     */ 
/*     */   static
/*     */   {
/*  46 */     log = Logger.getLogger(SimplePassivatingCache.class);
/*     */   }
/*     */ 
/*     */   private class SessionTimeoutThread extends Thread
/*     */   {
/*     */     public SessionTimeoutThread(String name)
/*     */     {
/*  81 */       super();
/*  82 */       setDaemon(true);
/*     */     }
/*     */ 
/*     */     public void run()
/*     */     {
/*     */       try
/*     */       {
/*  90 */         while (!Thread.currentThread().isInterrupted())
/*     */         {
/*  92 */           Thread.sleep(1000L);
/*     */ 
/*  94 */           synchronized (SimplePassivatingCache.this.cache)
/*     */           {
/*  96 */             if (Thread.currentThread().isInterrupted()) {
/*  97 */               return;
/*     */             }
/*  99 */             long then = System.currentTimeMillis() - SimplePassivatingCache.this.sessionTimeout * 1000;
/* 100 */             Iterator it = SimplePassivatingCache.this.cache.values().iterator();
/* 101 */             while (it.hasNext())
/*     */             {
/* 103 */               SimplePassivatingCache.Entry entry = (SimplePassivatingCache.Entry)it.next();
/* 104 */               if ((then >= entry.lastUsed) && (entry.state != SimplePassivatingCache.EntryState.IN_USE))
/*     */               {
/*     */                 try
/*     */                 {
/* 109 */                   SimplePassivatingCache.this.passivationManager.prePassivate((Serializable)entry.obj);
/*     */                 }
/*     */                 catch (Throwable t)
/*     */                 {
/* 113 */                   SimplePassivatingCache.log.warn("pre passivate failed for " + entry.obj, t);
/*     */                 }
/*     */ 
/* 116 */                 SimplePassivatingCache.this.store.store(entry.obj);
/*     */ 
/* 118 */                 it.remove();
/*     */               }
/*     */             }
/*     */           }
/*     */         }
/*     */       }
/*     */       catch (InterruptedException e)
/*     */       {
/*     */       }
/*     */     }
/*     */   }
/*     */ 
/*     */   private class Entry
/*     */   {
/*     */     long lastUsed;
/*     */     T obj;
/*     */     SimplePassivatingCache.EntryState state;
/*     */ 
/*     */     Entry()
/*     */     {
/*  69 */       assert (obj != null) : "obj is null";
/*     */ 
/*  71 */       this.lastUsed = System.currentTimeMillis();
/*  72 */       this.obj = obj;
/*  73 */       this.state = SimplePassivatingCache.EntryState.IN_USE;
/*     */     }
/*     */   }
/*     */ 
/*     */   private static enum EntryState
/*     */   {
/*  59 */     READY, IN_USE;
/*     */   }
/*     */ }

/* Location:           /home/mnovotny/projects/EMBEDDED_JBOSS_BETA3_COMMUNITY/embedded/output/lib/embedded-jboss/lib/jboss-embedded-all.jar
 * Qualified Name:     org.jboss.ejb3.cache.impl.SimplePassivatingCache
 * JD-Core Version:    0.6.0
 */