/*     */ package org.jboss.crypto;
/*     */ 
/*     */ import java.io.Serializable;
/*     */ import java.io.UnsupportedEncodingException;
/*     */ import java.lang.reflect.Constructor;
/*     */ import java.lang.reflect.Method;
/*     */ import java.math.BigInteger;
/*     */ import java.security.GeneralSecurityException;
/*     */ import java.security.KeyException;
/*     */ import java.security.MessageDigest;
/*     */ import java.security.NoSuchAlgorithmException;
/*     */ import java.security.Provider;
/*     */ import java.security.SecureRandom;
/*     */ import java.security.Security;
/*     */ import java.util.Random;
/*     */ import javax.crypto.Cipher;
/*     */ import javax.crypto.SealedObject;
/*     */ import javax.crypto.SecretKey;
/*     */ import javax.crypto.spec.IvParameterSpec;
/*     */ import org.jboss.crypto.digest.DigestCallback;
/*     */ import org.jboss.logging.Logger;
/*     */ import org.jboss.security.Base64Encoder;
/*     */ import org.jboss.security.Base64Utils;
/*     */ 
/*     */ public class CryptoUtil
/*     */ {
/*  65 */   private static Logger log = Logger.getLogger(CryptoUtil.class);
/*     */   private static final int HASH_LEN = 20;
/*     */   public static final String BASE64_ENCODING = "BASE64";
/*     */   public static final String BASE16_ENCODING = "HEX";
/*     */   public static final String RFC2617_ENCODING = "RFC2617";
/*  73 */   private static char[] MD5_HEX = "0123456789abcdef".toCharArray();
/*     */   private static SecureRandom psuedoRng;
/*     */   private static MessageDigest sha1Digest;
/*     */   private static boolean initialized;
/*     */ 
/*     */   public static void init()
/*     */     throws NoSuchAlgorithmException
/*     */   {
/*  81 */     if (initialized)
/*  82 */       return;
/*  83 */     init(null);
/*     */   }
/*     */ 
/*     */   public static void init(byte[] prngSeed) throws NoSuchAlgorithmException
/*     */   {
/*  88 */     sha1Digest = MessageDigest.getInstance("SHA");
/*     */ 
/*  90 */     psuedoRng = SecureRandom.getInstance("SHA1PRNG");
/*  91 */     if (prngSeed != null) {
/*  92 */       psuedoRng.setSeed(prngSeed);
/*     */     }
/*  94 */     Provider provider = new JBossSXProvider();
/*  95 */     Security.addProvider(provider);
/*  96 */     initialized = true;
/*     */   }
/*     */ 
/*     */   public static MessageDigest newDigest()
/*     */   {
/* 101 */     MessageDigest md = null;
/*     */     try
/*     */     {
/* 104 */       md = (MessageDigest)sha1Digest.clone();
/*     */     }
/*     */     catch (CloneNotSupportedException e)
/*     */     {
/*     */     }
/* 109 */     return md;
/*     */   }
/*     */ 
/*     */   public static MessageDigest copy(MessageDigest md) {
/* 113 */     MessageDigest copy = null;
/*     */     try
/*     */     {
/* 116 */       copy = (MessageDigest)md.clone();
/*     */     }
/*     */     catch (CloneNotSupportedException e)
/*     */     {
/*     */     }
/* 121 */     return copy;
/*     */   }
/*     */ 
/*     */   public static Random getPRNG()
/*     */   {
/* 126 */     return psuedoRng;
/*     */   }
/*     */ 
/*     */   public static double nextDouble()
/*     */   {
/* 133 */     return psuedoRng.nextDouble();
/*     */   }
/*     */ 
/*     */   public static long nextLong()
/*     */   {
/* 143 */     return psuedoRng.nextLong();
/*     */   }
/*     */ 
/*     */   public static void nextBytes(byte[] bytes)
/*     */   {
/* 151 */     psuedoRng.nextBytes(bytes);
/*     */   }
/*     */ 
/*     */   public static byte[] generateSeed(int numBytes)
/*     */   {
/* 159 */     return psuedoRng.generateSeed(numBytes);
/*     */   }
/*     */ 
/*     */   public static byte[] calculatePasswordHash(String username, char[] password, byte[] salt)
/*     */   {
/* 170 */     MessageDigest xd = newDigest();
/*     */ 
/* 172 */     byte[] user = null;
/* 173 */     byte[] colon = new byte[0];
/*     */     try
/*     */     {
/* 176 */       user = username.getBytes("UTF-8");
/* 177 */       colon = ":".getBytes("UTF-8");
/*     */     }
/*     */     catch (UnsupportedEncodingException e)
/*     */     {
/* 181 */       log.error("Failed to convert username to byte[] using UTF-8", e);
/*     */ 
/* 183 */       user = username.getBytes();
/* 184 */       colon = ":".getBytes();
/*     */     }
/* 186 */     byte[] passBytes = new byte[2 * password.length];
/* 187 */     int passBytesLength = 0;
/* 188 */     for (int p = 0; p < password.length; p++)
/*     */     {
/* 190 */       int c = password[p] & 0xFFFF;
/*     */ 
/* 192 */       byte b0 = (byte)(c & 0xFF);
/*     */ 
/* 194 */       byte b1 = (byte)((c & 0xFF00) >> 8);
/* 195 */       passBytes[(passBytesLength++)] = b0;
/*     */ 
/* 197 */       if (c > 255) {
/* 198 */         passBytes[(passBytesLength++)] = b1;
/*     */       }
/*     */     }
/*     */ 
/* 202 */     xd.update(user);
/* 203 */     xd.update(colon);
/* 204 */     xd.update(passBytes, 0, passBytesLength);
/* 205 */     byte[] h = xd.digest();
/* 206 */     xd.reset();
/* 207 */     xd.update(salt);
/* 208 */     xd.update(h);
/* 209 */     byte[] xb = xd.digest();
/* 210 */     return xb;
/*     */   }
/*     */ 
/*     */   public static byte[] calculateVerifier(String username, char[] password, byte[] salt, byte[] Nb, byte[] gb)
/*     */   {
/* 220 */     BigInteger g = new BigInteger(1, gb);
/* 221 */     BigInteger N = new BigInteger(1, Nb);
/* 222 */     return calculateVerifier(username, password, salt, N, g);
/*     */   }
/*     */ 
/*     */   public static byte[] calculateVerifier(String username, char[] password, byte[] salt, BigInteger N, BigInteger g)
/*     */   {
/* 231 */     byte[] xb = calculatePasswordHash(username, password, salt);
/* 232 */     BigInteger x = new BigInteger(1, xb);
/* 233 */     BigInteger v = g.modPow(x, N);
/* 234 */     return v.toByteArray();
/*     */   }
/*     */ 
/*     */   public static byte[] sessionKeyHash(byte[] number)
/*     */   {
/* 243 */     for (int offset = 0; (offset < number.length) && (number[offset] == 0); offset++);
/* 246 */     byte[] key = new byte[40];
/*     */ 
/* 249 */     int klen = (number.length - offset) / 2;
/* 250 */     byte[] hbuf = new byte[klen];
/*     */ 
/* 252 */     for (int i = 0; i < klen; i++)
/*     */     {
/* 254 */       hbuf[i] = number[(number.length - 2 * i - 1)];
/*     */     }
/* 256 */     byte[] hout = newDigest().digest(hbuf);
/* 257 */     for (i = 0; i < 20; i++) {
/* 258 */       key[(2 * i)] = hout[i];
/*     */     }
/* 260 */     for (i = 0; i < klen; i++)
/*     */     {
/* 262 */       hbuf[i] = number[(number.length - 2 * i - 2)];
/*     */     }
/* 264 */     hout = newDigest().digest(hbuf);
/* 265 */     for (i = 0; i < 20; i++) {
/* 266 */       key[(2 * i + 1)] = hout[i];
/*     */     }
/* 268 */     return key;
/*     */   }
/*     */ 
/*     */   public static byte[] trim(byte[] in)
/*     */   {
/* 277 */     if ((in.length == 0) || (in[0] != 0)) {
/* 278 */       return in;
/*     */     }
/* 280 */     int len = in.length;
/* 281 */     int i = 1;
/* 282 */     while ((in[i] == 0) && (i < len))
/* 283 */       i++;
/* 284 */     byte[] ret = new byte[len - i];
/* 285 */     System.arraycopy(in, i, ret, 0, len - i);
/* 286 */     return ret;
/*     */   }
/*     */ 
/*     */   public static byte[] xor(byte[] b1, byte[] b2, int length)
/*     */   {
/* 291 */     byte[] result = new byte[length];
/* 292 */     for (int i = 0; i < length; i++)
/* 293 */       result[i] = (byte)(b1[i] ^ b2[i]);
/* 294 */     return result;
/*     */   }
/*     */ 
/*     */   public static String encodeRFC2617(byte[] data)
/*     */   {
/* 317 */     char[] hash = new char[32];
/* 318 */     for (int i = 0; i < 16; i++)
/*     */     {
/* 320 */       int j = data[i] >> 4 & 0xF;
/* 321 */       hash[(i * 2)] = MD5_HEX[j];
/* 322 */       j = data[i] & 0xF;
/* 323 */       hash[(i * 2 + 1)] = MD5_HEX[j];
/*     */     }
/* 325 */     return new String(hash);
/*     */   }
/*     */ 
/*     */   public static String encodeBase16(byte[] bytes)
/*     */   {
/* 334 */     StringBuffer sb = new StringBuffer(bytes.length * 2);
/* 335 */     for (int i = 0; i < bytes.length; i++)
/*     */     {
/* 337 */       byte b = bytes[i];
/*     */ 
/* 339 */       char c = (char)(b >> 4 & 0xF);
/* 340 */       if (c > '\t')
/* 341 */         c = (char)(c - '\n' + 97);
/*     */       else
/* 343 */         c = (char)(c + '0');
/* 344 */       sb.append(c);
/*     */ 
/* 346 */       c = (char)(b & 0xF);
/* 347 */       if (c > '\t')
/* 348 */         c = (char)(c - '\n' + 97);
/*     */       else
/* 350 */         c = (char)(c + '0');
/* 351 */       sb.append(c);
/*     */     }
/* 353 */     return sb.toString();
/*     */   }
/*     */ 
/*     */   public static String encodeBase64(byte[] bytes)
/*     */   {
/* 363 */     String base64 = null;
/*     */     try
/*     */     {
/* 366 */       base64 = Base64Encoder.encode(bytes);
/*     */     }
/*     */     catch (Exception e)
/*     */     {
/*     */     }
/* 371 */     return base64;
/*     */   }
/*     */ 
/*     */   public static String createPasswordHash(String hashAlgorithm, String hashEncoding, String hashCharset, String username, String password)
/*     */   {
/* 390 */     return createPasswordHash(hashAlgorithm, hashEncoding, hashCharset, username, password, null);
/*     */   }
/*     */ 
/*     */   public static String createPasswordHash(String hashAlgorithm, String hashEncoding, String hashCharset, String username, String password, DigestCallback callback)
/*     */   {
/* 413 */     String passwordHash = null;
/*     */     byte[] passBytes;
/*     */     try
/*     */     {
/*     */       byte[] passBytes;
/* 418 */       if (hashCharset == null)
/* 419 */         passBytes = password.getBytes();
/*     */       else
/* 421 */         passBytes = password.getBytes(hashCharset);
/*     */     }
/*     */     catch (UnsupportedEncodingException uee)
/*     */     {
/* 425 */       log.error("charset " + hashCharset + " not found. Using platform default.", uee);
/* 426 */       passBytes = password.getBytes();
/*     */     }
/*     */ 
/*     */     try
/*     */     {
/* 432 */       MessageDigest md = MessageDigest.getInstance(hashAlgorithm);
/* 433 */       if (callback != null)
/* 434 */         callback.preDigest(md);
/* 435 */       md.update(passBytes);
/* 436 */       if (callback != null)
/* 437 */         callback.postDigest(md);
/* 438 */       byte[] hash = md.digest();
/* 439 */       if (hashEncoding.equalsIgnoreCase("BASE64"))
/*     */       {
/* 441 */         passwordHash = encodeBase64(hash);
/*     */       }
/* 443 */       else if (hashEncoding.equalsIgnoreCase("HEX"))
/*     */       {
/* 445 */         passwordHash = encodeBase16(hash);
/*     */       }
/* 447 */       else if (hashEncoding.equalsIgnoreCase("RFC2617"))
/*     */       {
/* 449 */         passwordHash = encodeRFC2617(hash);
/*     */       }
/*     */       else
/*     */       {
/* 453 */         log.error("Unsupported hash encoding format " + hashEncoding);
/*     */       }
/*     */     }
/*     */     catch (Exception e)
/*     */     {
/* 458 */       log.error("Password hash calculation failed ", e);
/*     */     }
/* 460 */     return passwordHash;
/*     */   }
/*     */ 
/*     */   public static String tob64(byte[] buffer)
/*     */   {
/* 468 */     return Base64Utils.tob64(buffer);
/*     */   }
/*     */ 
/*     */   public static byte[] fromb64(String str) throws NumberFormatException
/*     */   {
/* 473 */     return Base64Utils.fromb64(str);
/*     */   }
/*     */ 
/*     */   public static boolean hasUnlimitedCrypto()
/*     */   {
/* 484 */     boolean hasUnlimitedCrypto = false;
/*     */     try
/*     */     {
/* 487 */       ClassLoader loader = Thread.currentThread().getContextClassLoader();
/* 488 */       Class keyGenClass = loader.loadClass("javax.crypto.KeyGenerator");
/* 489 */       Class[] sig = { String.class };
/* 490 */       Object[] args = { "Blowfish" };
/* 491 */       Method kgenInstance = keyGenClass.getDeclaredMethod("getInstance", sig);
/* 492 */       Object kgen = kgenInstance.invoke(null, args);
/*     */ 
/* 494 */       Class[] sig2 = { Integer.TYPE };
/* 495 */       Object[] args2 = { new Integer(256) };
/* 496 */       Method init = keyGenClass.getDeclaredMethod("init", sig2);
/* 497 */       init.invoke(kgen, args2);
/* 498 */       hasUnlimitedCrypto = true;
/*     */     }
/*     */     catch (Throwable e)
/*     */     {
/* 502 */       log.debug("hasUnlimitedCrypto error", e);
/*     */     }
/* 504 */     return hasUnlimitedCrypto;
/*     */   }
/*     */ 
/*     */   public static Object createSecretKey(String cipherAlgorithm, Object key)
/*     */     throws KeyException
/*     */   {
/* 514 */     Class[] signature = { key.getClass(), String.class };
/* 515 */     Object[] args = { key, cipherAlgorithm };
/* 516 */     Object secretKey = null;
/*     */     try
/*     */     {
/* 519 */       ClassLoader loader = Thread.currentThread().getContextClassLoader();
/* 520 */       Class secretKeySpecClass = loader.loadClass("javax.crypto.spec.SecretKeySpec");
/* 521 */       Constructor ctor = secretKeySpecClass.getDeclaredConstructor(signature);
/* 522 */       secretKey = ctor.newInstance(args);
/*     */     }
/*     */     catch (Exception e)
/*     */     {
/* 526 */       throw new KeyException("Failed to create SecretKeySpec from session key, msg=" + e.getMessage());
/*     */     }
/*     */     catch (Throwable e)
/*     */     {
/* 530 */       throw new KeyException("Unexpected exception during SecretKeySpec creation, msg=" + e.getMessage());
/*     */     }
/* 532 */     return secretKey;
/*     */   }
/*     */ 
/*     */   public static Object createCipher(String cipherAlgorithm)
/*     */     throws GeneralSecurityException
/*     */   {
/* 543 */     Cipher cipher = Cipher.getInstance(cipherAlgorithm);
/* 544 */     return cipher;
/*     */   }
/*     */ 
/*     */   public static Object createSealedObject(String cipherAlgorithm, Object key, byte[] cipherIV, Serializable data)
/*     */     throws GeneralSecurityException
/*     */   {
/* 550 */     Object sealedObject = null;
/*     */     try
/*     */     {
/* 553 */       Cipher cipher = Cipher.getInstance(cipherAlgorithm);
/* 554 */       SecretKey skey = (SecretKey)key;
/* 555 */       if (cipherIV != null)
/*     */       {
/* 557 */         IvParameterSpec iv = new IvParameterSpec(cipherIV);
/* 558 */         cipher.init(1, skey, iv);
/*     */       }
/*     */       else
/*     */       {
/* 562 */         cipher.init(1, skey);
/*     */       }
/* 564 */       sealedObject = new SealedObject(data, cipher);
/*     */     }
/*     */     catch (GeneralSecurityException e)
/*     */     {
/* 568 */       throw e;
/*     */     }
/*     */     catch (Throwable e)
/*     */     {
/* 572 */       throw new GeneralSecurityException("Failed to create SealedObject, msg=" + e.getMessage());
/*     */     }
/* 574 */     return sealedObject;
/*     */   }
/*     */ 
/*     */   public static Object accessSealedObject(String cipherAlgorithm, Object key, byte[] cipherIV, Object obj)
/*     */     throws GeneralSecurityException
/*     */   {
/* 581 */     Object data = null;
/*     */     try
/*     */     {
/* 584 */       Cipher cipher = Cipher.getInstance(cipherAlgorithm);
/* 585 */       SecretKey skey = (SecretKey)key;
/* 586 */       if (cipherIV != null)
/*     */       {
/* 588 */         IvParameterSpec iv = new IvParameterSpec(cipherIV);
/* 589 */         cipher.init(2, skey, iv);
/*     */       }
/*     */       else
/*     */       {
/* 593 */         cipher.init(2, skey);
/*     */       }
/* 595 */       SealedObject sealedObj = (SealedObject)obj;
/* 596 */       data = sealedObj.getObject(cipher);
/*     */     }
/*     */     catch (GeneralSecurityException e)
/*     */     {
/* 600 */       throw e;
/*     */     }
/*     */     catch (Throwable e)
/*     */     {
/* 604 */       throw new GeneralSecurityException("Failed to access SealedObject, msg=" + e.getMessage());
/*     */     }
/* 606 */     return data;
/*     */   }
/*     */ }

/* Location:           /home/mnovotny/projects/EMBEDDED_JBOSS_BETA3_COMMUNITY/embedded/output/lib/embedded-jboss/lib/jboss-embedded-all.jar
 * Qualified Name:     org.jboss.crypto.CryptoUtil
 * JD-Core Version:    0.6.0
 */