/*     */ package org.jboss.classloader.spi.base;
/*     */ 
/*     */ import java.io.IOException;
/*     */ import java.net.URL;
/*     */ import java.security.ProtectionDomain;
/*     */ import java.util.Iterator;
/*     */ import java.util.List;
/*     */ import java.util.Map;
/*     */ import java.util.Set;
/*     */ import java.util.concurrent.ConcurrentHashMap;
/*     */ import java.util.concurrent.CopyOnWriteArrayList;
/*     */ import java.util.concurrent.CopyOnWriteArraySet;
/*     */ import org.jboss.classloader.plugins.ClassLoaderUtils;
/*     */ import org.jboss.classloader.spi.ClassLoaderPolicy;
/*     */ import org.jboss.classloader.spi.DelegateLoader;
/*     */ import org.jboss.classloader.spi.Loader;
/*     */ import org.jboss.logging.Logger;
/*     */ 
/*     */ public abstract class BaseClassLoaderDomain
/*     */   implements Loader
/*     */ {
/*  53 */   private static final Logger log = Logger.getLogger(BaseClassLoaderDomain.class);
/*     */   private BaseClassLoaderSystem system;
/*  59 */   private List<ClassLoaderInformation> classLoaders = new CopyOnWriteArrayList();
/*     */ 
/*  62 */   private Map<ClassLoader, ClassLoaderInformation> infos = new ConcurrentHashMap();
/*     */ 
/*  65 */   private Map<String, List<ClassLoaderInformation>> classLoadersByPackageName = new ConcurrentHashMap();
/*     */ 
/*  68 */   private Map<String, Loader> globalClassCache = new ConcurrentHashMap();
/*     */ 
/*  71 */   private Set<String> globalClassBlackList = new CopyOnWriteArraySet();
/*     */ 
/*  74 */   private Map<String, URL> globalResourceCache = new ConcurrentHashMap();
/*     */ 
/*  77 */   private Set<String> globalResourceBlackList = new CopyOnWriteArraySet();
/*     */ 
/*  80 */   private int order = 0;
/*     */ 
/*     */   public void flushCaches()
/*     */   {
/*  87 */     this.globalClassCache.clear();
/*  88 */     this.globalClassBlackList.clear();
/*  89 */     this.globalResourceCache.clear();
/*  90 */     this.globalResourceBlackList.clear();
/*     */   }
/*     */ 
/*     */   synchronized BaseClassLoaderSystem getClassLoaderSystem()
/*     */   {
/* 100 */     return this.system;
/*     */   }
/*     */ 
/*     */   synchronized void setClassLoaderSystem(BaseClassLoaderSystem system)
/*     */   {
/* 110 */     if (system == null)
/* 111 */       shutdownDomain();
/* 112 */     this.system = system;
/*     */   }
/*     */ 
/*     */   protected void shutdownDomain()
/*     */   {
/* 122 */     log.debug(toLongString() + " shutdown!");
/*     */     while (true)
/*     */     {
/* 127 */       Iterator iterator = this.classLoaders.iterator();
/* 128 */       if (!iterator.hasNext()) {
/*     */         break;
/*     */       }
/* 131 */       while (iterator.hasNext())
/*     */       {
/* 133 */         ClassLoaderInformation info = (ClassLoaderInformation)iterator.next();
/* 134 */         if (info != null) {
/* 135 */           unregisterClassLoader(info.getClassLoader());
/*     */         }
/*     */       }
/*     */     }
/* 139 */     flushCaches();
/*     */   }
/*     */ 
/*     */   public boolean hasClassLoaders()
/*     */   {
/* 149 */     return !this.classLoaders.isEmpty();
/*     */   }
/*     */ 
/*     */   protected byte[] transform(ClassLoader classLoader, String className, byte[] byteCode, ProtectionDomain protectionDomain)
/*     */     throws Exception
/*     */   {
/* 166 */     BaseClassLoaderSystem system = getClassLoaderSystem();
/* 167 */     if (system != null)
/* 168 */       return system.transform(classLoader, className, byteCode, protectionDomain);
/* 169 */     return byteCode;
/*     */   }
/*     */ 
/*     */   Class<?> loadClass(BaseClassLoader classLoader, String name, boolean allExports)
/*     */     throws ClassNotFoundException
/*     */   {
/* 183 */     boolean trace = log.isTraceEnabled();
/*     */ 
/* 185 */     String path = ClassLoaderUtils.classNameToPath(name);
/*     */ 
/* 187 */     Loader loader = findLoader(classLoader, path, allExports);
/* 188 */     if (loader != null)
/*     */     {
/* 190 */       Thread thread = Thread.currentThread();
/* 191 */       ClassLoadingTask task = new ClassLoadingTask(name, classLoader, thread);
/* 192 */       ClassLoaderManager.scheduleTask(task, loader, false);
/* 193 */       return ClassLoaderManager.process(thread, task);
/*     */     }
/*     */ 
/* 196 */     if (classLoader != null)
/*     */     {
/* 198 */       BaseClassLoaderPolicy policy = classLoader.getPolicy();
/* 199 */       ClassLoader hack = policy.isJDKRequest(name);
/* 200 */       if (hack != null)
/*     */       {
/* 202 */         if (trace)
/* 203 */           log.trace(this + " trying to load " + name + " using hack " + hack);
/* 204 */         Class result = hack.loadClass(name);
/* 205 */         if (result != null)
/*     */         {
/* 207 */           if (trace)
/* 208 */             log.trace(this + " loaded from hack " + hack + " " + ClassLoaderUtils.classToString(result));
/* 209 */           return result;
/*     */         }
/*     */       }
/*     */ 
/*     */     }
/*     */ 
/* 215 */     return null;
/*     */   }
/*     */ 
/*     */   protected Loader findLoader(String name)
/*     */   {
/* 226 */     return findLoader(null, name, true);
/*     */   }
/*     */ 
/*     */   Loader findLoader(BaseClassLoader classLoader, String name, boolean allExports)
/*     */   {
/* 239 */     boolean trace = log.isTraceEnabled();
/* 240 */     if (trace) {
/* 241 */       log.trace(this + " findLoader " + name + " classLoader=" + classLoader + " allExports=" + allExports);
/*     */     }
/* 243 */     if (getClassLoaderSystem() == null) {
/* 244 */       throw new IllegalStateException("Domain is not registered with a classloader system: " + toLongString());
/*     */     }
/*     */ 
/* 247 */     Loader loader = findBeforeLoader(name);
/* 248 */     if (loader != null) {
/* 249 */       return loader;
/*     */     }
/*     */ 
/* 252 */     ClassLoaderInformation info = null;
/* 253 */     BaseClassLoaderPolicy policy = null;
/* 254 */     if (classLoader != null)
/*     */     {
/* 256 */       info = (ClassLoaderInformation)this.infos.get(classLoader);
/* 257 */       policy = classLoader.getPolicy();
/* 258 */       if (policy.isImportAll()) {
/* 259 */         allExports = true;
/*     */       }
/*     */     }
/*     */ 
/* 263 */     if (allExports)
/*     */     {
/* 265 */       loader = findLoaderInExports(classLoader, name, trace);
/* 266 */       if (loader != null)
/* 267 */         return loader;
/*     */     }
/* 269 */     else if (trace) {
/* 270 */       log.trace(this + " not loading " + name + " from all exports");
/*     */     }
/*     */ 
/* 273 */     if (info != null)
/*     */     {
/* 275 */       loader = findLoaderInImports(info, name, trace);
/* 276 */       if (loader != null) {
/* 277 */         return loader;
/*     */       }
/*     */     }
/*     */ 
/* 281 */     if (classLoader != null)
/*     */     {
/* 283 */       if (trace)
/* 284 */         log.trace(this + " trying to load " + name + " from requesting " + classLoader);
/* 285 */       if (classLoader.getResourceLocally(name) != null) {
/* 286 */         return classLoader.getLoader();
/*     */       }
/*     */     }
/*     */ 
/* 290 */     return findAfterLoader(name);
/*     */   }
/*     */ 
/*     */   URL getResource(BaseClassLoader classLoader, String name, boolean allExports)
/*     */   {
/* 303 */     boolean trace = log.isTraceEnabled();
/*     */ 
/* 305 */     if (getClassLoaderSystem() == null) {
/* 306 */       throw new IllegalStateException("Domain is not registered with a classloader system: " + toLongString());
/*     */     }
/*     */ 
/* 309 */     URL result = beforeGetResource(name);
/* 310 */     if (result != null) {
/* 311 */       return result;
/*     */     }
/*     */ 
/* 314 */     ClassLoaderInformation info = null;
/* 315 */     BaseClassLoaderPolicy policy = null;
/* 316 */     if (classLoader != null)
/*     */     {
/* 318 */       policy = classLoader.getPolicy();
/* 319 */       info = (ClassLoaderInformation)this.infos.get(classLoader);
/* 320 */       if (policy.isImportAll()) {
/* 321 */         allExports = true;
/*     */       }
/*     */     }
/*     */ 
/* 325 */     if (allExports)
/*     */     {
/* 327 */       result = getResourceFromExports(classLoader, name, trace);
/* 328 */       if (result != null)
/* 329 */         return result;
/*     */     }
/* 331 */     else if (trace) {
/* 332 */       log.trace(this + " not getting resource " + name + " from all exports");
/*     */     }
/*     */ 
/* 335 */     if (info != null)
/*     */     {
/* 337 */       result = getResourceFromImports(info, name, trace);
/* 338 */       if (result != null) {
/* 339 */         return result;
/*     */       }
/*     */     }
/*     */ 
/* 343 */     if (classLoader != null)
/*     */     {
/* 345 */       if (trace)
/* 346 */         log.trace(this + " trying to get resource " + name + " from requesting " + classLoader);
/* 347 */       result = classLoader.getResourceLocally(name);
/* 348 */       if (result != null)
/*     */       {
/* 350 */         if (trace)
/* 351 */           log.trace(this + " got resource from requesting " + classLoader + " " + result);
/* 352 */         return result;
/*     */       }
/*     */ 
/*     */     }
/*     */ 
/* 357 */     result = afterGetResource(name);
/* 358 */     if (result != null) {
/* 359 */       return result;
/*     */     }
/*     */ 
/* 362 */     return null;
/*     */   }
/*     */ 
/*     */   void getResources(BaseClassLoader classLoader, String name, Set<URL> urls, boolean allExports)
/*     */     throws IOException
/*     */   {
/* 376 */     boolean trace = log.isTraceEnabled();
/*     */ 
/* 378 */     if (getClassLoaderSystem() == null) {
/* 379 */       throw new IllegalStateException("Domain is not registered with a classloader system: " + toLongString());
/*     */     }
/*     */ 
/* 382 */     beforeGetResources(name, urls);
/*     */ 
/* 385 */     ClassLoaderInformation info = null;
/* 386 */     BaseClassLoaderPolicy policy = null;
/* 387 */     if (classLoader != null)
/*     */     {
/* 389 */       policy = classLoader.getPolicy();
/* 390 */       info = (ClassLoaderInformation)this.infos.get(classLoader);
/* 391 */       if (policy.isImportAll()) {
/* 392 */         allExports = true;
/*     */       }
/*     */     }
/*     */ 
/* 396 */     if (allExports)
/* 397 */       getResourcesFromExports(classLoader, name, urls, trace);
/* 398 */     else if (trace) {
/* 399 */       log.trace(this + " not getting resource " + name + " from all exports");
/*     */     }
/*     */ 
/* 402 */     if (info != null) {
/* 403 */       getResourcesFromImports(info, name, urls, trace);
/*     */     }
/*     */ 
/* 406 */     if (classLoader != null)
/*     */     {
/* 408 */       if (trace)
/* 409 */         log.trace(this + " trying to get resources " + name + " from requesting " + classLoader);
/* 410 */       classLoader.getResourcesLocally(name, urls);
/*     */     }
/*     */ 
/* 414 */     afterGetResources(name, urls);
/*     */   }
/*     */ 
/*     */   private Loader findLoaderInExports(BaseClassLoader classLoader, String name, boolean trace)
/*     */   {
/* 427 */     Loader loader = (Loader)this.globalClassCache.get(name);
/* 428 */     if (loader != null)
/*     */     {
/* 430 */       if (trace) {
/* 431 */         log.trace(this + " found in global class cache " + name);
/*     */       }
/* 433 */       return loader;
/*     */     }
/*     */ 
/* 436 */     if (this.globalClassBlackList.contains(name))
/*     */     {
/* 438 */       if (trace)
/* 439 */         log.trace(this + " class is black listed " + name);
/* 440 */       return null;
/*     */     }
/* 442 */     boolean canCache = true;
/* 443 */     boolean canBlackList = true;
/*     */ 
/* 445 */     String packageName = ClassLoaderUtils.getResourcePackageName(name);
/* 446 */     List list = (List)this.classLoadersByPackageName.get(packageName);
/* 447 */     if (trace)
/* 448 */       log.trace(this + " trying to load " + name + " from all exports of package " + packageName + " " + list);
/* 449 */     if ((list != null) && (!list.isEmpty()))
/*     */     {
/* 451 */       for (ClassLoaderInformation info : list)
/*     */       {
/* 453 */         BaseDelegateLoader exported = info.getExported();
/*     */ 
/* 456 */         BaseClassLoaderPolicy loaderPolicy = exported.getPolicy();
/* 457 */         if (!loaderPolicy.isCachable())
/* 458 */           canCache = false;
/* 459 */         if (!loaderPolicy.isBlackListable()) {
/* 460 */           canBlackList = false;
/*     */         }
/* 462 */         if (exported.getResource(name) != null)
/*     */         {
/* 464 */           if (canCache)
/* 465 */             this.globalClassCache.put(name, exported);
/* 466 */           return exported;
/*     */         }
/*     */       }
/*     */     }
/*     */ 
/* 471 */     if (canBlackList) {
/* 472 */       this.globalClassBlackList.add(name);
/*     */     }
/* 474 */     return null;
/*     */   }
/*     */ 
/*     */   private URL getResourceFromExports(BaseClassLoader classLoader, String name, boolean trace)
/*     */   {
/* 487 */     URL result = (URL)this.globalResourceCache.get(name);
/* 488 */     if (result != null)
/*     */     {
/* 490 */       if (trace) {
/* 491 */         log.trace(this + " got resource from cache " + name);
/*     */       }
/*     */     }
/* 494 */     if (this.globalResourceBlackList.contains(name))
/*     */     {
/* 496 */       if (trace)
/* 497 */         log.trace(this + " resource is black listed, not looking at exports " + name);
/* 498 */       return null;
/*     */     }
/* 500 */     boolean canCache = true;
/* 501 */     boolean canBlackList = true;
/*     */ 
/* 503 */     String packageName = ClassLoaderUtils.getResourcePackageName(name);
/* 504 */     List list = (List)this.classLoadersByPackageName.get(packageName);
/* 505 */     if (trace)
/* 506 */       log.trace(this + " trying to get resource " + name + " from all exports " + list);
/* 507 */     if ((list != null) && (!list.isEmpty()))
/*     */     {
/* 509 */       for (ClassLoaderInformation info : list)
/*     */       {
/* 511 */         BaseDelegateLoader loader = info.getExported();
/*     */ 
/* 514 */         BaseClassLoaderPolicy loaderPolicy = loader.getPolicy();
/* 515 */         if (!loaderPolicy.isCachable())
/* 516 */           canCache = false;
/* 517 */         if (!loaderPolicy.isBlackListable()) {
/* 518 */           canBlackList = false;
/*     */         }
/* 520 */         result = loader.getResource(name);
/* 521 */         if (result != null)
/*     */         {
/* 523 */           if (canCache)
/* 524 */             this.globalResourceCache.put(name, result);
/* 525 */           return result;
/*     */         }
/*     */       }
/*     */     }
/*     */ 
/* 530 */     if (canBlackList)
/* 531 */       this.globalResourceBlackList.add(name);
/* 532 */     return null;
/*     */   }
/*     */ 
/*     */   void getResourcesFromExports(BaseClassLoader classLoader, String name, Set<URL> urls, boolean trace)
/*     */     throws IOException
/*     */   {
/* 546 */     String packageName = ClassLoaderUtils.getResourcePackageName(name);
/* 547 */     List list = (List)this.classLoadersByPackageName.get(packageName);
/* 548 */     if (trace)
/* 549 */       log.trace(this + " trying to get resources " + name + " from all exports " + list);
/* 550 */     if ((list != null) && (!list.isEmpty()))
/*     */     {
/* 552 */       for (ClassLoaderInformation info : list)
/*     */       {
/* 554 */         BaseDelegateLoader loader = info.getExported();
/* 555 */         loader.getResources(name, urls);
/*     */       }
/*     */     }
/*     */   }
/*     */ 
/*     */   Loader findLoaderInImports(ClassLoaderInformation info, String name, boolean trace)
/*     */   {
/* 570 */     List delegates = info.getDelegates();
/* 571 */     if ((delegates == null) || (delegates.isEmpty()))
/*     */     {
/* 573 */       if (trace)
/* 574 */         log.trace(this + " not loading " + name + " from imports it has no delegates");
/* 575 */       return null;
/*     */     }
/*     */ 
/* 578 */     Loader loader = info.getCachedLoader(name);
/* 579 */     if (loader != null)
/*     */     {
/* 581 */       if (trace)
/* 582 */         log.trace(this + " found in import cache " + name);
/* 583 */       return loader;
/*     */     }
/*     */ 
/* 586 */     if (info.isBlackListedClass(name))
/*     */     {
/* 588 */       if (trace)
/* 589 */         log.trace(this + " class is black listed in imports " + name);
/* 590 */       return null;
/*     */     }
/*     */ 
/* 593 */     for (DelegateLoader delegate : delegates)
/*     */     {
/* 595 */       if (trace)
/* 596 */         log.trace(this + " trying to load " + name + " from import " + delegate + " for " + info.getClassLoader());
/* 597 */       if (delegate.getResource(name) != null)
/*     */       {
/* 599 */         info.cacheLoader(name, delegate);
/* 600 */         return delegate;
/*     */       }
/*     */     }
/* 603 */     info.blackListClass(name);
/* 604 */     return null;
/*     */   }
/*     */ 
/*     */   private URL getResourceFromImports(ClassLoaderInformation info, String name, boolean trace)
/*     */   {
/* 617 */     List delegates = info.getDelegates();
/* 618 */     if ((delegates == null) || (delegates.isEmpty()))
/*     */     {
/* 620 */       if (trace)
/* 621 */         log.trace(this + " not getting resource " + name + " from imports it has no delegates");
/* 622 */       return null;
/*     */     }
/*     */ 
/* 625 */     URL url = info.getCachedResource(name);
/* 626 */     if (url != null)
/*     */     {
/* 628 */       if (trace)
/* 629 */         log.trace(this + " found resource in import cache " + name);
/* 630 */       return url;
/*     */     }
/*     */ 
/* 633 */     if (info.isBlackListedResource(name))
/*     */     {
/* 635 */       if (trace)
/* 636 */         log.trace(this + " resource is black listed in imports " + name);
/* 637 */       return null;
/*     */     }
/*     */ 
/* 640 */     if (trace) {
/* 641 */       log.trace(this + " trying to get resource " + name + " from imports " + delegates + " for " + info.getClassLoader());
/*     */     }
/* 643 */     for (DelegateLoader delegate : delegates)
/*     */     {
/* 645 */       URL result = delegate.getResource(name);
/* 646 */       if (result != null)
/*     */       {
/* 648 */         info.cacheResource(name, result);
/* 649 */         return result;
/*     */       }
/*     */     }
/* 652 */     info.blackListResource(name);
/* 653 */     return null;
/*     */   }
/*     */ 
/*     */   void getResourcesFromImports(ClassLoaderInformation info, String name, Set<URL> urls, boolean trace)
/*     */     throws IOException
/*     */   {
/* 667 */     List delegates = info.getDelegates();
/* 668 */     if ((delegates == null) || (delegates.isEmpty()))
/*     */     {
/* 670 */       if (trace)
/* 671 */         log.trace(this + " not getting resource " + name + " from imports it has no delegates");
/* 672 */       return;
/*     */     }
/* 674 */     if (trace)
/* 675 */       log.trace(this + " trying to get resources " + name + " from imports " + delegates + " for " + info.getClassLoader());
/* 676 */     for (DelegateLoader delegate : delegates)
/* 677 */       delegate.getResources(name, urls);
/*     */   }
/*     */ 
/*     */   protected abstract Loader findBeforeLoader(String paramString);
/*     */ 
/*     */   protected abstract Loader findAfterLoader(String paramString);
/*     */ 
/*     */   protected abstract void beforeGetResources(String paramString, Set<URL> paramSet)
/*     */     throws IOException;
/*     */ 
/*     */   protected abstract void afterGetResources(String paramString, Set<URL> paramSet)
/*     */     throws IOException;
/*     */ 
/*     */   protected abstract URL beforeGetResource(String paramString);
/*     */ 
/*     */   protected abstract URL afterGetResource(String paramString);
/*     */ 
/*     */   public Class<?> loadClass(String name)
/*     */   {
/*     */     try
/*     */     {
/* 734 */       return loadClass(null, name, true);
/*     */     }
/*     */     catch (ClassNotFoundException e) {
/*     */     }
/* 738 */     return null;
/*     */   }
/*     */ 
/*     */   Class<?> loadClass(BaseClassLoader classLoader, String name)
/*     */     throws ClassNotFoundException
/*     */   {
/* 752 */     return loadClass(classLoader, name, false);
/*     */   }
/*     */ 
/*     */   public URL getResource(String name)
/*     */   {
/* 757 */     return getResource(null, name, true);
/*     */   }
/*     */ 
/*     */   URL getResource(BaseClassLoader classLoader, String name)
/*     */   {
/* 769 */     return getResource(classLoader, name, false);
/*     */   }
/*     */ 
/*     */   public void getResources(String name, Set<URL> urls) throws IOException
/*     */   {
/* 774 */     getResources(null, name, urls, true);
/*     */   }
/*     */ 
/*     */   void getResources(BaseClassLoader classLoader, String name, Set<URL> urls)
/*     */     throws IOException
/*     */   {
/* 787 */     getResources(classLoader, name, urls, false);
/*     */   }
/*     */ 
/*     */   public String toLongString()
/*     */   {
/* 797 */     StringBuilder builder = new StringBuilder();
/* 798 */     builder.append(getClass().getSimpleName());
/* 799 */     builder.append("@").append(Integer.toHexString(System.identityHashCode(this)));
/* 800 */     builder.append("{");
/* 801 */     toLongString(builder);
/* 802 */     builder.append('}');
/* 803 */     return builder.toString();
/*     */   }
/*     */ 
/*     */   protected void toLongString(StringBuilder builder)
/*     */   {
/*     */   }
/*     */ 
/*     */   protected void beforeRegisterClassLoader(ClassLoader classLoader, ClassLoaderPolicy policy)
/*     */   {
/*     */   }
/*     */ 
/*     */   protected void afterRegisterClassLoader(ClassLoader classLoader, ClassLoaderPolicy policy)
/*     */   {
/*     */   }
/*     */ 
/*     */   protected void beforeUnregisterClassLoader(ClassLoader classLoader, ClassLoaderPolicy policy)
/*     */   {
/*     */   }
/*     */ 
/*     */   protected void afterUnregisterClassLoader(ClassLoader classLoader, ClassLoaderPolicy policy)
/*     */   {
/*     */   }
/*     */ 
/*     */   protected ClassLoader getParentClassLoader()
/*     */   {
/* 866 */     return getClass().getClassLoader();
/*     */   }
/*     */ 
/*     */   void registerClassLoader(BaseClassLoader classLoader)
/*     */   {
/* 876 */     log.debug(this + " registerClassLoader " + classLoader.toLongString());
/*     */ 
/* 878 */     if (getClassLoaderSystem() == null) {
/* 879 */       throw new IllegalStateException("Domain is not registered with a classloader system: " + toLongString());
/*     */     }
/*     */     try
/*     */     {
/* 883 */       beforeRegisterClassLoader(classLoader, classLoader.getPolicy());
/*     */     }
/*     */     catch (Throwable t)
/*     */     {
/* 887 */       log.warn("Error in beforeRegisterClassLoader: " + this + " classLoader=" + classLoader.toLongString(), t);
/*     */     }
/*     */ 
/* 890 */     BaseClassLoaderPolicy policy = classLoader.getPolicy();
/* 891 */     policy.setClassLoaderDomain(this);
/*     */ 
/* 893 */     synchronized (this.classLoaders)
/*     */     {
/* 896 */       ClassLoaderInformation info = new ClassLoaderInformation(classLoader, policy, this.order++);
/* 897 */       this.classLoaders.add(info);
/* 898 */       this.infos.put(classLoader, info);
/*     */ 
/* 901 */       String[] packageNames = policy.getPackageNames();
/* 902 */       if ((packageNames != null) && (info.getExported() != null))
/*     */       {
/* 904 */         for (String packageName : packageNames)
/*     */         {
/* 906 */           List list = (List)this.classLoadersByPackageName.get(packageName);
/* 907 */           if (list == null)
/*     */           {
/* 909 */             list = new CopyOnWriteArrayList();
/* 910 */             this.classLoadersByPackageName.put(packageName, list);
/*     */           }
/* 912 */           list.add(info);
/* 913 */           log.trace("Registered " + policy + " as providing package=" + packageName);
/*     */         }
/*     */       }
/*     */ 
/* 917 */       flushCaches();
/*     */     }
/*     */ 
/*     */     try
/*     */     {
/* 922 */       afterRegisterClassLoader(classLoader, classLoader.getPolicy());
/*     */     }
/*     */     catch (Throwable t)
/*     */     {
/* 926 */       log.warn("Error in afterRegisterClassLoader: " + this + " classLoader=" + classLoader.toLongString(), t);
/*     */     }
/*     */   }
/*     */ 
/*     */   synchronized void unregisterClassLoader(BaseClassLoader classLoader)
/*     */   {
/* 937 */     log.debug(this + " unregisterClassLoader " + classLoader.toLongString());
/*     */     try
/*     */     {
/* 941 */       beforeUnregisterClassLoader(classLoader, classLoader.getPolicy());
/*     */     }
/*     */     catch (Throwable t)
/*     */     {
/* 945 */       log.warn("Error in beforeUnegisterClassLoader: " + this + " classLoader=" + classLoader.toLongString(), t);
/*     */     }
/*     */ 
/* 948 */     BaseClassLoaderPolicy policy = classLoader.getPolicy();
/* 949 */     policy.unsetClassLoaderDomain(this);
/*     */ 
/* 951 */     synchronized (this.classLoaders)
/*     */     {
/* 954 */       ClassLoaderInformation info = (ClassLoaderInformation)this.infos.remove(classLoader);
/* 955 */       this.classLoaders.remove(info);
/*     */ 
/* 958 */       String[] packageNames = policy.getPackageNames();
/* 959 */       if ((packageNames != null) && (info.getExported() != null))
/*     */       {
/* 961 */         for (String packageName : packageNames)
/*     */         {
/* 963 */           List list = (List)this.classLoadersByPackageName.get(packageName);
/* 964 */           if (list == null)
/*     */             continue;
/* 966 */           list.remove(info);
/* 967 */           log.trace("Unregistered " + policy + " as providing package=" + packageName);
/* 968 */           if (list.isEmpty()) {
/* 969 */             this.classLoadersByPackageName.remove(packageName);
/*     */           }
/*     */         }
/*     */       }
/*     */ 
/* 974 */       flushCaches();
/*     */     }
/*     */ 
/*     */     try
/*     */     {
/* 979 */       afterUnregisterClassLoader(classLoader, classLoader.getPolicy());
/*     */     }
/*     */     catch (Throwable t)
/*     */     {
/* 983 */       log.warn("Error in afterUnegisterClassLoader: " + this + " classLoader=" + classLoader.toLongString(), t);
/*     */     }
/*     */   }
/*     */ 
/*     */   protected void clearBlackList(String name)
/*     */   {
/* 994 */     if (this.globalClassBlackList != null)
/*     */     {
/* 996 */       this.globalClassBlackList.remove(name);
/*     */     }
/* 998 */     if (this.globalResourceBlackList != null)
/*     */     {
/* 1000 */       this.globalResourceBlackList.remove(name);
/*     */     }
/*     */   }
/*     */ }

/* Location:           /home/mnovotny/projects/EMBEDDED_JBOSS_BETA3_COMMUNITY/embedded/output/lib/embedded-jboss/lib/jboss-embedded-all.jar
 * Qualified Name:     org.jboss.classloader.spi.base.BaseClassLoaderDomain
 * JD-Core Version:    0.6.0
 */