package org.immutables.value.processor.meta;

import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import com.google.common.primitives.Booleans;
import java.lang.annotation.Annotation;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import javax.lang.model.element.*;
import javax.lang.model.type.*;
import javax.lang.model.util.*;

/**
 * ImmutableMirror used to parse data of AnnotationMirror for original annotation {@code org.immutables.value.Value.Immutable}
 * during annotation processing. Interface is being described using {@link org.immutables.value.processor.meta.ValueMirrors.Immutable} annotation,
 * which should be structurally compatible to the annotation being modelled.
 * @see #find(Iterable)
 * @see #from(AnnotationMirror)
 */
@SuppressWarnings("all")
public class ImmutableMirror implements ValueMirrors.Immutable {
  public static final String QUALIFIED_NAME = "org.immutables.value.Value.Immutable";
  public static final String MIRROR_QUALIFIED_NAME = "org.immutables.value.processor.meta.ValueMirrors.Immutable";

  public static String mirrorQualifiedName() {
    return QUALIFIED_NAME;
  }

  public static String qualifiedName() {
    return QUALIFIED_NAME;
  }

  public static String simpleName() {
    return "Immutable";
  }

  public static boolean isPresent(Element annotatedElement) {
    for (AnnotationMirror mirror : annotatedElement.getAnnotationMirrors()) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      if (element.getQualifiedName().contentEquals(QUALIFIED_NAME)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Finds first annotation of this type on the element.
   * @param element annotated element
   * @return optional {@code ImmutableMirror}, present if this annotation found
   */
  public static Optional<ImmutableMirror> find(Element element) {
    return find(element.getAnnotationMirrors());
  }

  /**
   * Finds first annotation of this type in an iterable of annotation mirrors.
   * @param mirrors annotation mirrors
   * @return optional {@code ImmutableMirror}, present if this annotation found
   */
  public static Optional<ImmutableMirror> find(Iterable<? extends AnnotationMirror> mirrors) {
    for (AnnotationMirror mirror : mirrors) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      if (element.getQualifiedName().contentEquals(QUALIFIED_NAME)) {
        return Optional.of(new ImmutableMirror(mirror));
      }
    }
    return Optional.absent();
  }

  /**
   * Converts iterable of annotation mirrors where all annotation are of this type. Otherwise it fails
   * @param mirrors of this annotation type.
   * @return list of converted {@code ImmutableMirror}s
   */
  public static ImmutableList<ImmutableMirror> fromAll(Iterable<? extends AnnotationMirror> mirrors) {
    ImmutableList.Builder<ImmutableMirror> builder = ImmutableList.builder();
    for (AnnotationMirror mirror : mirrors) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      Preconditions.checkState(element.getQualifiedName().contentEquals(QUALIFIED_NAME),
          "Supplied mirrors should all be of this annotation type");
      builder.add(new ImmutableMirror(mirror));
    }
    return builder.build();
  }

  /**
   * Creates mirror with default values using annotation element (i.e. declaration, not usage).
   * @param element annotation type element
   * @return {@code ImmutableMirror}
   */
  public static ImmutableMirror from(TypeElement element) {
    return new ImmutableMirror(element);
  }

  /**
   * Tries to convert annotation mirror to this annotation type.
   * @param mirror annotation mirror
   * @return optional {@code ImmutableMirror}, present if mirror matched this annotation type
   */
  public static Optional<ImmutableMirror> from(AnnotationMirror mirror) {
    return find(Collections.singleton(mirror));
  }

  private final AnnotationMirror annotationMirror;
  private final boolean singleton;
  private final boolean intern;
  private final boolean copy;
  private final boolean prehash;
  private final boolean builder;

  private ImmutableMirror(TypeElement defaultAnnotationElement) {
    Preconditions.checkArgument(defaultAnnotationElement.getQualifiedName().contentEquals(QUALIFIED_NAME)
        || defaultAnnotationElement.getQualifiedName().contentEquals(MIRROR_QUALIFIED_NAME));
    this.annotationMirror = null;

    // TBD TODO BIG

    boolean singleton = false;
    boolean intern = false;
    boolean copy = false;
    boolean prehash = false;
    boolean builder = false;

    for (ExecutableElement attributeElement$
        : ElementFilter.methodsIn(defaultAnnotationElement.getEnclosedElements())) {
      String name$ = attributeElement$.getSimpleName().toString();
      if ("singleton".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Immutable");
        }
        SingletonExtractor singletonExtractor$ = new SingletonExtractor();
        annotationValue$.accept(singletonExtractor$, null);

        singleton = singletonExtractor$.get();
        continue;
      }
      if ("intern".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Immutable");
        }
        InternExtractor internExtractor$ = new InternExtractor();
        annotationValue$.accept(internExtractor$, null);

        intern = internExtractor$.get();
        continue;
      }
      if ("copy".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Immutable");
        }
        CopyExtractor copyExtractor$ = new CopyExtractor();
        annotationValue$.accept(copyExtractor$, null);

        copy = copyExtractor$.get();
        continue;
      }
      if ("prehash".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Immutable");
        }
        PrehashExtractor prehashExtractor$ = new PrehashExtractor();
        annotationValue$.accept(prehashExtractor$, null);

        prehash = prehashExtractor$.get();
        continue;
      }
      if ("builder".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Immutable");
        }
        BuilderExtractor builderExtractor$ = new BuilderExtractor();
        annotationValue$.accept(builderExtractor$, null);

        builder = builderExtractor$.get();
        continue;
      }
    }
    this.singleton = singleton;
    this.intern = intern;
    this.copy = copy;
    this.prehash = prehash;
    this.builder = builder;
  }

  private ImmutableMirror(AnnotationMirror annotationMirror) {
    this.annotationMirror = annotationMirror;

    boolean singleton = false;
    boolean intern = false;
    boolean copy = false;
    boolean prehash = false;
    boolean builder = false;

    Map<? extends ExecutableElement, ? extends AnnotationValue> attributeValues$ = annotationMirror.getElementValues();
    for (ExecutableElement attributeElement$
        : ElementFilter.methodsIn(annotationMirror.getAnnotationType().asElement().getEnclosedElements())) {
      String name$ = attributeElement$.getSimpleName().toString();
      if ("singleton".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'singleton' attribute of @Immutable");
        }
        SingletonExtractor singletonExtractor$ = new SingletonExtractor();
        annotationValue$.accept(singletonExtractor$, null);

        singleton = singletonExtractor$.get();
        continue;
      }
      if ("intern".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'intern' attribute of @Immutable");
        }
        InternExtractor internExtractor$ = new InternExtractor();
        annotationValue$.accept(internExtractor$, null);

        intern = internExtractor$.get();
        continue;
      }
      if ("copy".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'copy' attribute of @Immutable");
        }
        CopyExtractor copyExtractor$ = new CopyExtractor();
        annotationValue$.accept(copyExtractor$, null);

        copy = copyExtractor$.get();
        continue;
      }
      if ("prehash".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'prehash' attribute of @Immutable");
        }
        PrehashExtractor prehashExtractor$ = new PrehashExtractor();
        annotationValue$.accept(prehashExtractor$, null);

        prehash = prehashExtractor$.get();
        continue;
      }
      if ("builder".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'builder' attribute of @Immutable");
        }
        BuilderExtractor builderExtractor$ = new BuilderExtractor();
        annotationValue$.accept(builderExtractor$, null);

        builder = builderExtractor$.get();
        continue;
      }
    }
    this.singleton = singleton;
    this.intern = intern;
    this.copy = copy;
    this.prehash = prehash;
    this.builder = builder;
  }

  /**
   * @return value of attribute {@code singleton}
   */
  @Override
  public boolean singleton() {
    return singleton;
  }

  /**
   * @return value of attribute {@code intern}
   */
  @Override
  public boolean intern() {
    return intern;
  }

  /**
   * @return value of attribute {@code copy}
   */
  @Override
  public boolean copy() {
    return copy;
  }

  /**
   * @return value of attribute {@code prehash}
   */
  @Override
  public boolean prehash() {
    return prehash;
  }

  /**
   * @return value of attribute {@code builder}
   */
  @Override
  public boolean builder() {
    return builder;
  }

  /**
   * @return underlying annotation mirror
   */
  public AnnotationMirror getAnnotationMirror() {
    Preconditions.checkState(annotationMirror != null, "this is default mirror without originating AnnotationMirror");
    return annotationMirror;
  }

  /**
   * @return {@code Immutable.class}
   */
  @Override
  public Class<? extends Annotation> annotationType() {
    return ValueMirrors.Immutable.class;
  }

  @Override
  public int hashCode() {
    int h = 0;
    h += 127 * "singleton".hashCode() ^ Booleans.hashCode(singleton);
    h += 127 * "intern".hashCode() ^ Booleans.hashCode(intern);
    h += 127 * "copy".hashCode() ^ Booleans.hashCode(copy);
    h += 127 * "prehash".hashCode() ^ Booleans.hashCode(prehash);
    h += 127 * "builder".hashCode() ^ Booleans.hashCode(builder);
    return h;
  }

  @Override
  public boolean equals(Object other) {
    if (other instanceof ImmutableMirror) {
      ImmutableMirror otherMirror = (ImmutableMirror) other;
      return singleton == otherMirror.singleton
          && intern == otherMirror.intern
          && copy == otherMirror.copy
          && prehash == otherMirror.prehash
          && builder == otherMirror.builder;
    }
    return false;
  }

  @Override
  public String toString() {
    return "ImmutableMirror:" + annotationMirror;
  }

  private static class SingletonExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'singleton' in @" + QUALIFIED_NAME);
    }
  }

  private static class InternExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'intern' in @" + QUALIFIED_NAME);
    }
  }

  private static class CopyExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'copy' in @" + QUALIFIED_NAME);
    }
  }

  private static class PrehashExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'prehash' in @" + QUALIFIED_NAME);
    }
  }

  private static class BuilderExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'builder' in @" + QUALIFIED_NAME);
    }
  }
}
