/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.business.services;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.business.converters.DataPollAccountConverter;
import org.chorem.pollen.business.converters.DataPollConverter;
import org.chorem.pollen.business.dto.PollAccountDTO;
import org.chorem.pollen.business.dto.PollDTO;
import org.chorem.pollen.business.persistence.PersonToList;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.PollAccount;
import org.chorem.pollen.business.persistence.PollDAO;
import org.chorem.pollen.business.persistence.PollenModelDAOHelper;
import org.chorem.pollen.business.persistence.UserAccount;
import org.chorem.pollen.business.persistence.UserAccountDAO;
import org.chorem.pollen.business.persistence.Vote;
import org.chorem.pollen.business.persistence.VoteDAO;
import org.chorem.pollen.business.persistence.VotingList;
import org.chorem.pollen.business.utils.ContextUtil;
import org.nuiton.topia.TopiaContext;

/**
 * Implémentation du service de gestion des sondages.
 *
 * @author Erwan Nema
 * @author rannou
 * @version $Id: ServicePollImpl.java 2933 2010-03-12 14:31:05Z fdesbois $
 */
public class ServicePollImpl implements ServicePoll {
    private TopiaContext rootContext = ContextUtil.getInstance().getContext();
    private PollDAO pollDAO = null;
    private DataPollConverter converter = new DataPollConverter();

    /** log. */
    private static final Log log = LogFactory.getLog(ServicePollImpl.class);

    public ServicePollImpl() {
    }

    @Override
    public String createPoll(PollDTO pollDTO) {
        TopiaContext transaction = null;
        String topiaId = "";
        try {
            transaction = rootContext.beginTransaction();

            pollDAO = PollenModelDAOHelper.getPollDAO(transaction);

            Poll pollEntity = pollDAO.create();

            // Identifiant du sondage
            String id = UUID.randomUUID().toString();
            pollDTO.setPollId(id.replaceAll("-", ""));

            converter.setTransaction(transaction);
            converter.populatePollEntity(pollDTO, pollEntity);

            // Création du pollAccount associé au sondage
            // FIXME do not call a Service from an other one
            ServicePollAccount spa = new ServicePollAccountImpl();
            PollAccount pollAccountEntity = spa.createPollAccount(pollDTO
                    .getCreatorId(), pollDTO.getCreatorEmail(), pollDTO
                    .getUserId());
            pollEntity.setCreator(pollAccountEntity);

            if (log.isDebugEnabled()) {
                log.debug("userId: " + pollDTO.getUserId()
                        + ", pollAccountEntity created: "
                        + (pollAccountEntity != null));
            }

            topiaId = pollEntity.getTopiaId();

            // Création des choix du sondage
            // FIXME do not call a Service from an other one
            ServiceChoice sChoice = new ServiceChoiceImpl();
            pollEntity.setChoice(sChoice.createChoices(pollDTO.getChoiceDTOs(),
                    topiaId, transaction));

            // Création des commentaires du sondage
            // FIXME do not call a Service from an other one
            ServiceComment sComment = new ServiceCommentImpl();
            pollEntity.setComment(sComment.createComments(pollDTO
                    .getCommentDTOs(), topiaId, transaction));

            // Création des listes de votants du sondage
            // FIXME do not call a Service from an other one
            ServiceList sList = new ServiceListImpl();
            pollEntity.setVotingList(sList.createVotingLists(pollDTO
                    .getVotingListDTOs(), topiaId, transaction));

            // Création des règle de notification pour le sondage
            // FIXME do not call a Service from an other one
            ServicePreventRule sPreventRule = new ServicePreventRuleImpl();
            pollEntity.setPreventRule(sPreventRule.createPreventRules(pollDTO
                    .getPreventRuleDTOs(), topiaId, transaction));

            transaction.commitTransaction();

            if (log.isDebugEnabled()) {
                log.debug("creator after creation: " + pollEntity.getCreator());
            }
            if (log.isDebugEnabled()) {
                log.debug("Entity created: " + topiaId);
            }

            return topiaId;
        } catch (Exception e) {
            ContextUtil.doCatch(e, transaction);
            return "";
        } finally {
            ContextUtil.doFinally(transaction);
        }
    }

    @Override
    public boolean updatePoll(PollDTO pollDTO) {
        TopiaContext transaction = null;
        try {
            transaction = rootContext.beginTransaction();

            pollDAO = PollenModelDAOHelper.getPollDAO(transaction);

            Poll pollEntity = pollDAO.findByTopiaId(pollDTO.getId());

            converter.setTransaction(transaction);
            converter.populatePollEntity(pollDTO, pollEntity);

            converter.persistChoices(pollDTO, pollEntity);
            converter.persistPreventRules(pollDTO, pollEntity);

            pollDAO.update(pollEntity);

            transaction.commitTransaction();

            if (log.isDebugEnabled()) {
                log.debug("Entity updated: " + pollDTO.getId());
            }

            return true;
        } catch (Exception e) {
            ContextUtil.doCatch(e, transaction);
            return false;
        } finally {
            ContextUtil.doFinally(transaction);
        }
    }

    @Override
    public boolean deletePoll(String pollId) {
        TopiaContext transaction = null;
        try {
            transaction = rootContext.beginTransaction();

            pollDAO = PollenModelDAOHelper.getPollDAO(transaction);

            Poll pollEntity = pollDAO.findByTopiaId(pollId);

            pollDAO.delete(pollEntity);
            transaction.commitTransaction();

            if (log.isDebugEnabled()) {
                log.debug("Entity deleted: " + pollId);
            }

            return true;
        } catch (Exception e) {
            ContextUtil.doCatch(e, transaction);
            return false;
        } finally {
            ContextUtil.doFinally(transaction);
        }
    }

    @Override
    public PollDTO findPollById(String pollId) {
        TopiaContext transaction = null;
        PollDTO result = null;
        try {
            transaction = rootContext.beginTransaction();

            pollDAO = PollenModelDAOHelper.getPollDAO(transaction);

            Poll pollEntity = pollDAO.findByTopiaId(pollId);

            if (pollEntity != null) {
                converter.setTransaction(transaction);
                result = converter.createPollDTO(pollEntity);
            }

            transaction.commitTransaction();

            if (log.isDebugEnabled()) {
                log.debug("Entity found: "
                        + ((result == null) ? "null" : result.getId()));
            }

            return result;
        } catch (Exception e) {
            ContextUtil.doCatch(e, transaction);
            return null;
        } finally {
            ContextUtil.doFinally(transaction);
        }
    }

    @Override
    public PollDTO findPollByPollId(String pollId) {
        TopiaContext transaction = null;
        PollDTO result = null;
        try {
            transaction = rootContext.beginTransaction();

            pollDAO = PollenModelDAOHelper.getPollDAO(transaction);

            Poll pollEntity = pollDAO.findByPollId(pollId);

            if (pollEntity != null) {
                converter.setTransaction(transaction);
                result = converter.createPollDTO(pollEntity);
                // Need the number total of votes to push in the dto
                List<Long> tmp = transaction.find(
                        "SELECT COUNT(*)" +
                        " FROM " + Vote.class.getName() +
                        " WHERE poll.pollId = :pollUId",
                        "pollUId", pollId);
                int count = tmp.get(0).intValue();
                result.setNbVotes(count);
            }

            transaction.commitTransaction();

            if (log.isDebugEnabled()) {
                log.debug("Entity found: "
                        + ((result == null) ? "null" : result.getId()));
            }

            return result;
        } catch (Exception e) {
            ContextUtil.doCatch(e, transaction);
            return null;
        } finally {
            ContextUtil.doFinally(transaction);
        }
    }

    @Override
    public List<PollDTO> findPollsByName(String pollName) {
        TopiaContext transaction = null;
        List<PollDTO> results = null;
        List<Poll> pollEntities = null;
        try {
            transaction = rootContext.beginTransaction();

            pollDAO = PollenModelDAOHelper.getPollDAO(transaction);

            pollEntities = pollDAO.findAllByTitle(pollName);

            converter.setTransaction(transaction);
            results = converter.createPollDTOs(pollEntities);

            transaction.commitTransaction();

            if (log.isDebugEnabled()) {
                log.debug("Entities found: "
                        + ((results == null) ? "null" : results.size()));
            }

            return results;
        } catch (Exception e) {
            ContextUtil.doCatch(e, transaction);
            return null;
        } finally {
            ContextUtil.doFinally(transaction);
        }
    }

    @Override
    public List<PollDTO> findPollsByUser(String userId) {
        TopiaContext transaction = null;
        List<PollDTO> results = null;
        UserAccountDAO userDAO = null;
        try {
            transaction = rootContext.beginTransaction();

            userDAO = PollenModelDAOHelper.getUserAccountDAO(transaction);
            UserAccount user = userDAO.findByTopiaId(userId);

            List<Poll> pollEntities = new ArrayList<Poll>();
            for (PollAccount pollAccount : user.getPollAccount()) {
                pollEntities.addAll(pollAccount.getPollsCreated());
            }

            converter.setTransaction(transaction);
            results = converter.createPollDTOs(pollEntities);

            transaction.commitTransaction();

            if (log.isDebugEnabled()) {
                log.debug("Entities found: "
                        + ((results == null) ? "null" : results.size()));
            }

            return results;
        } catch (Exception e) {
            ContextUtil.doCatch(e, transaction);
            return null;
        } finally {
            ContextUtil.doFinally(transaction);
        }
    }

    @Override
    public List<PollDTO> findParticipatedPolls(String userId) {
        TopiaContext transaction = null;
        List<PollDTO> results = null;
        UserAccountDAO userDAO = null;
        try {
            transaction = rootContext.beginTransaction();

            userDAO = PollenModelDAOHelper.getUserAccountDAO(transaction);
            UserAccount user = userDAO.findByTopiaId(userId);

            Set<Poll> pollEntities = new HashSet<Poll>();
            for (PollAccount pollAccount : user.getPollAccount()) {
                for (Vote vote : pollAccount.getVote()) {
                    pollEntities.add(vote.getPoll());
                }
            }

            converter.setTransaction(transaction);
            results = converter
                    .createPollDTOs(new ArrayList<Poll>(pollEntities));

            transaction.commitTransaction();

            if (log.isDebugEnabled()) {
                log.debug("Entities found: "
                        + ((results == null) ? "null" : results.size()));
            }

            return results;
        } catch (Exception e) {
            ContextUtil.doCatch(e, transaction);
            return null;
        } finally {
            ContextUtil.doFinally(transaction);
        }
    }

    @Override
    public List<PollDTO> findInvitedPolls(String userId) {
        TopiaContext transaction = null;
        List<PollDTO> results = null;
        UserAccountDAO userDAO = null;
        try {
            transaction = rootContext.beginTransaction();

            userDAO = PollenModelDAOHelper.getUserAccountDAO(transaction);
            UserAccount user = userDAO.findByTopiaId(userId);

            pollDAO = PollenModelDAOHelper.getPollDAO(transaction);
            List<Poll> polls = pollDAO.findAll();

            Set<Poll> pollEntities = new HashSet<Poll>();
            for (Poll poll : polls) {
                for (VotingList votingList : poll.getVotingList()) {
                    for (PersonToList personToList : votingList
                            .getPollAccountPersonToList()) {
                        if (!personToList.getHasVoted()) {
                            PollAccount pollAccount = personToList
                                    .getPollAccount();
                            if (pollAccount != null
                                    && pollAccount.getEmail() != null
                                    && pollAccount.getEmail().equals(
                                            user.getEmail())) {
                                pollEntities.add(poll);
                            }
                        }
                    }
                }
            }

            converter.setTransaction(transaction);
            results = converter
                    .createPollDTOs(new ArrayList<Poll>(pollEntities));

            transaction.commitTransaction();

            if (log.isDebugEnabled()) {
                log.debug("Entities found: "
                        + ((results == null) ? "null" : results.size()));
            }

            return results;
        } catch (Exception e) {
            ContextUtil.doCatch(e, transaction);
            return null;
        } finally {
            ContextUtil.doFinally(transaction);
        }
    }

    @Override
    public List<PollDTO> findRunningPolls(boolean withEndDate) {
        TopiaContext transaction = null;
        List<PollDTO> results = null;
        List<Poll> pollEntities = null;
        try {
            transaction = rootContext.beginTransaction();

            // requête de sélection des sondages
            if (withEndDate) {
                pollEntities = transaction
                        .find("from "
                                + Poll.class.getName()
                                + " as poll where (poll.endDate is not null and poll.endDate > current_timestamp())"
                                + " and (poll.beginDate is null or poll.beginDate < current_timestamp())");
            } else {
                pollEntities = transaction
                        .find("from "
                                + Poll.class.getName()
                                + " as poll where (poll.endDate is null or poll.endDate > current_timestamp())"
                                + " and (poll.beginDate is null or poll.beginDate < current_timestamp())");
            }

            converter.setTransaction(transaction);
            results = converter.createPollDTOs(pollEntities);

            transaction.commitTransaction();

            if (log.isDebugEnabled()) {
                log.debug("Entities found: "
                        + ((results == null) ? "null" : results.size()));
            }

            return results;
        } catch (Exception e) {
            ContextUtil.doCatch(e, transaction);
            return null;
        } finally {
            ContextUtil.doFinally(transaction);
        }
    }

    @Override
    public List<PollDTO> selectPolls(Map<String, Object> properties) {
        TopiaContext transaction = null;
        List<PollDTO> results = null;
        List<Poll> pollEntities = null;
        try {
            transaction = rootContext.beginTransaction();

            pollDAO = PollenModelDAOHelper.getPollDAO(transaction);

            if (properties == null) {
                pollEntities = pollDAO.findAll();
                if (log.isWarnEnabled()) {
                    log
                            .warn("Attention : tous les sondages ont été sélectionnés !");
                }
            } else {
                pollEntities = pollDAO.findAllByProperties(properties);
            }
            converter.setTransaction(transaction);
            results = converter.createPollDTOs(pollEntities);

            transaction.commitTransaction();

            if (log.isDebugEnabled()) {
                log.debug("Entities found: "
                        + ((results == null) ? "null" : results.size()));
            }

            return results;
        } catch (Exception e) {
            ContextUtil.doCatch(e, transaction);
            return null;
        } finally {
            ContextUtil.doFinally(transaction);
        }
    }

    @Override
    public boolean addVoteToPoll(String pollId, String voteId) {
        TopiaContext transaction = null;
        try {
            transaction = rootContext.beginTransaction();

            pollDAO = PollenModelDAOHelper.getPollDAO(transaction);
            Poll pollEntity = pollDAO.findByTopiaId(pollId);
            VoteDAO voteDAO = PollenModelDAOHelper.getVoteDAO(transaction);
            Vote voteEntity = voteDAO.findByTopiaId(voteId);

            if (log.isDebugEnabled()) {
                log.debug(pollEntity + " " + voteEntity);
            }

            pollEntity.addVote(voteEntity);
            pollEntity.update();

            transaction.commitTransaction();

            if (log.isDebugEnabled()) {
                log.debug("Entity updated: " + pollId);
            }

            return true;
        } catch (Exception e) {
            ContextUtil.doCatch(e, transaction);
            return false;
        } finally {
            ContextUtil.doFinally(transaction);
        }
    }

    @Override
    public PollAccountDTO getRestrictedAccount(String accountUId, PollDTO poll) {
        TopiaContext transaction = null;
        PollAccountDTO result = null;
        try {
            transaction = rootContext.beginTransaction();

            if (log.isInfoEnabled()) {
                log.info("getRestrictedAccount : accountUId = " + accountUId +
                        " _ pollUId = " + poll.getPollId());
            }

            // Use PersonToList association entity to find coherence between
            // the poll and votingId
            List<PersonToList> tmp = transaction.find(
                    "FROM " + PersonToList.class.getName() +
                    " WHERE pollAccount.accountId = :accountUId" +
                    " AND votingList.poll.pollId = :pollUId",
                    "accountUId", accountUId,
                    "pollUId", poll.getPollId());

            if (tmp.size() > 0) {
                PollAccount account = tmp.get(0).getPollAccount();
                DataPollAccountConverter converter =
                        new DataPollAccountConverter();
                converter.setTransaction(transaction);
                result = converter.createPollAccountDTO(account);
            }

        } catch (Exception eee) {
            ContextUtil.doCatch(eee, transaction,
                    "Unable to retrieve restricted account with accountUId = " +
                        accountUId + " and poll with uid = " + poll.getPollId());
        } finally {
            ContextUtil.doFinally(transaction);
        }
        return result;
    }
}