/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.business.converters;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.business.dto.ChoiceDTO;
import org.chorem.pollen.business.dto.PollDTO;
import org.chorem.pollen.business.dto.PreventRuleDTO;
import org.chorem.pollen.business.persistence.Choice;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.PreventRule;
import org.chorem.pollen.business.services.ServiceChoice;
import org.chorem.pollen.business.services.ServiceChoiceImpl;
import org.chorem.pollen.business.services.ServicePreventRule;
import org.chorem.pollen.business.services.ServicePreventRuleImpl;
import org.nuiton.topia.TopiaException;

/**
 * Gestion de la conversion des dtos Poll en entité et vice-versa.
 *
 * @author enema
 * @version $Id: DataPollConverter.java 2910 2010-03-09 15:59:47Z fdesbois $
 */
public class DataPollConverter extends DataConverter {
    DataChoiceConverter choiceConverter = new DataChoiceConverter();
    DataVoteConverter voteConverter = new DataVoteConverter();
    DataCommentConverter commentConverter = new DataCommentConverter();
    DataPreventRuleConverter preventRuleConverter = new DataPreventRuleConverter();
    DataVotingListConverter votingListConverter = new DataVotingListConverter();

    /** log. */
    private static final Log log = LogFactory.getLog(DataPollConverter.class);

    /**
     * Création d'un sondage à partir d'un dto sondage.
     *
     * @param pollDTO le dto sondage
     * @param ePoll l'entité sondage
     */
    public void populatePollEntity(PollDTO pollDTO, Poll ePoll)
            throws TopiaException {
        ePoll.setPollId(pollDTO.getPollId());
        ePoll.setTitle(pollDTO.getTitle());
        ePoll.setDescription(pollDTO.getDescription());
        ePoll.setBeginChoiceDate(pollDTO.getBeginChoiceDate());
        ePoll.setEndChoiceDate(pollDTO.getEndChoiceDate());
        ePoll.setBeginDate(pollDTO.getBeginDate());
        ePoll.setEndDate(pollDTO.getEndDate());
        ePoll.setClosed(pollDTO.isClosed());
        ePoll.setAnonymous(pollDTO.isAnonymous());
        ePoll.setAnonymousVoteAllowed(pollDTO.isAnonymousVoteAllowed());
        ePoll.setPublicResults(pollDTO.isPublicResults());
        ePoll.setContinuousResults(pollDTO.isContinuousResults());
        ePoll.setChoiceAddAllowed(pollDTO.isChoiceAddAllowed());
        ePoll.setMaxChoiceNb(pollDTO.getMaxChoiceNb());

        EnumController enumCtrl = new EnumController(super.transaction);
        enumCtrl.setChoiceType(pollDTO.getChoiceType(), ePoll);
        enumCtrl.setPollType(pollDTO.getPollType(), ePoll);
        enumCtrl.setVoteCounting(pollDTO.getVoteCounting(), ePoll);

        /*if (pollDTO.getCreatorId().length() > 0) {
            PollAccountDAO pollAccountDAO = PollenModelDAOHelper.getPollAccountDAO(super.transaction);
            PollAccount PollAccountEntity = pollAccountDAO.findByTopiaId(pollDTO.getCreatorId());
            ePoll.setCreator(PollAccountEntity);
        }*/
    }

    /**
     * Création d'un dto sondage à partir d'une entité.
     *
     * @param ePoll l'entité sondage
     * @return le dto sondage
     */
    public PollDTO createPollDTO(Poll ePoll) {
        PollDTO pollDTO = new PollDTO();
        pollDTO.setId(ePoll.getTopiaId());
        pollDTO.setPollId(ePoll.getPollId());
        pollDTO.setTitle(ePoll.getTitle());
        pollDTO.setDescription(ePoll.getDescription());
        pollDTO.setBeginChoiceDate(ePoll.getBeginChoiceDate());
        pollDTO.setEndChoiceDate(ePoll.getEndChoiceDate());
        pollDTO.setBeginDate(ePoll.getBeginDate());
        pollDTO.setEndDate(ePoll.getEndDate());
        pollDTO.setAnonymous(ePoll.getAnonymous());
        pollDTO.setAnonymousVoteAllowed(ePoll.getAnonymousVoteAllowed());
        pollDTO.setPublicResults(ePoll.getPublicResults());
        pollDTO.setContinuousResults(ePoll.getContinuousResults());
        pollDTO.setChoiceAddAllowed(ePoll.getChoiceAddAllowed());
        pollDTO.setClosed(ePoll.getClosed());
        pollDTO.setMaxChoiceNb(ePoll.getMaxChoiceNb());
        pollDTO.setChoiceType(EnumController.getChoiceType(ePoll));
        pollDTO.setPollType(EnumController.getPollType(ePoll));
        pollDTO.setVoteCounting(EnumController.getVoteCountingType(ePoll));

        if (ePoll.getCreator() != null) {
            pollDTO.setCreatorId(ePoll.getCreator().getTopiaId());
            pollDTO.setCreatorEmail(ePoll.getCreator().getEmail());
        }
        if (ePoll.getChoice().size() > 0) {
            pollDTO.setChoiceDTOs(choiceConverter.createChoiceDTOs(ePoll
                    .getChoice()));
        }
//        if (ePoll.getVote().size() > 0) {
//            // FD-20100308 Votes are not loaded for an anonymous vote :
//            // see ref ANO #130
//            if (!ePoll.getAnonymous()) {
//                pollDTO.setVoteDTOs(voteConverter.createVoteDTOs(ePoll.getVote()));
//            }
//            pollDTO.setNbVotes(ePoll.getVote().size());
//        }
        if (ePoll.getComment().size() > 0) {
            pollDTO.setCommentDTOs(commentConverter.createCommentDTOs(ePoll
                    .getComment()));
        }
        if (ePoll.getPreventRule().size() > 0) {
            pollDTO.setPreventRuleDTOs(preventRuleConverter
                    .createPreventRuleDTOs(ePoll.getPreventRule()));
        }
        if (ePoll.getVotingList().size() > 0) {
            pollDTO.setVotingListDTOs(votingListConverter
                    .createVotingListDTOs(ePoll.getVotingList()));
        }

        return pollDTO;
    }

    /**
     * Retourne une liste de dtos sondage à partir d'une liste d'entités.
     *
     * @param lPolls liste de entités sondages
     * @return la liste des dtos sondages
     */
    public List<PollDTO> createPollDTOs(List<Poll> lPolls) {
        ArrayList<PollDTO> results = new ArrayList<PollDTO>();
        PollDTO dto;
        for (Poll p : lPolls) {
            dto = this.createPollDTO(p);
            results.add(dto);
        }
        return results;
    }

    /**
     * Persistance des choix d'une entité sondage à partir d'un DTO.
     *
     * @param pollDTO le dto sondage
     * @param ePoll l'entité sondage
     */
    public void persistChoices(PollDTO pollDTO, Poll ePoll)
            throws TopiaException {

        // mise à jour ou création des choix
        ServiceChoice sChoice = new ServiceChoiceImpl();
        for (ChoiceDTO choiceDTO : pollDTO.getChoiceDTOs()) {
            boolean updated = sChoice.updateChoice(choiceDTO);
            if (!updated) {
                choiceDTO.setId(sChoice.createChoice(choiceDTO));
            }

            if (log.isDebugEnabled()) {
                log.debug("Choice " + choiceDTO.getName() + " ("
                        + choiceDTO.getId() + ") updated: " + updated);
            }
        }

        // suppression des choix
        boolean validChoice = false;
        Iterator<Choice> itChoice = ePoll.getChoice().iterator();
        while (itChoice.hasNext()) {
            Choice currentChoice = itChoice.next();
            for (ChoiceDTO choiceDTO : pollDTO.getChoiceDTOs()) {
                if (currentChoice.getTopiaId().equals(choiceDTO.getId())) {
                    validChoice = true;
                }
            }
            if (!validChoice) {
                sChoice.deleteChoice(currentChoice.getTopiaId());
            }
            validChoice = false;
        }
    }

    /**
     * Persistance des règles de notification d'une entité sondage à partir d'un
     * DTO.
     *
     * @param pollDTO le dto sondage
     * @param ePoll l'entité sondage
     */
    public void persistPreventRules(PollDTO pollDTO, Poll ePoll)
            throws TopiaException {

        // mise à jour ou création des règles de notification
        ServicePreventRule sPreventRule = new ServicePreventRuleImpl();
        for (PreventRuleDTO preventRuleDTO : pollDTO.getPreventRuleDTOs()) {
            boolean updated = sPreventRule.updatePreventRule(preventRuleDTO);
            if (!updated) {
                preventRuleDTO.setId(sPreventRule
                        .createPreventRule(preventRuleDTO));
            }

            if (log.isDebugEnabled()) {
                log.debug("PreventRule (" + preventRuleDTO.getId()
                        + ") updated: " + updated);
            }
        }

        // suppression des règles de notification
        boolean validRule = false;
        Iterator<PreventRule> it = ePoll.getPreventRule().iterator();
        while (it.hasNext()) {
            PreventRule currentRule = it.next();
            for (PreventRuleDTO preventRuleDTO : pollDTO.getPreventRuleDTOs()) {
                if (currentRule.getTopiaId().equals(preventRuleDTO.getId())) {
                    validRule = true;
                }
            }
            if (!validRule) {
                sPreventRule.deletePreventRule(currentRule.getTopiaId());
            }
            validRule = false;
        }
    }
}
