package org.chorem.lima.web.service;

/*
 * #%L
 * Lima :: web
 * %%
 * Copyright (C) 2008 - 2014 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import org.apache.commons.io.Charsets;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.lima.business.LimaServiceConfig;
import org.chorem.lima.business.api.AccountService;
import org.chorem.lima.business.utils.DocumentsEnum;
import org.chorem.lima.business.utils.FormatsEnum;
import org.chorem.lima.entity.Account;
import org.chorem.lima.service.LimaServiceFactory;
import org.chorem.lima.web.action.ReportBuilder;
import org.eclipse.jetty.server.Server;
import org.eclipse.jetty.servlet.ServletContextHandler;
import org.eclipse.jetty.servlet.ServletHolder;
import org.hibernate.exception.GenericJDBCException;
import org.nuiton.util.FileUtil;
import org.nuiton.util.Resource;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;
import java.net.URLDecoder;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import static org.nuiton.i18n.I18n.t;

public class HttpServerService {

    protected static final Log log = LogFactory.getLog(HttpServerService.class);

    private DocumentService documentService;

    private AccountService accountService;

    protected Server server;

    //protected String path;

    protected int port;

    private static final SimpleDateFormat DATEFORMAT = new SimpleDateFormat("yyyy-MM-dd");

    protected ReportBuilder reportBuilder;

    public HttpServerService() {
        //File reportDir = (File) props.get("REPORTS_MODEL_DIR");
        //path = reportDir.getAbsolutePath();
        port = LimaServiceConfig.getInstance().getHttpPort();
        accountService = LimaServiceFactory.getService(AccountService.class);
        documentService = new DocumentService();
        reportBuilder = new ReportBuilder();
    }

    /** start the server */
    public void start() {
        if (server == null) {
            try {
                LimaServiceConfig config = LimaServiceConfig.getInstance();
                File reportDir = config.getReportsModelDir();

                FileUtil.createDirectoryIfNecessary(reportDir);

                server = new Server(port);

                ServletContextHandler context = new ServletContextHandler(ServletContextHandler.SESSIONS);
                context.setContextPath("/");
                context.addServlet(new ServletHolder(new MainServlet()),"/");
                server.setHandler(context);
                server.start();

                if (log.isInfoEnabled()) {
                    log.info("Web server are running on port: " + port);
                }

            } catch (Exception eee) {
                log.error("error while booting http server", eee);
            }
        }
    }

    public int getHttpPort() {
        return port;
    }

    /** 
     * This servlet send a static html page.
     */
    public class MainServlet extends HttpServlet {
        private static final long serialVersionUID = 1L;

        private String serverAddressConfig = LimaServiceConfig.getInstance().getAddressServer();

        @Override
        protected void doGet(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {

            try {

                if (log.isDebugEnabled()) {
                    log.debug("doGet");
                }

                //get all params
                String imageParam = req.getParameter("img");
                String model = req.getParameter("model");
                String beginDate = req.getParameter("beginDate");
                String endDate = req.getParameter("endDate");
                String account = (StringUtils.isBlank(req.getParameter("account")) ? null : URLDecoder.decode(req.getParameter("account"), "UTF-8"));

                //if image
                if (imageParam != null) {
                    URL image = Resource.getURLOrNull("images/" + imageParam);
                    if (image != null) {
                        resp.setContentType("image/png");
                        InputStream in = image.openStream();
                        OutputStream out = resp.getOutputStream();
                        IOUtils.copy(in, out);
                    }
                }
                else if (model != null && beginDate != null && endDate != null) {
                    String accountReport = null;
                    String financialReport = null;

                    //create docs
                    try {
                        Date beginDateFormat = DATEFORMAT.parse(beginDate);
                        Date endDateFormat = DATEFORMAT.parse(endDate);

                        switch (DocumentsEnum.valueOfLink(model)) {
                            case BALANCE:
                                documentService.createBalanceDocuments(beginDateFormat, endDateFormat, null);
                                break;
                            case ACCOUNT:
                                accountReport = documentService.createAccountDocument(
                                        beginDateFormat, endDateFormat, account);
                                break;
                            case ENTRY_BOOKS:
                                documentService.createEntryBooksDocuments(beginDateFormat, endDateFormat, null);
                                break;
                            case GENERAL_ENTRY_BOOK:
                                documentService.createGeneralEntryBooksDocuments(beginDateFormat, endDateFormat);
                                break;
                            case FINANCIAL_STATEMENT:
                                financialReport = documentService.createFinancialStatementsDocuments(
                                        beginDateFormat, endDateFormat);
                                break;
                            case LEDGER:
                                documentService.createLedgerDocuments(beginDateFormat, endDateFormat);
                                break;
                        }
                    } catch (ParseException eeePE) {
                        log.error("Can't parse date", eeePE);
                    }

                    if ( (accountReport != null || financialReport != null)) {
                        String report;
                        resp.setContentType(".html");
                        OutputStream out = resp.getOutputStream();
                        if (accountReport != null) {
                            report = accountReport;
                        } else {
                            report = financialReport;
                        }
                        IOUtils.write(report, out, Charsets.UTF_8);
                    } else {
                        LimaServiceConfig config = LimaServiceConfig.getInstance();
                        File reportDir = config.getReportsModelDir();
                        String path = reportDir.getAbsolutePath();
                        URL doc = new URL("file:" + path + File.separator + model + ".pdf");
                        if (log.isDebugEnabled()) {
                            log.debug("file:" + path + File.separator + model + ".pdf");
                        }
                        resp.setContentType(".pdf");
                        OutputStream out = resp.getOutputStream();
                        InputStream in = doc.openStream();
                        IOUtils.copy(in, out);
                    }
                }
                // else return home html
                else {

                    if (log.isDebugEnabled()) {
                        log.debug("Home html");
                    }

                    resp.setContentType(FormatsEnum.HTML.getMimeType());
                    StringBuilder pageContent = new StringBuilder();
                    String serverAddress = "http://";

                    log.info("Page load");

                    /** create server Address : static config if exist or dynamical adress */
                    if (serverAddressConfig.equals("")) {
                        serverAddress += req.getServerName()
                                        + ":" + req.getServerPort();
                    } else {
                        serverAddress += serverAddressConfig + ":"
                                        + LimaServiceConfig.getInstance().getHttpPort();
                    }

                    Calendar calendar = Calendar.getInstance();
                    Date endDatePicker = calendar.getTime();
                    Date beginDatePicker = calendar.getTime();
                    beginDatePicker =
                            DateUtils.truncate(beginDatePicker, Calendar.YEAR);

                    pageContent.append("<!DOCTYPE html>\n" + "<html lang=\"fr\">\n"
                            + "<head>\n"
                                + "<script type=\"text/javascript\">" +
                            "        function showAccountInput(){" +
                            "          var request = document.getElementById('model');" +
                            "          var requestValue = request == null ? '': request.value;" +
                            "          var isDocumentAccount = requestValue === 'lima_account'; " +
                            "          if (document.getElementById('accountField') != null) {" +
                            "            if (isDocumentAccount) {" +
                            "              document.getElementById('accountField').style.display = 'block';" +
                            "            } else {" +
                            "              document.getElementById('accountField').style.display = 'none';" +
                            "            }" +
                            "          }" +
                            "        }" +
                            "      showAccountInput();" +
                            "      </script>"
                                + "<title>LIMA Documents Report</title>\n" + "<style type=\"text/css\">" + "body { font: 14px sans-serif; }" + "h1 { font: 20px sans-serif; text-align: center; }" + "table.padding td {padding-right:20px;}" + ".tdright {text-align: right;}" + "</style>"
                            + "</head>\n"
                            + "<body>\n" + "<table class=\"padding\">\n" + "<tr><td><img src=\"?img=puzzle_icon_mini.png\"/></td>\n" + "<td><h1>LIMA Documents Report</h1></td></tr>\n" + "<tr><td class=\"tdright\"><img src=\"?img=identity.png\"/></td>\n"
                                + "<td rowspan=3><form method=GET action=")
                            .append(serverAddress).append(">\n")
                            .append(t("lima-business.document.date.begin"))
                            .append("<input value=\"")
                            .append(DATEFORMAT.format(beginDatePicker))
                            .append("\" type=\"date\" name=\"beginDate\">\n")
                            .append(t("lima-business.document.date.end"))
                            .append("<input value=\"")
                            .append(DATEFORMAT.format(endDatePicker))
                            .append("\" type=\"date\" name=\"endDate\">\n<br/><br/>");
                            //+ "Format : <select name=\"format\">");


                    pageContent.append(/*"</select>\n"
                                       + */"Documents : <select id=\"model\" name=\"model\" onchange='showAccountInput()'>");

                    for (DocumentsEnum documentsEnum : DocumentsEnum.values()) {
                        pageContent.append("<option value=\"")
                                .append(documentsEnum.getFileName())
                                .append("\">")
                                .append(documentsEnum.getDescription())
                                .append("</option>\n");
                    }

                    List<Account> accounts = accountService.getAllAccounts();

                    StringBuilder optionAccounts = new StringBuilder();
                    for (Account account1:accounts) {
                        optionAccounts.append("<option value=\"" + account1.getTopiaId() + "\">");
                        optionAccounts.append(account1.getAccountNumber() + " - " + account1.getLabel());
                        optionAccounts.append("</option>\n");
                    }

                    pageContent.append("</select>"
                                       + "<div id='accountField'>Compte : <select id=\"account\" name=\"account\">"
                                       +   optionAccounts.toString()
                                       + "</select></div>"
                                       + "<input type=\"submit\">\n"
                                       + "</form></td>"
                                       + "<tr><td class=\"tdright\"><img src=\"?img=entries.png\"/></td></tr>\n"
                                       + "<tr><td class=\"tdright\"><img src=\"?img=accounts.png\"/></td></tr>\n"
                                       + "<tr><td class=\"tdright\"><img src=\"?img=entrybooks.png\"/></td>\n"
                                       + "<tr><td class=\"tdright\"><img src=\"?img=fiscalperiods.png\"/></td></tr>\n"
                                       + "</table>\n"
                                       + "<script type=\"text/javascript\">"
                                       + "  showAccountInput();"
                                       + "</script>"
                                       + "</body>\n"
                                       + "</html>");

                    resp.getWriter().write(pageContent.toString());

                }
            } catch (GenericJDBCException eee) {
                log.error("Can't read db", eee);
            }
        }

        @Override
        protected void doPost(HttpServletRequest req, HttpServletResponse resp)
                throws ServletException, IOException {
            doGet(req, resp);
        }
    }
}
