/*
 * #%L
 * $Id: JTimerConfig.java 2867 2012-05-18 17:01:18Z echatellier $
 * $HeadURL: http://svn.chorem.org/svn/jtimer/tags/jtimer-1.4.2/src/main/java/org/chorem/jtimer/JTimerConfig.java $
 * %%
 * Copyright (C) 2012 Codelutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package org.chorem.jtimer;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.ApplicationConfig;
import org.nuiton.util.ArgumentsParserException;

/**
 * JTimer configuration.
 * 
 * @author chatellier
 * @version $Revision: 2867 $
 * 
 * Last update : $Date: 2012-05-18 19:01:18 +0200 (Fri, 18 May 2012) $
 * By : $Author: echatellier $
 */
public class JTimerConfig {

    private static final Log log = LogFactory.getLog(JTimerConfig.class);

    /** To tag action to be run before ui start. */
    protected static final int STEP_BEFORE_UI = 0;
    /** To tag action to be run after ui start. */
    protected static final int STEP_AFTER_UI = 1;

    protected ApplicationConfig appConfig;

    public JTimerConfig() {
        appConfig = new ApplicationConfig();
        for (JTimerOption o : JTimerOption.values()) {
            if (o.defaultValue != null) {
                appConfig.setDefaultOption(o.key, o.defaultValue);
            }
        }
        for (JTimerAction a : JTimerAction.values()) {
            for (String alias : a.aliases) {
                appConfig.addActionAlias(alias, a.action);
            }
        }
    }

    public void parse(String... args) {
        try {
            appConfig.parse(args);
        } catch (ArgumentsParserException ex) {
            if (log.isErrorEnabled()) {
                log.error("Can't parse configuration", ex);
            }
        }
    }

    /**
     * Put new custom object into app config context.
     *
     * @param o object value
     */
    public void putObject(Object o) {
        appConfig.putObject(o);
    }

    /**
     * Run command line actions for step.
     * 
     * @param step action step
     */
    public void doAction(int step) {
        try {
            appConfig.doAction(step);
        } catch (Exception ex) {
            if (log.isErrorEnabled()) {
                log.error("Can't run command line actions", ex);
            }
        }
    }

    public Class getServiceClass() {
        Class serviceClass = null;
        try {
            serviceClass = appConfig.getOptionAsClass(JTimerOption.SERVICE_CLASS.key);
        } catch (Exception ex) {
            if (log.isWarnEnabled()) {
                log.warn("Can't find service class implementation, sync will not work", ex);
            }
        }
        return serviceClass;
    }

    public String getServiceEndpoint() {
        return appConfig.getOption(JTimerOption.SERVICE_ENDPOINT.key);
    }

    public String getServiceResource() {
        return appConfig.getOption(JTimerOption.SERVICE_RESOURCE.key);
    }

    public Class getIOSaverClass() {
        return appConfig.getOptionAsClass(JTimerOption.IO_SAVER_CLASS.key);
    }

    public String getIOSaverDirectory() {
        return appConfig.getOption(JTimerOption.IO_SAVER_DIRECTORY.key);
    }

    /**
     * Return auto save delay in seconds.
     * 
     * @return auto save delay
     */
    public long getIOSaverAutoSaveDelay() {
        return appConfig.getOptionAsLong(JTimerOption.IO_SAVER_AUTOSAVEDELAY.key);
    }

    /**
     * Return user idle time threshold in seconds.
     * 
     * @return idle time threshold
     */
    public long getIdleTime() {
        return appConfig.getOptionAsLong(JTimerOption.UI_IDLE_TIME.key);
    }

    /**
     * Show closed project on main ui tree.
     * 
     * @return show closed project
     */
    public boolean isShowClosed() {
        return appConfig.getOptionAsBoolean(JTimerOption.UI_SHOW_CLOSED.key);
    }

    /**
     * Set show closed option.
     * 
     * @param showClosed show closed
     */
    public void setShowClosed(boolean showClosed) {
        appConfig.setOption(JTimerOption.UI_SHOW_CLOSED.key, String.valueOf(showClosed));
        appConfig.saveForUser();
    }

    /**
     * Get close to systray option.
     * 
     * @return close to systray
     */
    public boolean isCloseToSystray() {
        return appConfig.getOptionAsBoolean(JTimerOption.UI_CLOSE_TO_SYSTRAY.key);
    }

    /**
     * Set close to systray option
     * 
     * @param closeToSystray close to systray
     */
    public void setCloseToSystray(boolean closeToSystray) {
        appConfig.setOption(JTimerOption.UI_CLOSE_TO_SYSTRAY.key, String.valueOf(closeToSystray));
        appConfig.saveForUser();
    }

    /**
     * Return first day of week.
     * Default to {@code -1} (no preference).
     * 
     * @return first day of week in report
     */
    public int getReportFirstDayOfWeek() {
        return appConfig.getOptionAsInt(JTimerOption.UI_REPORT_FIRSTDAYOFWEEK.key);
    }

    /**
     * Report first day of week.
     * 
     * @param firstDayOfWeek first day of week
     */
    public void setReportFirstDayOfWeek(int firstDayOfWeek) {
        appConfig.setOption(JTimerOption.UI_REPORT_FIRSTDAYOFWEEK.key, String.valueOf(firstDayOfWeek));
        appConfig.saveForUser();
    }

    protected enum JTimerOption {
        CONFIG_FILENAME(ApplicationConfig.CONFIG_FILE_NAME, "jtimer.properties"),
        SERVICE_CLASS("jtimer.service.class", null),
        SERVICE_ENDPOINT("jtimer.service.endpoint", null),
        SERVICE_RESOURCE("jtimer.service.resource", "JTimer"),
        IO_SAVER_CLASS("jtimer.io.saver.class", "org.chorem.jtimer.io.GTimerIncrementalSaver"),
        IO_SAVER_DIRECTORY("jtimer.io.saver.directory", "${user.home}/.gtimer"),
        IO_SAVER_AUTOSAVEDELAY("jtimer.io.saver.autosavedelay", "300"),
        UI_IDLE_TIME("jtimer.ui.idletime", "300"),
        UI_SHOW_CLOSED("jtimer.ui.showclosed", "false"),
        UI_CLOSE_TO_SYSTRAY("jtimer.ui.closetosystray", "true"),
        UI_REPORT_FIRSTDAYOFWEEK("jtimer.ui.report.firstdayofweek", "0");

        protected String key;
        protected String defaultValue;

        private JTimerOption(String key, String defaultValue) {
            this.key = key;
            this.defaultValue = defaultValue;
        }
    }
    
    protected enum JTimerAction {
    	HELP("Show help", JTimerActions.class.getName() + "#help", "--help", "-h"),
    	VERSION("Display application version", JTimerActions.class.getName() + "#version", "--version", "-v"),
    	START("Start task", JTimerActions.class.getName() + "#start", "--start", "-s");
    	
    	protected String description;
    	protected String action;
    	protected String[] aliases;
    	
    	private JTimerAction(String description, String action, String... aliases) {
    		this.description = description;
    		this.action = action;
    		this.aliases = aliases;
    	}
    }
}
