/* *##%
 * Copyright (C) 2009 Code Lutin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *##%*/

package org.chorem.jtimer.ui.widget;

import java.awt.Dimension;
import java.awt.Frame;
import java.awt.GraphicsEnvironment;
import java.awt.Image;
import java.awt.Point;
import java.awt.TrayIcon;
import java.awt.Window;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;

import javax.swing.JDialog;
import javax.swing.JPopupMenu;
import javax.swing.JWindow;
import javax.swing.RootPaneContainer;
import javax.swing.event.PopupMenuEvent;
import javax.swing.event.PopupMenuListener;

/**
 * JPopupMenu compatible TrayIcon based on Alexander Potochkin's JXTrayIcon
 * (http://weblogs.java.net/blog/alexfromsun/archive/2008/02/jtrayicon_updat.html)
 * but uses a JWindow instead of a JDialog to workaround some bugs on linux.
 * 
 * Use code based from, with some modifications :
 * https://fishfarm.dev.java.net/source/browse/fishfarm/trunk/FishFarm/src/net/java/fishfarm/ui/JPopupTrayIcon.java?rev=198&view=markup
 * 
 * @author Michael Bien
 * @author Chatellier Eric
 */
public class JPopupTrayIcon extends TrayIcon implements MouseListener,
        PopupMenuListener {

    private JPopupMenu menu;

    private Window window;

    private static final boolean IS_WINDOWS = System.getProperty("os.name")
            .toLowerCase().contains("windows");

    public JPopupTrayIcon(Image image, String tooltip, JPopupMenu popup) {
        super(image, tooltip);
        addMouseListener(this);
        setJPopupMenu(popup);
    }

    protected void showJPopupMenu(MouseEvent e) {
        if (e.isPopupTrigger() && menu != null) {
            if (window == null) {

                if (IS_WINDOWS) {
                    window = new JDialog((Frame) null);
                    ((JDialog) window).setUndecorated(true);
                } else {
                    window = new JWindow((Frame) null);
                }
                window.setAlwaysOnTop(true);
                Dimension size = menu.getPreferredSize();

                Point centerPoint = GraphicsEnvironment
                        .getLocalGraphicsEnvironment().getCenterPoint();
                if (e.getY() > centerPoint.getY()) {
                    window.setLocation(e.getX(), e.getY() - size.height);
                } else {
                    window.setLocation(e.getX(), e.getY());
                }

                window.setVisible(true);

                menu.show(((RootPaneContainer) window).getContentPane(), 0, 0);

                // popup works only for focused windows
                window.toFront();
            }
        }
    }

    public JPopupMenu getJPopupMenu() {
        return menu;
    }

    public void setJPopupMenu(JPopupMenu menu) {
        if (this.menu != null) {
            this.menu.removePopupMenuListener(this);
        }
        this.menu = menu;
        menu.addPopupMenuListener(this);
    }

    /*
     * @see javax.swing.event.PopupMenuListener#popupMenuWillBecomeVisible(javax.swing.event.PopupMenuEvent)
     */
    @Override
    public void popupMenuWillBecomeVisible(PopupMenuEvent e) {

    }

    /*
     * @see javax.swing.event.PopupMenuListener#popupMenuWillBecomeInvisible(javax.swing.event.PopupMenuEvent)
     */
    @Override
    public void popupMenuWillBecomeInvisible(PopupMenuEvent e) {
        if (window != null) {
            window.dispose();
            window = null;
        }
    }

    /*
     * @see javax.swing.event.PopupMenuListener#popupMenuCanceled(javax.swing.event.PopupMenuEvent)
     */
    @Override
    public void popupMenuCanceled(PopupMenuEvent e) {
        if (window != null) {
            window.dispose();
            window = null;
        }
    }

    /*
     * @see java.awt.event.MouseListener#mouseClicked(java.awt.event.MouseEvent)
     */
    @Override
    public void mouseClicked(MouseEvent e) {

    }

    /*
     * @see java.awt.event.MouseListener#mouseEntered(java.awt.event.MouseEvent)
     */
    @Override
    public void mouseEntered(MouseEvent e) {

    }

    /*
     * @see java.awt.event.MouseListener#mouseExited(java.awt.event.MouseEvent)
     */
    @Override
    public void mouseExited(MouseEvent e) {

    }

    /*
     * @see java.awt.event.MouseListener#mousePressed(java.awt.event.MouseEvent)
     */
    @Override
    public void mousePressed(MouseEvent e) {
        showJPopupMenu(e);

    }

    /*
     * @see java.awt.event.MouseListener#mouseReleased(java.awt.event.MouseEvent)
     */
    @Override
    public void mouseReleased(MouseEvent e) {
        showJPopupMenu(e);
    }
}
