/* *##%
 * Copyright (C) 2009 Code Lutin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *##%*/

package org.chorem.jtimer.io;

import java.io.File;
import java.io.IOException;
import java.util.TimerTask;

import org.apache.commons.io.FileUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Abstract saver class.
 * 
 * Contains methods that can be used by various savers.
 * 
 * @author chatellier
 * @version $Revision: 2650 $
 * 
 * Last update : $Date: 2009-08-21 11:41:24 +0200 (ven., 21 août 2009) $
 * By : $Author: echatellier $
 */
public abstract class AbstractSaver extends TimerTask implements Saver {

    /** log. */
    private static Log log = LogFactory.getLog(GTimerIncrementalSaver.class);

    /** Backup file extension. */
    public static final String BACKUP_EXTENSION = ".tmp";

    /**
     * Make to backup of file if exists.
     * 
     * Copy file to filename + ".tmp"
     * 
     * @param file file to backup
     * @return backup file or {@code null} if input file doesn't exist
     * @throws IOException 
     */
    protected File makeBackupFile(File file) throws IOException {

        // if not exist, don't do anything
        if (!file.exists()) {
            return null;
        }

        File backupFile = new File(file.getAbsoluteFile() + BACKUP_EXTENSION);

        if (log.isDebugEnabled()) {
            log.debug("Backuping file " + file.getName() + " to "
                    + backupFile.getName());
        }

        backupFile.delete();
        FileUtils.copyFile(file, backupFile);
        return backupFile;
    }

    /**
     * Rename backup file to original file name;
     * 
     * @param backupFile backup file
     */
    protected boolean restoreBackupFile(File backupFile) {

        String fileName = backupFile.getAbsolutePath();
        if (!fileName.endsWith(BACKUP_EXTENSION)) {
            throw new IllegalArgumentException("Not a valid backup file"
                    + backupFile);
        }

        fileName = fileName.substring(0, fileName.length()
                - BACKUP_EXTENSION.length());
        File file = new File(fileName);

        if (log.isDebugEnabled()) {
            log.debug("Renaming " + backupFile.getName() + " to "
                    + file.getName());
        }

        //file.delete();
        boolean done = backupFile.renameTo(file);
        return done;
    }

    /**
     * Delete backup file.
     * 
     * This function NEVER throw IOException.
     * 
     * @param backupFile backup file (can be {@code null})
     */
    protected void deleteBackupFile(File backupFile) {
        if (backupFile != null) {
            backupFile.delete();
        }
    }
}
