/* *##%
 * Copyright (C) 2007,2009 Code Lutin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *##%*/

package org.chorem.jtimer.ui;

import java.awt.GridLayout;
import java.util.Collection;
import java.util.Date;
import java.util.List;

import javax.swing.BorderFactory;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SwingConstants;

import org.apache.commons.lang.time.DurationFormatUtils;
import org.chorem.jtimer.data.DataEventListener;
import org.chorem.jtimer.data.TimerDataManager;
import org.chorem.jtimer.entities.TimerProject;
import org.chorem.jtimer.entities.TimerTask;
import org.chorem.jtimer.entities.TimerTaskHelper;
import org.jdesktop.application.Application;
import org.jdesktop.application.ApplicationContext;
import org.jdesktop.application.ResourceManager;
import org.jdesktop.application.ResourceMap;

/**
 * StatusBar.
 * 
 * @author chatellier
 * @version $Revision: 2630 $
 * 
 * Last update : $Date: 2008-06-17 10:50:40 +0200 (mar., 17 juin 2008)
 * $ By : $Author: echatellier $
 */
public class StatusBar extends JPanel implements DataEventListener {

    /** VersionUID */
    private static final long serialVersionUID = -1283108564414026297L;

    /** I18n resources map */
    protected ResourceMap resourceMap;

    /** DataManager */
    protected TimerDataManager dataManager;

    /** Message label */
    protected JLabel lblMessage;

    /** Time label */
    protected JLabel lblTime;

    /**
     * Reference how many tasks are running
     */
    protected int nbTasksRunning = 0;

    /**
     * Constucteur.
     * 
     * @param application parent application
     * @param dataManager data manager
     */
    public StatusBar(Application application, TimerDataManager dataManager) {
        // set layout
        super(new GridLayout());

        // init resources map
        ApplicationContext ctxt = application.getContext();
        ResourceManager mgr = ctxt.getResourceManager();
        resourceMap = mgr.getResourceMap(StatusBar.class);

        // message label
        lblMessage = new JLabel(resourceMap.getString("loadingMessage"));
        lblMessage.setBorder(BorderFactory.createEtchedBorder());
        this.add(lblMessage);

        // time label
        lblTime = new JLabel(resourceMap.getString("todayTotalMessage", "--"));
        lblTime.setBorder(BorderFactory.createEtchedBorder());
        lblTime.setHorizontalAlignment(SwingConstants.CENTER);
        this.add(lblTime);

        // save tdm
        this.dataManager = dataManager;
    }

    /**
     * Udpate today time in status bar.
     */
    protected void updateTodayTime() {
        // refresh time
        long duration = 0L;
        for (TimerProject p : dataManager.getProjectsList()) {
            duration += TimerTaskHelper.getTotalTime(p, new Date());
        }
        lblTime.setText(resourceMap
                .getString("todayTotalMessage", DurationFormatUtils
                        .formatDuration(duration * 1000, "HH:mm:ss")));
    }

    /*
     * @see org.chorem.jtimer.data.event.DataEventListener#addProject(org.chorem.jtimer.entities.TimerProject)
     */
    @Override
    public void addProject(TimerProject project) {
        // can happen for synchronized project
        updateTodayTime();
    }

    /*
     * @see org.chorem.jtimer.data.event.DataEventListener#addTask(org.chorem.jtimer.entities.TimerTask)
     */
    @Override
    public void addTask(TimerTask task) {
        // can happen for synchronized project
        updateTodayTime();
    }

    /*
     * @see org.chorem.jtimer.data.event.DataEventListener#dataLoaded(java.util.Collection)
     */
    @Override
    public void dataLoaded(Collection<TimerProject> projects) {
        lblMessage.setText(resourceMap.getString("idleMessage"));

        // and call modifyTask
        modifyTask(null);
    }

    /*
     * @see org.chorem.jtimer.data.DataEventListener#preDeleteProject(org.chorem.jtimer.entities.TimerProject)
     */
    @Override
    public void preDeleteProject(TimerProject project) {

    }

    /*
     * @see org.chorem.jtimer.data.DataEventListener#preDeleteTask(org.chorem.jtimer.entities.TimerTask)
     */
    @Override
    public void preDeleteTask(TimerTask task) {

    }

    /*
     * @see org.chorem.jtimer.data.event.DataEventListener#deleteProject(org.chorem.jtimer.entities.TimerProject)
     */
    @Override
    public void deleteProject(TimerProject project) {
        updateTodayTime();
    }

    /*
     * @see org.chorem.jtimer.data.event.DataEventListener#deleteTask(org.chorem.jtimer.entities.TimerTask)
     */
    @Override
    public void deleteTask(TimerTask task) {
        updateTodayTime();
    }

    /*
     * @see org.chorem.jtimer.data.event.DataEventListener#modifyProject(org.chorem.jtimer.entities.TimerProject)
     */
    @Override
    public void modifyProject(TimerProject project) {

    }

    /*
     * @see org.chorem.jtimer.event.DataEventListener#setTaskTime(org.chorem.jtimer.entities.TimerTask, java.util.Date, java.lang.Long)
     */
    @Override
    public void setTaskTime(TimerTask task, Date date, Long time) {
        modifyTask(task);
    }

    /*
     * @see org.chorem.jtimer.event.DataEventListener#setAnnotation(org.chorem.jtimer.entities.TimerTask, java.util.Date, java.lang.String)
     */
    @Override
    public void setAnnotation(TimerTask task, Date date, String annotation) {

    }

    /*
     * @see org.chorem.jtimer.data.event.DataEventListener#modifyTask(org.chorem.jtimer.entities.TimerTask)
     */
    @Override
    public void modifyTask(TimerTask task) {
        updateTodayTime();
    }

    /*
     * @see org.chorem.jtimer.data.event.DataEventListener#postChangeClosedState(org.chorem.jtimer.entities.TimerTask)
     */
    @Override
    public void postChangeClosedState(TimerTask task) {

    }

    /*
     * @see org.chorem.jtimer.data.event.DataEventListener#preChangeClosedState(org.chorem.jtimer.entities.TimerTask)
     */
    @Override
    public void preChangeClosedState(TimerTask task) {

    }

    /*
     * @see org.chorem.jtimer.data.event.DataEventListener#startTask(org.chorem.jtimer.entities.TimerTask)
     */
    @Override
    public void startTask(TimerTask task) {
        startStopTask(task, true);

    }

    /*
     * @see org.chorem.jtimer.data.event.DataEventListener#stopTask(org.chorem.jtimer.entities.TimerTask)
     */
    @Override
    public void stopTask(TimerTask task) {
        startStopTask(task, false);

    }

    /**
     * Start and stop task common code.
     * 
     * @param task task
     * @param start start(true) or stop (false)
     */
    protected void startStopTask(TimerTask task, boolean start) {

        // increment or decrement task
        if (start) {
            ++nbTasksRunning;
        } else {
            --nbTasksRunning;
        }

        // display message
        String message = null;
        if (nbTasksRunning == 0) {
            message = resourceMap.getString("idleMessage");
        } else {
            if (nbTasksRunning == 1) {
                message = resourceMap.getString("runningTaskMessage", Integer
                        .valueOf(nbTasksRunning));
            } else {
                message = resourceMap.getString("runningTasksMessage", Integer
                        .valueOf(nbTasksRunning));
            }
        }
        lblMessage.setText(message);
    }

    /*
     * @see org.chorem.jtimer.data.event.DataEventListener#postMoveTask(org.chorem.jtimer.entities.TimerTask)
     */
    @Override
    public void moveTask(TimerTask task) {

    }

    /*
     * @see org.chorem.jtimer.data.event.DataEventListener#preMoveTask(org.chorem.jtimer.entities.TimerTask)
     */
    @Override
    public void preMoveTask(TimerTask task) {

    }

    /*
     * @see org.chorem.jtimer.data.DataEventListener#postMergeTasks(org.chorem.jtimer.entities.TimerTask, java.util.List)
     */
    @Override
    public void postMergeTasks(TimerTask destinationTask,
            List<TimerTask> otherTasks) {

    }

    /*
     * @see org.chorem.jtimer.data.DataEventListener#preMergeTasks(org.chorem.jtimer.entities.TimerTask, java.util.List)
     */
    @Override
    public void preMergeTasks(TimerTask destinationTask,
            List<TimerTask> otherTasks) {

    }
}
