/* *##%
 * Copyright (C) 2007,2009 Code Lutin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *##%*/

package org.chorem.jtimer.ui.tasks;

import java.util.Calendar;
import java.util.Date;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.jtimer.JTimer;
import org.chorem.jtimer.JTimerFactory;
import org.chorem.jtimer.data.TimerDataManager;
import org.chorem.jtimer.entities.TimerTask;
import org.chorem.jtimer.ui.system.SystemInfo;
import org.chorem.jtimer.ui.system.SystemInfoFactory;
import org.chorem.jtimer.ui.system.UnsupportedSystemInfoException;
import org.jdesktop.application.Task;

/**
 * RunTaskJob.
 * 
 * Notify every second the core controler.
 * 
 * @author chatellier
 * @version $Revision: 2518 $
 * 
 * Last update : $Date: 2009-05-22 13:17:51 +0200 (ven. 22 mai 2009) $
 * By : $Author: chatellier $
 */
public class RunTaskJob extends Task<Void, Long> {

    /** log */
    private static Log log = LogFactory.getLog(RunTaskJob.class);

    /** Parent app reference */
    protected JTimer parentApp;

    /** The task to manage */
    protected TimerTask managedTask;

    /** Data manager */
    protected TimerDataManager dataManager;

    /** System information (idle...) */
    protected SystemInfo systemInfo;

    /** The today total task time at start time */
    protected long taskTimingBeforeStartInMs;

    /** Calendar instance when timing start */
    protected Calendar taskStartCalendar;

    /**
     * Offset to adjust time. Include : - Task idle time (task was running but
     * time not counted) - Manual user incremented time
     */
    protected long offsetTimeInMs;

    /** Last publish time to detect hibernate */
    protected long lastPublishTimestamp;

    /** Want to stop flag */
    protected Boolean bWantToStop;

    /** Option after idle detect */
    public static final int REVERT = 0;
    /** Option after idle detect */
    public static final int CONTINUE = 1;
    /** Option after idle detect */
    public static final int RESUME = 2;

    /**
     * Constructor.
     * 
     * Take the task to manage
     * 
     * @param parentApp parent application
     * @param managedTask task
     * @param dataManager data
     */
    public RunTaskJob(JTimer parentApp, TimerTask managedTask,
            TimerDataManager dataManager) {
        super(parentApp);
        this.parentApp = parentApp;

        // init with False
        bWantToStop = Boolean.FALSE;

        // save vars
        this.managedTask = managedTask;
        this.dataManager = dataManager;

        // init system info to get idleTime
        try {
            systemInfo = SystemInfoFactory.getSystemInfo();
        } catch (UnsupportedSystemInfoException e) {
            if (log.isErrorEnabled()) {
                log.error("Can't get system info", e);
            }
            systemInfo = null;
        }
    }

    /**
     * Task getter.
     * 
     * @return managed task
     */
    public TimerTask getTask() {
        return managedTask;
    }

    /**
     * Method to increment time while task is running (in ms).
     * 
     * @param increment increment, can be negative
     */
    public void incrementTaskTime(long increment) {

        long nowTimestamp = Calendar.getInstance().getTimeInMillis();

        // new var, to not modify parameter
        long localIncrement = increment;

        // do go over 0
        if (taskTimingBeforeStartInMs + nowTimestamp
                - taskStartCalendar.getTimeInMillis() + offsetTimeInMs
                + increment < 0) {
            localIncrement = -(taskTimingBeforeStartInMs + nowTimestamp
                    - taskStartCalendar.getTimeInMillis() + offsetTimeInMs);
        }

        offsetTimeInMs += localIncrement;
    }

    /**
     * Reset timing.
     * 
     * For example if task goes over 0h00, need to reset...
     */
    public void resetTiming() {

        taskStartCalendar = Calendar.getInstance();
        lastPublishTimestamp = taskStartCalendar.getTimeInMillis();
        // need to be 0 here not managedTask.getTodayTime();
        // for idle detected when 0hOO goes...
        taskTimingBeforeStartInMs = 0; // managedTask.getTodayTime();
        offsetTimeInMs = 0;
    }

    /**
     * Check if task need to be reset.
     * 
     * Test if today number if different of taskStartTimestamp day number
     */
    protected void checkReset() {

        // TODO Put it in JTimerUtil
        Calendar nowCalendar = Calendar.getInstance();

        // si les jours actuel et jours de debut de start ne sont
        // pas identiques : reset
        if (nowCalendar.get(Calendar.DAY_OF_YEAR) != taskStartCalendar
                .get(Calendar.DAY_OF_YEAR)) {
            if (log.isInfoEnabled()) {
                log.info("Day change detected, reset timing");
            }
            resetTiming();
        }

    }

    /*
     * @see org.jdesktop.swingworker.SwingWorker#doInBackground()
     */
    @Override
    protected Void doInBackground() throws Exception {

        // task effective start
        // notify ui
        parentApp.startedTask(managedTask);

        // reset on start just for init
        resetTiming();
        // redeclare taskTimingBeforeStartInSeconds
        taskTimingBeforeStartInMs = managedTask.getTime(new Date()) * 1000;

        // get idle time
        long configIdleTime = JTimerFactory.getIdleTime();

        boolean dontWantToStop = true;
        while (dontWantToStop) {
            long loopTimestamp = Calendar.getInstance().getTimeInMillis();

            // try to detect hibernate idle time
            if (loopTimestamp - lastPublishTimestamp >= configIdleTime) {
                // hibernate detected
                // update time without idle time for specified day
                if (log.isInfoEnabled()) {
                    log.info("Hibernate detected, reseting timing");
                }
                resetTiming();
            }

            lastPublishTimestamp = loopTimestamp;

            // check user idle time
            long idleTime = 0;
            if (systemInfo != null) { // idle time available
                idleTime = systemInfo.getIdleTime();

                if (log.isDebugEnabled()) {
                    log.debug("User is idle since " + (idleTime / 1000) + " s");
                }
            }

            // check reset before publish (day change)
            checkReset();

            // if long idleTime is unavailable, if is always false
            if (idleTime >= JTimerFactory.getIdleTime()) {

                // TODO check reset, but is hard when idle is detected

                // idle detected
                // update time without idle time
                offsetTimeInMs -= configIdleTime;
                publish(taskTimingBeforeStartInMs + loopTimestamp
                        - taskStartCalendar.getTimeInMillis() + offsetTimeInMs);

                // parent application
                JTimer parentApplication = (JTimer)getApplication();
                
                // send idle detect event
                parentApplication.preIdleDetect();
                // ask user what to do
                int option = ((JTimer) getApplication()).askIdleOption();
                // send idle detect event
                parentApplication.postIdleDetect();

                // get time after user
                long currentTime = Calendar.getInstance().getTimeInMillis();
                lastPublishTimestamp = currentTime;

                switch (option) {

                case REVERT:
                    // just stop the task
                    ((JTimer) getApplication()).stopTask(managedTask);
                    break;

                case CONTINUE:
                    // refresh time
                    // remove idle time previously added
                    offsetTimeInMs += configIdleTime;
                    publish(taskTimingBeforeStartInMs + currentTime
                            - taskStartCalendar.getTimeInMillis()
                            + offsetTimeInMs);
                    break;

                default: // RESUME

                    // resume = increment idle time
                    offsetTimeInMs -= currentTime - loopTimestamp;
                    // update time
                    publish(taskTimingBeforeStartInMs + currentTime
                            - taskStartCalendar.getTimeInMillis()
                            + offsetTimeInMs);
                    break;

                }
            } else {

                // pas de idle, met a jour le temps
                publish(taskTimingBeforeStartInMs + loopTimestamp
                        - taskStartCalendar.getTimeInMillis() + offsetTimeInMs);
            }

            Thread.sleep(1000); // 1s

            // test if task want to stop
            synchronized (bWantToStop) {
                dontWantToStop = !bWantToStop.booleanValue();
            }
        }

        return null;
    }

    /*
     * (non-Javadoc)
     * 
     * @see application.Task#process(java.util.List)
     */
    @Override
    protected void process(List<Long> durations) {

        // take last notification
        // can be notified of many result
        // for example, if UI lag...
        long currentDuration = durations.get(durations.size() - 1);

        dataManager.changeTaskTime(managedTask, new Date(), currentDuration / 1000);
    }

    /**
     * Notify that task want to stop
     */
    public void wantToStop() {
        synchronized (bWantToStop) {
            bWantToStop = Boolean.TRUE;
        }
    }

    /**
     * Tell if task is trying to stop.
     * 
     * @return stopping flag
     */
    public boolean isStopping() {
        boolean stopping = false;
        synchronized (bWantToStop) {
            stopping = bWantToStop.booleanValue();
        }
        return stopping;
    }
}
