/* *##%
 * Copyright (C) 2007,2009 Code Lutin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *##%*/

package org.chorem.jtimer.entities;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.SortedMap;
import java.util.TreeMap;

import org.apache.commons.lang.SerializationUtils;
import org.chorem.jtimer.utils.DailySortedMap;

/**
 * Represents a task.
 * 
 * This task is cloneable using SerializationUtils (commons-lang).
 * 
 * @author chatellier
 * @version $Revision: 2573 $
 * 
 * Last update : $Date: 2009-06-01 18:32:28 +0200 (lun. 01 juin 2009) $
 * By : $Author: chatellier $
 * 
 * @see SerializationUtils
 */
public class TimerTask implements Cloneable, Comparable<TimerTask>,
        Serializable {

    /** serialVersionUID */
    private static final long serialVersionUID = -7590755569706702695L;

    /** Task number. */
    protected int number;

    /** Task name. */
    protected String name;

    /** Creation date. */
    protected Date creationDate;

    /** Closed task. */
    protected boolean closed;

    /**
     * Map calendar of day -> time.
     * (ordered on keys)
     */
    protected SortedMap<Date, Long> allDaysAndTimes;

    /** 
     * Map date -> annotation text.
     */
    protected SortedMap<Date, String> allDaysAnnotations;

    /**
     * Parent Task.
     * More convenient
     */
    protected TimerTask parent;

    /** 
     * Sub tasks.
     */
    protected List<TimerTask> subTasks;

    /**
     * Constructor.
     */
    public TimerTask() {
        allDaysAndTimes = new DailySortedMap<Long>();
        // les annoation sont à la seconde pres
        allDaysAnnotations = new TreeMap<Date, String>();
        subTasks = new ArrayList<TimerTask>();

        // wrong value to detect bug
        number = -1;
    }

    /**
     * Constructor with name.
     * 
     * @param name task name
     */
    public TimerTask(String name) {
        this();
        this.name = name;
    }

    /**
     * Get task number.
     * 
     * @return the number
     */
    public int getNumber() {
        return number;
    }

    /**
     * Set task number.
     * 
     * @param number the number to set
     */
    public void setNumber(int number) {
        this.number = number;
    }

    /**
     * Get task name.
     * 
     * @return the name
     */
    public String getName() {
        return name;
    }

    /**
     * Set task name.
     * 
     * @param name the name to set
     */
    public void setName(String name) {
        this.name = name;
    }

    /**
     * Get task creation date.
     * 
     * @return task creation date
     */
    public Date getCreationDate() {
        return creationDate;
    }

    /**
     * Set task creation date.
     * 
     * @param creationDate creation date
     */
    public void setCreationDate(Date creationDate) {
        this.creationDate = creationDate;
    }

    /**
     * Get closed task state.
     * 
     * @return <tt>true</tt> if task is closed
     */
    public boolean isClosed() {
        return closed;
    }

    /**
     * Set closed.
     * 
     * @param closed closed
     */
    public void setClosed(boolean closed) {
        this.closed = closed;
    }

    /**
     * Get task's subtasks.
     * 
     * @return the subTasks
     */
    public List<TimerTask> getSubTasks() {
        return subTasks;
    }

    /**
     * Add task's subtask.
     * 
     * Also add parent reference.
     * 
     * @param t the task to add
     * @return success flag
     */
    public boolean addTask(TimerTask t) {

        // set parent
        t.setParent(this);

        return subTasks.add(t);
    }

    /**
     * Add time.
     * 
     * Calendar will be forced to 0h00:00.000
     * 
     * @param date date
     * @param seconds seconds
     */
    public void setTime(Date date, Long seconds) {

        allDaysAndTimes.put(date, seconds);
    }

    /**
     * Get time at date.
     * 
     * @param date date
     * @return time at specified date
     */
    public long getTime(Date date) {
        long result = 0;

        Long t = allDaysAndTimes.get(date);
        if (t != null) {
            result = t.longValue();
        }

        return result;
    }

    /**
     * Return all data. Sorted on date.
     * 
     * @return total duration of all projects
     */
    public SortedMap<Date, Long> getAllDaysAndTimes() {
        return allDaysAndTimes;
    }

    /**
     * Add annotation.
     * 
     * @param date date
     * @param note note text
     */
    public void addAnnotation(Date date, String note) {
        allDaysAnnotations.put(date, note);
    }

    /**
     * Return all annotation, sorted on date.
     * 
     * @return annotations
     */
    public SortedMap<Date, String> getAllDaysAnnotations() {
        return allDaysAnnotations;
    }

    /**
     * Get parent.
     * 
     * Can be null if there is no parent.
     * 
     * @return the parent
     */
    public TimerTask getParent() {
        return parent;
    }

    /**
     * Set parent.
     * 
     * @param parent the parent to set
     */
    protected void setParent(TimerTask parent) {
        
        // will cause an infinite loop
        if (parent == this) {
            throw new IllegalArgumentException("Parent can't be current task");
        }
        
        this.parent = parent;
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        return name + subTasks.toString();
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#equals(java.lang.Object)
     */
    @Override
    public boolean equals(Object o) {

        if (!(o instanceof TimerTask)) {
            return false;
        }

        TimerTask otherTask = (TimerTask) o;

        // name equals
        boolean equals = name != null && name.equals(otherTask.name);
        // creation date equals
        equals &= this.creationDate == null
                || creationDate.equals(otherTask.creationDate);
        // subtask equals
        equals &= subTasks.equals(otherTask.subTasks);
        // compare times
        equals &= allDaysAndTimes.equals(otherTask.allDaysAndTimes);
        // compare times
        equals &= allDaysAnnotations.equals(otherTask.allDaysAnnotations);

        return equals;
    }

    /**
     * Clone task.
     * 
     * @return task copy
     */
    @Override
    public TimerTask clone() {

        TimerTask task = null;

        try {
            task = (TimerTask) super.clone();

            // clone collections using
            // commons-lang SerializationUtils

            // copy name
            task.name = name;

            // copy date
            task.creationDate = creationDate == null ? null
                    : (Date) creationDate.clone();

            // copy durations
            task.allDaysAndTimes = (DailySortedMap<Long>) SerializationUtils
                    .clone((DailySortedMap<Long>) allDaysAndTimes);

            // copy notes
            task.allDaysAnnotations = (TreeMap<Date, String>) SerializationUtils
                    .clone((TreeMap<Date, String>) allDaysAnnotations);

            // copy subtasks
            task.subTasks = (ArrayList<TimerTask>) SerializationUtils
                    .clone((ArrayList<TimerTask>) subTasks);
        } catch (CloneNotSupportedException e) {
            throw new RuntimeException("Can't clone", e);
        }

        return task;
    }

    /*
     * @see java.lang.Comparable#compareTo(java.lang.Object)
     */
    @Override
    public int compareTo(TimerTask o) {

        int result;

        if (this.getName() == null) {
            result = -1;
        } else if (o.getName() == null) {
            result = 1;
        } else {
            // sort on name
            result = this.getName().compareTo(o.getName());
        }

        return result;
    }
}
