/**
 * *##% Callao EntryServiceImpl
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*
 */

package org.chorem.callao.service;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.callao.entity.CallaoDAOHelper;
import org.chorem.callao.entity.Entry;
import org.chorem.callao.entity.EntryDAO;
import org.chorem.callao.entity.Account;
import org.chorem.callao.entity.Transaction;
import org.chorem.callao.service.convertObject.ConvertEntry;
import org.chorem.callao.service.dto.AccountDTO;
import org.chorem.callao.service.dto.EntryDTO;
import org.chorem.callao.service.dto.TransactionDTO;
import org.chorem.callao.service.utils.ContextCallao;
import org.chorem.callao.service.utils.NumberUtil;
import org.chorem.callao.service.utils.ServiceHelper;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;



/**
 * Cette classe permet de contrôler les différentes saisies possibles dans
 * Callao. Ces saisies sont transmises à la base de données par la classe
 * Entry. Chaque action sur une entrée déclenche une création d'un log.
 * Chaque opération comptable doit faire appel également au contrôl de la
 * période comptable sur laquelle l'entry repose. Ainsi il n'est possible
 * d'ajouter/modifier/supprimer une entrée si la période est bloquée.
 *
 * @author Rémi Chapelet
 */
public class EntryServiceImpl { // implements EntryService {

	private static final Log log = LogFactory.getLog(EntryServiceImpl.class);

	private TopiaContext rootContext = ContextCallao.getInstance().getContext();

    private LogServiceImpl logServiceImpl = new LogServiceImpl();

    // TODO A revoir car transaction a besoin de entry et vice versa
    private static TransactionServiceImpl transactionServiceImpl = new TransactionServiceImpl();

    private AccountServiceImpl accountServiceImpl = new AccountServiceImpl();

    private ConvertEntry convertEntry = new ConvertEntry();

    private static NumberUtil numberUtil = new NumberUtil();


    /**
     * Création d'une entrée comptable.
     * La période ne doit PAS être bloquée.
     * @param description
     * @param amount
     * @param debit
     * @param lettering
     * @return
     */
    public String createEntry(String description,String amount, boolean debit, 
            String lettering,String detail,Transaction transaction, Account account)
    {
        String result = isCorrectEntry(transaction,account);
        if ( result.equals(ServiceHelper.RESPOND_SUCCESS) )
        {
            result = ServiceHelper.RESPOND_ERROR;
            try {
                // Acces BDD
                TopiaContext topiaContext = rootContext.beginTransaction();
                // Chargement du DAO
                EntryDAO entryDAO = CallaoDAOHelper.getEntryDAO(topiaContext);
                // Creation de l'entrée comptable
                Entry newEntry = entryDAO.create();
                newEntry.setDescription(description);
                // On formate le nombre
                newEntry.setAmount(numberUtil.format(amount));
                newEntry.setDebit(debit);
                newEntry.setLettering(lettering);
                newEntry.setDetail(detail);
                newEntry.setTransaction(transaction);
                newEntry.setAccount(account);
                // Création BDD
                topiaContext.commitTransaction();
                // Fermeture BDD
                topiaContext.closeContext();
                if (log.isInfoEnabled()) {
                    log.info("Ajout avec succes de l'entrée comptable.");
                }
                // Création du log
                Date logDate = new Date();
                logServiceImpl.addLog(logDate, transaction.getTransDate(), ServiceHelper.LOG_ADD,
                        transaction.getVoucherRef(), transaction.getDescription(), detail, amount, debit, lettering);
                /**
                 * Il renvoie soit l'identifiant topiaId ou SUCCESS.
                 */
                result = newEntry.getTopiaId();
                //result = ServiceHelper.RESPOND_SUCCESS;
            }catch (TopiaException e) {
                log.error(e);
            }
        }
        return result;
    }

    /**
     * Permet de créer une entrée comptable à partir de son DTO.
     * @param entryDTO
     * @return
     */
    public String createEntry (EntryDTO entryDTO)
    {
        // Recherche TransactionDTO
        TransactionDTO transactonDTO = entryDTO.getTransactionDTO();
        Transaction transaction = transactionServiceImpl.searchTransactionWithTopiaId(transactonDTO);
        // Recherche AccountDTO
        AccountDTO accountDTO = entryDTO.getAccountDTO();
        Account account = accountServiceImpl.searchAccount(accountDTO.getAccountNumber());
        // Création entry
        String result = createEntry(entryDTO.getDescription(),entryDTO.getAmount(),
                entryDTO.isDebit(), entryDTO.getLettering(),entryDTO.getDetail(),transaction,account);
        return result;
    }


    /**
     * Vérifie que les objets transaction et account ne sont pas nulls.
     * @param transaction
     * @param account
     * @return
     */
    public String isCorrectEntry (Transaction transaction, Account account)
    {
        String result = ServiceHelper.RESPOND_SUCCESS;
        // Vérification si timeSpan et Journal existe bien
        if (account == null)
        {
            if (log.isErrorEnabled()) {
                log.error("Création entry : Le account n'existe pas.");
            }
            result = ServiceHelper.ACCOUNT_NOT_EXIST;
        }
        if (transaction == null)
        {
            if (log.isErrorEnabled()) {
                log.error("Création entry : La transaction n'existe pas.");
            }
            result = ServiceHelper.TRANSACTION_NOT_EXIST;
        } 
        return result;
    }


    /**
     * Recherche toutes les entrées comptables d'une transaction.
     * @param transaction
     * @return
     */
    public List<Entry> searchEntryWithTransaction (Transaction transaction)
    {
        List<Entry> listEntry = null;
        if (transaction != null)
        {
            try {
                // Acces BDD
                TopiaContext topiaContext = rootContext.beginTransaction();
                // Chargement du DAO
                EntryDAO entryDAO = CallaoDAOHelper.getEntryDAO(topiaContext);
                // Recherche
                listEntry = entryDAO.findAllByTransaction(transaction);
                // Fermeture BDD
                topiaContext.closeContext();
            }catch (TopiaException e) {
                log.error(e);
            }
        }
        return listEntry;
    }

    /**
     * Recherche toutes les entrées comptables d'une transaction, et renvoie
     * au format DTO.
     * @param transaction
     * @return
     */
    public List<EntryDTO> searchEntryDTOWithTransaction (Transaction transaction)
    {
        List<EntryDTO> listEntryDTO = new ArrayList<EntryDTO>();
        // Si la transaction est trouvée
        if ( transaction != null )
        {
            try {
                // Acces BDD
                TopiaContext topiaContext = rootContext.beginTransaction();
                // Chargement du DAO
                EntryDAO entryDAO = CallaoDAOHelper.getEntryDAO(topiaContext);
                // Recherche toutes les entrées comptables de la transaction
                List<Entry> listEntry = entryDAO.findAllByTransaction(transaction);
                // Pour chaque entry, on fait appel au convertisseur pour avoir en DTO
                convertEntry.setTransaction(topiaContext);
                for (Entry entry : listEntry)
                {
                    EntryDTO entryDTO = convertEntry.entryEntityToDto(entry);
                    // On ajoute le nouvel DTO
                    listEntryDTO.add(entryDTO);
                }
                // Fermeture BDD
                topiaContext.closeContext();
            }catch (TopiaException e) {
                log.error(e);
            }
        }
        return listEntryDTO;
    }


    /**
     * Recherche les entrées comptables pour une transaction donnée au format DTO.
     * @param transactionTopiaId identifiant de la transaction
     * @return
     */
    public List<EntryDTO> searchEntryDTOWithTransaction (String transactionTopiaId)
    {
        List<EntryDTO> listEntryDTO = new ArrayList<EntryDTO>();
        try {
            // Acces BDD
            TopiaContext topiaContext = rootContext.beginTransaction();
            // Chargement du DAO
            EntryDAO entryDAO = CallaoDAOHelper.getEntryDAO(topiaContext);
            // Recherche de la transaction
            Transaction transaction = transactionServiceImpl.searchTransactionWithTopiaId(transactionTopiaId);
            // Si la transaction est trouvée
            if ( transaction != null )
            {
                // Recherche toutes les entrées comptables de la transaction
                List<Entry> listEntry = entryDAO.findAllByTransaction(transaction);
                // Pour chaque entry, on fait appel au convertisseur pour avoir en DTO
                convertEntry.setTransaction(topiaContext);
                for (Entry entry : listEntry)
                {
                    EntryDTO entryDTO = convertEntry.entryEntityToDto(entry);
                    // On ajoute l'entrée DTO
                    listEntryDTO.add(entryDTO);
                }
            }
            // Fermeture BDD
			topiaContext.closeContext();
        }catch (TopiaException e) {
			log.error(e);
		}
        return listEntryDTO;
    }


    /**
     * Recherche les entrées comptables pour une transaction donnée au format DTO.
     * @param transactionDTO
     * @return
     */
    public List<EntryDTO> searchEntryDTOWithTransaction (TransactionDTO transactionDTO)
    {
        List<EntryDTO> listEntryOfTransaction = searchEntryDTOWithTransaction(transactionDTO.getId());
        return listEntryOfTransaction;
    }


    /**
     * Permet d'effacer une entrée comptable à partir de son identifiant topia.
     * @param topiaId
     * @return
     */
    public String removeEntry (String topiaId)
    {
        String result = ServiceHelper.RESPOND_ERROR;
        try {
            // Acces BDD
            TopiaContext topiaContext = rootContext.beginTransaction();
            // Chargement du DAO
            EntryDAO entryDAO = CallaoDAOHelper.getEntryDAO(topiaContext);
            // Recherche de l'entry
            Entry entryDelete = entryDAO.findByTopiaId(topiaId);
            // Si l'entry existe alors on supprime
            if ( entryDelete != null )
            {
                // Supprime
                entryDAO.delete(entryDelete);
                // Création BDD
                topiaContext.commitTransaction();                
                result = ServiceHelper.RESPOND_SUCCESS;
            } else {
                result = ServiceHelper.ENTRY_NOT_EXIST;
            }
            // Fermeture BDD
			topiaContext.closeContext();
        }catch (TopiaException e) {
			log.error(e);
		}
        return result;
    }


    /**
     * Permet d'effacer une entrée comptable au format DTO.
     * @param entryDTO
     * @return
     */
    public String removeEntry (EntryDTO entryDTO)
    {
        String result = removeEntry(entryDTO.getId());
        return result;
    }

    
    /**
     * Modification d'une entrée comptable.
     * @param topiaId
     * @param description
     * @param amount
     * @param debit
     * @param lettering
     * @param detail
     * @param transaction
     * @param account
     * @return
     */
    public String modifyEntry (String topiaId,String description,String amount, boolean debit,
            String lettering,String detail,Account account)
    {
        String result = ServiceHelper.RESPOND_ERROR;
        try {
            // Acces BDD
            TopiaContext topiaContext = rootContext.beginTransaction();
            // Chargement du DAO
            EntryDAO entryDAO = CallaoDAOHelper.getEntryDAO(topiaContext);
            // Recherche Entry
            Entry entry = entryDAO.findByTopiaId(topiaId);
            entry.setDescription(description);
            // On formate le nombre
            entry.setAmount(numberUtil.format(amount));
            entry.setDebit(debit);
            entry.setLettering(lettering);
            entry.setDetail(detail);
            entry.setAccount(account);
            entry.update();
            // Création BDD
            topiaContext.commitTransaction();
            // Chargement DAO transaction
            Transaction transaction = entry.getTransaction();
            if (log.isInfoEnabled()) {
                log.info("Edit avec succes de l'entrée comptable.");
            }
            // Création du log
            Date logDate = new Date();
            logServiceImpl.addLog(logDate, transaction.getTransDate(), ServiceHelper.LOG_MODIFY,
                    transaction.getVoucherRef(), transaction.getDescription(), detail, amount, debit, lettering);
            result = ServiceHelper.RESPOND_SUCCESS;
            // Fermeture BDD
            topiaContext.closeContext();
        }catch (TopiaException e) {
            log.error(e);
        }
        return result;
    }

    
    /**
     * Permet de modifier une entrée comptable.
     * @param entryDTO
     * @return
     */
    public String modifyEntry (EntryDTO entryDTO)
    {
        // Recherche AccountDTO
        AccountDTO accountDTO = entryDTO.getAccountDTO();
        Account account = accountServiceImpl.searchAccount(accountDTO.getAccountNumber());
        // Modification entry 
        String result = modifyEntry(entryDTO.getId(),entryDTO.getDescription(),entryDTO.getAmount(),
                entryDTO.isDebit(), entryDTO.getLettering(),entryDTO.getDetail(),account);
        return result;
    }

    //@Override
	public String[] getMethods() {
		// TODO Auto-generated method stub
		return null;
	}

	//@Override
	public void destroy() {
		// TODO Auto-generated method stub

	}

	//@Override
	public void init(TopiaContext arg0) {
		// TODO Auto-generated method stub
	}


}
