/*
 * #%L
 * bow
 * 
 * $Id: BookmarkActions.java 144 2010-12-23 10:24:48Z bpoussin $
 * $HeadURL: http://svn.chorem.org/svn/bow/tags/bow-0.3/src/main/java/org/chorem/bow/BookmarkActions.java $
 * %%
 * Copyright (C) 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.bow;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.nuiton.util.StringUtil;
import org.nuiton.wikitty.FacetTopic;

/**
 *
 * @author bbrossaud
 */
public class BookmarkActions {

    protected String fullTextLine = "";
    protected List<Bookmark> bookmarks = new ArrayList<Bookmark>(); // bookmarks which contain the search tags
    protected List<Bookmark> lastBookmarks = new ArrayList<Bookmark>();
    protected List<FacetTopic> tagCloud = new ArrayList<FacetTopic>(); // associate a tag with its frequency
    protected List<String> tagsSearch = new ArrayList<String>(); // contains the tags taped in the search field
    protected int tmax = -1;
    protected int tmin = -1;

    /* @param html      String html
     * @param name      String name
     * @param user      User user
     * @return Bookmark the bookmark filled
     */
    public static Bookmark createBookmarkFromHtml(String html, String name, User user) {
        String link = getUrlFromHtml(html);
        String tags = getTagsFromHtml(html);
        Date date = getDateFromHtml(html);
        Bookmark bookmark = createBookmark(link, name, tags, user, null, date);
        return bookmark;
    }

    /* @param html      String html
     * @return Date     from the html
     */
    public static Date getDateFromHtml(String html) {
        Date date = null;
        if (html != null) {
            Pattern p = Pattern.compile("ADD_DATE=\"([^\"]*)\"");
            Matcher m = p.matcher(html);
            if (m.find()) {
                String str = m.group(1);
                if (str != null && !str.isEmpty()) {
                    long time = Long.valueOf(str) * 1000; // the date in ms
                    date = new Date(time);
                }
            }
        }
        return date;
    }

    /* @param bookmarks List<Bookmarks> bookmarks
     * @return String   the html file
     */
    public static String getExportHtmlBookmark(List<Bookmark> bookmarks) {
        String export;
        export = "<!DOCTYPE NETSCAPE-Bookmark-file-1>\n"
                + "<META HTTP-EQUIV=\"Content-Type\" CONTENT=\"text/html; charset=UTF-8\">\n"
                + "<DL><p>\n";
        if (bookmarks != null) {
            for (Bookmark bookmark : bookmarks) {
                export += "<DT><A HREF=\"";
                export += bookmark.getLink() + "\" ";
                Date date = bookmark.getDate();
                if (date != null) {
                    long milli = date.getTime();
                    String time = String.valueOf(milli / 1000);
                    export += "ADD_DATE=\"" + time + "\" LAST_MODIFIED=\"" + time + "\" ";
                }
                export += "LAST_CHARSET=\"UTF-8\" ";
                Set<String> tags = bookmark.getTags();
                if (tags != null && !tags.isEmpty()) {
                    export += "SHORTCUTURL=\"" + getBookmarkTagsString(bookmark) + "\"";
                }
                export += ">" + bookmark.getDescription() + "</A>\n";
            }
        }
        export += "</DL><p>";
        return export;
    }

    /* @param html   String html
     *
     */
    public static String getUrlFromHtml(String html) {
        String url = "";
        if (html != null) {
            Pattern p = Pattern.compile("A HREF=\"([^\"]*)\"");
            Matcher m = p.matcher(html);
            if (m.find()) {
                url = m.group(1);
            }
        }
        return url;
    }

    public static String getTagsFromHtml(String html) {
        String tags = "";
        if (html != null) {
            Pattern p = Pattern.compile("SHORTCUTURL=\"([^\"]*)\"");
            Matcher m = p.matcher(html);
            if (m.find()) {
                tags = m.group(1);
            }
        }
        return tags;
    }

    /* @param url           String which contains the bookmark url
     * @param nameAndTags   String which contains the name and the tags of the
     *                      bookmark separated by '|' ==> name|tag1 tag2...
     * @param user          bookmark user
     * @return null         if all fields are not correctly filled
     * @return Bookmark     the website bookmark
     */
    public static Bookmark createBookmark(String url, String nameAndTags, User user) {

        Bookmark bookmark = new BookmarkImpl();

        int nameIndex = nameAndTags.indexOf('|'); // get the index name of the website
        if (nameIndex > 0) {
            String name = nameAndTags.substring(0, nameIndex); // get the website name
            if (!name.isEmpty()) {
                name = name.trim();
                bookmark.setDescription(name); // set the description (website name)
                bookmark.setAlias(name);
            }
        }
        String tags = nameAndTags;
        if (nameIndex >= 0) {
            tags = nameAndTags.substring(nameIndex + 1); // get tags, +1 because of '|'
        }
        addTagsToBookmark(tags, bookmark);
        addUrlToBookmark(url, bookmark);
        bookmark.setClick(0);
        bookmark.setEmail(user.getEmail()); // set the email (user name)
        bookmark.setDate(new Date()); // set the date
        return bookmark;
    }

    public static void addUrlToBookmark(String url, Bookmark bookmark) {
        if (url != null && !url.isEmpty()) {
            String link = url.trim();
            if (! link.matches("[a-z]+://.+") ) {
                link = "http://" + link;
            }
            bookmark.setLink(link);
        } else {
            bookmark.setLink("");
        }
    }

    public static void addTagsToBookmark(String tags, Bookmark bookmark) {
        if (tags != null && !tags.isEmpty()) {
            tags = tags.trim();
            String[] tagsTab = tags.split("\\s+"); // put the tags in an array
            for (String aTagsTab : tagsTab) {
                if (aTagsTab == null) {

                    continue;
                }
                Set<String> tagList = bookmark.getTags();
                if (tagList != null) {
                    if (!tagList.contains(aTagsTab) && !aTagsTab.isEmpty()) {
                        bookmark.addTags(aTagsTab); // Added tag for each rank (one tag by rank)
                    }
                } else if (!aTagsTab.isEmpty()) {
                    bookmark.addTags(aTagsTab);
                }
            }
        }
    }

    public static void addTagsToBookmark(List<String> tagList, Bookmark bookmark) {
        if (tagList != null && !tagList.isEmpty()) {
            for (String tag : tagList) {
                addTagsToBookmark(tag, bookmark);
            }
        }
    }

    public static Bookmark createBookmark(String url, String name, String tags, User user, String alias, Date date) {
        Bookmark bookmark = new BookmarkImpl();
        if (name != null) {
            name = name.trim();
            bookmark.setDescription(name);
        } else {
            bookmark.setDescription("");
        }

        addTagsToBookmark(tags, bookmark);

        addUrlToBookmark(url, bookmark);

        bookmark.setClick(0);

        bookmark.setEmail(user.getEmail()); // set the email (user name)
        if (date == null) {
            date = new Date();
        }
        bookmark.setDate(date); // set the date
        if (alias != null) {
            alias = alias.trim();
            bookmark.setAlias(alias);
        } else {
            bookmark.setAlias("");
        }
        return bookmark;
    }

    public static void updateBookmarkName(Bookmark bookmark, String name) {
        if (name != null) {
            name = name.trim();
            bookmark.setDescription(name);
        }
    }

    public static void updateBookmark(Bookmark bookmark, String name, String link, String tags, String alias) {
        if (name != null) {
            name = name.trim();
            bookmark.setDescription(name);
        }
        addUrlToBookmark(link, bookmark);
        bookmark.clearTags();
        addTagsToBookmark(tags, bookmark);
        if (alias != null) {
            alias = alias.trim();
            bookmark.setAlias(alias);
        }
    }

    public static String getBookmarkTagsString(Bookmark bookmark) {
        if (bookmark != null) {
            Set<String> tags = bookmark.getTags();
            if (tags != null) {
                return StringUtil.join(tags, " ", true);
            }
        }
        return "";
    }
    
    public void createTagCloud(List<FacetTopic> cpy) {
        setTagCloud(cpy);
        deleteTagBySearch();
        defineTValues();
    }

    public void setTagCloud(List<FacetTopic> cpy) {
        if (cpy != null) {
            tagCloud = new ArrayList<FacetTopic>(cpy);
        }
    }

    public void deleteTagBySearch() {
        List<FacetTopic> save = new ArrayList<FacetTopic>(tagCloud);
        for (FacetTopic topic : save) {
            String name = topic.getTopicName();
            if (tagsSearch.contains(name)) {
                tagCloud.remove(topic);
            }
        }
    }

    protected void defineTValues() {
        tmax = -1; // correspond to the most tag frequency in the tag cloud
        tmin = -1; // correspond to the less tag frequency in the tag cloud
        for (FacetTopic tag : tagCloud) {
            int value = tag.getCount();
            if (tmax < value) { // search the most tag frequancy
                tmax = value;
            }
            if (tmin == -1) {
                tmin = value;
            } else if (tmin > value) { // search less tag frequency
                tmin = value;
            }
        }
    }

    public void addTags(String str) {
        if (str != null) {
            if (!str.isEmpty()) {
                str = str.trim();
                String[] tags = str.split("\\s+");
                for (String tag : tags) {
                    if (!tagsSearch.contains(tag)) {
                        tagsSearch.add(tag);
                    }
                }
            }
        }
    }

    public void emptySearchline() {
        List<Bookmark> bookmarkList = new ArrayList<Bookmark>(bookmarks);
        for (Bookmark bookmark : bookmarkList) {
            Set<String> tags = bookmark.getTags();
            if (tags != null && !tags.isEmpty()) {
                bookmarks.remove(bookmark);
            }
        }
    }

    public void reset() {
        bookmarks.clear();
        tagCloud.clear();
        tagsSearch.clear();
        tmax = -1;
        tmin = -1;
    }

    public void setFullTextLine(String line) {
        if (line == null) {
            fullTextLine = "";
        } else {
            fullTextLine = line;
        }
    }

    public void setBookmarks(List<Bookmark> bookmarksList) {
        List<Bookmark> newList = new ArrayList<Bookmark>(bookmarksList);
        bookmarks = newList;
    }

    public void setLastBookmarks(List<Bookmark> bookmarksList) {
        List<Bookmark> newList = new ArrayList<Bookmark>(bookmarksList);
        lastBookmarks = newList;
    }

    public void setTagSearch(List<String> cpy) {
        tagsSearch = cpy;
    }

    public String getSearchLine() {
        if (tagsSearch != null) {
            return StringUtil.join(tagsSearch, " ", true); // return the search line created with the tags
        }
        return null;
    }

    public int getFont(int ti) {
        int font = 1;
        if (tmax > tmin) {
            font = 30 * (ti - tmin) / (tmax - tmin); // get the font size for a tag frequency
        }
        if (font < 10) {
            font = 10;
        }
        return font;
    }

    public List<FacetTopic> getTagsCloud() {
        return tagCloud;
    }

    public List<String> getTagsSearch() {
        return tagsSearch;
    }

    public List<Bookmark> getBookmarks() {
        return bookmarks;
    }

    public List<Bookmark> getLastBookmarks() {
        return lastBookmarks;
    }

    public int getTmin() {
        return tmin;
    }

    public int getTmax() {
        return tmax;
    }

    public String getFullTextLine() {
        return fullTextLine;
    }
}
