/*
 * #%L
 * bow
 * 
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.bow;

/**
 *
 * @author bbrossaud
 */
import java.io.IOException;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Session;
import javax.mail.Transport;
import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeMessage;
import javax.servlet.ServletException;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import org.apache.commons.fileupload.FileItem;
import org.apache.commons.fileupload.FileUploadException;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.apache.commons.lang.RandomStringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.htmlparser.Node;
import org.htmlparser.Parser;
import org.htmlparser.util.NodeList;
import org.htmlparser.util.ParserException;
import org.htmlparser.util.SimpleNodeIterator;
import org.nuiton.util.ArgumentsParserException;
import org.nuiton.util.StringUtil;
import org.nuiton.wikitty.Criteria;
import org.nuiton.wikitty.FacetTopic;
import org.nuiton.wikitty.PagedResult;
import org.nuiton.wikitty.WikittyProxy;
import org.nuiton.wikitty.search.Search;

/**
 *
 * @author bbrossaud
 */
public class ControllerServlet extends HttpServlet {

    private static final Log log = LogFactory.getLog(ControllerServlet.class);
    protected String version = "";
    protected String bowServletUrl = "";

    public ControllerServlet() throws ArgumentsParserException, Exception {
        BowConfig config = BowConfig.getInstance();
        version = config.getVersion();
        bowServletUrl = config.getBowUrl();
        if (bowServletUrl == null) {
            throw new Exception("No bow.url=\"SERVER URL\" in bow.properties");
        }
        bowServletUrl += config.getServletBow();
    }

    @Override
    public void doGet(HttpServletRequest request, HttpServletResponse response)
            throws IOException, ServletException {
        this.doPost(request, response);
    }

    /* @param request              servlet request
     * @param response             servlet response
     * @throws ServletException    if a servlet error occurs
     */
    @Override
    public void doPost(HttpServletRequest request, HttpServletResponse response)
            throws IOException, ServletException {
        try {
            request.setAttribute("version", version);
            request.setAttribute("bowUrl", bowServletUrl);
            HttpSession session = request.getSession(true);
            User user = (User) session.getAttribute("user");

            String token = request.getParameter("token"); // token or not
            if (token != null && !token.isEmpty()) {
                user = checkToken(token, session); // retrieve user by token
            }

            String action = request.getParameter("action");
            if (action != null) {
                if (action.equals("register")) {
                    if (log.isDebugEnabled()) {
                        log.debug("Going to actionRegister");
                    }
                    this.actionRegister(request, response, session);
                } else if (action.equals("registration")) {
                    if (log.isDebugEnabled()) {
                        log.debug("Going to Register");
                    }
                    request.getRequestDispatcher("register.jsp").forward(request, response);
                } else if (action.equals("forgotPassword")) {
                    if (log.isDebugEnabled()) {
                        log.debug("Going to forgotPassword");
                    }
                    request.getRequestDispatcher("forgotPassword.jsp").forward(request, response);
                } else if (action.equals("login")) {
                    if (log.isDebugEnabled()) {
                        log.debug("Going to actionLogin");
                    }
                    this.actionLogin(request, response, session);
                } else if (action.equals("logout")) {
                    if (log.isDebugEnabled()) {
                        log.debug("Going to actionLogout");
                    }
                    this.actionLogout(request, response, session);
                } else if (action.equals("home") && user != null) {
                    if (log.isDebugEnabled()) {
                        log.debug("Going to actionHome.jsp");
                    }
                    this.actionHome(request, user);
                    request.getRequestDispatcher("main.jsp").forward(request, response);
                } else if (action.equals("addUrl") && user != null) {
                    if (log.isDebugEnabled()) {
                        log.debug("Going to actionAddUrl");
                    }
                    this.actionAddUrl(request, response, user);
                } else if (action.equals("sendPassword")) {
                    if (log.isDebugEnabled()) {
                        log.debug("Going to actionSendPassword");
                    }
                    this.actionSendPassword(request, response);
                } else if (action.equals("modifyBookmark") && user != null) {
                    if (log.isDebugEnabled()) {
                        log.debug("Going to actionModif");
                    }
                    this.actionModifyBookmark(request, response, user);
                } else if (action.equals("importBookmarks") && user != null) {
                    if (log.isDebugEnabled()) {
                        log.debug("Going to actionImportBookmarks");
                    }
                    this.actionImportBookmarks(request, response, user);
                } else if (action.equals("exportBookmarks") && user != null) {
                    if (log.isDebugEnabled()) {
                        log.debug("Going to actionExportBookmarks");
                    }
                    this.actionExportBookmarks(response, user);
                } else if (action.equals("search") && user != null) {
                    if (log.isDebugEnabled()) {
                        log.debug("Going to actionSearch");
                    }
                    this.actionSearch(request, user);
                    request.getRequestDispatcher("main.jsp").forward(request, response);
                } else if (action.equals("generateToken") && user != null) {
                    if (log.isDebugEnabled()) {
                        log.debug("Going to actionGenerateToken");
                    }
                    this.actionGenerateToken(request, response, user, session);
                    request.getRequestDispatcher("preferences.jsp").forward(request, response);
                } else if (action.equals("deleteTag") && user != null) {
                    if (log.isDebugEnabled()) {
                        log.debug("Going to actionDeleteTag");
                    }
                    this.actionDeleteTag(request, response, user);
                } else if (action.equals("addAlias") && user != null) {
                    if (log.isDebugEnabled()) {
                        log.debug("Going to actionAddAlias");
                    }
                    this.actionAddAlias(request, response);
                } else if (action.equals("removeBookmark") && user != null) {
                    if (log.isDebugEnabled()) {
                        log.debug("Going to actionRemoveBookmark");
                    }
                    this.actionRemoveBookmark(request, response, user);
                } else if (action.equals("editBookmark") && user != null) {
                    if (log.isDebugEnabled()) {
                        log.debug("Going to actionEditBookmark");
                    }
                    this.actionEditBookmark(request, response, user);
                } else if (action.equals("order") && user != null) {
                    if (log.isDebugEnabled()) {
                        log.debug("Going to actionOrder");
                    }
                    this.actionOrder(request, response, user);
                    request.getRequestDispatcher("main.jsp").forward(request, response);
                } else if (action.equals("addClick") && user != null) {
                    if (log.isDebugEnabled()) {
                        log.debug("Going to actionAddClic");
                    }
                    this.actionAddClick(request, response);
                } else if (action.equals("temporaryXml")) {
                    if (log.isDebugEnabled()) {
                        log.debug("Going to temporaryXml.jsp");
                    }
                    request.getRequestDispatcher("temporaryXml.jsp").forward(request, response);
                } else if (action.equals("permanentXml")) {
                    if (log.isDebugEnabled()) {
                        log.debug("Going to permanentXml.jsp");
                    }
                    request.getRequestDispatcher("permanentXml.jsp").forward(request, response);
                } else if (action.equals("openSearchSuggestion") && user != null) {
                    if (log.isDebugEnabled()) {
                        log.debug("Going to actionOpenSearchSuggestion");
                    }
                    this.actionOpenSearchSuggestion(request, response, user);
                    request.getRequestDispatcher("suggestions.jsp").forward(request, response);
                } else if (action.equals("openSearchResult") && user != null) {
                    if (log.isDebugEnabled()) {
                        log.debug("Going to actionOpenSearchResult");
                    }
                    this.actionOpenSearchResult(request, response, session, user, token);
                } else if (action.equals("fullText") && user != null) {
                    if (log.isDebugEnabled()) {
                        log.debug("Going to actionFullText");
                    }
                    this.actionFullText(request, user);
                    request.getRequestDispatcher("main.jsp").forward(request, response);
                } else if (action.equals("preferences") && user != null) {
                    if (log.isDebugEnabled()) {
                        log.debug("Going to actionPreferences");
                    }
                    request.getRequestDispatcher("preferences.jsp").forward(request, response);
                } else if (action.equals("admin") && user != null) {
                    if (log.isDebugEnabled()) {
                        log.debug("Going to actionAdmin");
                    }
                    request.getRequestDispatcher("admin.jsp").forward(request, response);
                } else if (action.equals("changePreferences") && user != null) {
                    if (log.isDebugEnabled()) {
                        log.debug("Going to actionChangePreferences");
                    }
                    this.actionChangePreferences(request, session, user);
                    request.getRequestDispatcher("preferences.jsp").forward(request, response);
                } else if (action.equals("reIndexation") && user != null) {
                    if (log.isDebugEnabled()) {
                        log.debug("Going to actionReIndexation");
                    }
                    this.actionReIndexation(request, session, user);
                    request.getRequestDispatcher("admin.jsp").forward(request, response);
                } else {
                    if (user != null) {
                        request.getRequestDispatcher("error.jsp").forward(request, response);
                    } else {
                        request.getRequestDispatcher("login.jsp").forward(request, response);
                    }
                }
            } else {
                request.getRequestDispatcher("login.jsp").forward(request, response);
            }
        } catch (Exception eee) {
            log.error("Can't do action", eee);
            request.getRequestDispatcher("error.jsp").forward(request, response);
        }
    }


    /* @param request               servlet request
     * @param response              servlet response
     * @param user                  the user
     * @param token                 the permanent or temporary token
     * @throws ServletException     if a servlet error occurs
     * @description                 set the openSearch result
     */
    protected void actionOpenSearchResult(HttpServletRequest request, HttpServletResponse response, HttpSession session, User user, String token)
            throws IOException, ServletException, NoSuchAlgorithmException {
        String searchLine = request.getParameter("searchLine");
        if (searchLine != null && searchLine.matches("^http://[^ ]*") && !searchLine.contains(" ")) {
            response.sendRedirect(searchLine);
        } else {
            session.setAttribute("user", user);
            initializeToken(session, user);
            WikittyProxy proxy = BowProxy.getInstance();
            Criteria criteria = getBookmarkListCriteriaByUser(user, searchLine);
            criteria = criteria.addSortDescending(Bookmark.FQ_FIELD_CLICK);
            PagedResult result = proxy.findAllByCriteria(Bookmark.class, criteria);// retrieve bookmarks by search
            BookmarkActions bookmarkActions = createBookmarkActions(request, result, searchLine);
            request.setAttribute("bookmarkActions", bookmarkActions);
            request.setAttribute("token", token);
            request.getRequestDispatcher("main.jsp").forward(request, response);
        }
    }

    /* @param request               servlet request
     * @param response              servlet response
     * @param user                  the user
     * @throws ServletException     if a servlet error occurs
     * @description                 set the openSearch suggestions
     */
    protected void actionOpenSearchSuggestion(HttpServletRequest request, HttpServletResponse response, User user)
            throws IOException, ServletException {
        if (user != null) {
            String search = request.getParameter("searchLine");
            if (search != null) {
                OpenSearchActions openSearchActions = new OpenSearchActions();
                WikittyProxy proxy = BowProxy.getInstance();
                String[] words = search.split("\\s+");
                List<String> searchLine = new ArrayList<String>(Arrays.asList(words));
                if (search.charAt(search.length() - 1) == ' ') {
                    searchLine.add(""); // if the user types nothing we have to propose suggestions
                }
                Criteria criteria = null;
                if (searchLine.size() > 1) {
                    List<String> cpy = new ArrayList<String>(searchLine);
                    cpy.remove(cpy.size() - 1);
                    criteria = Search.query().eq(Bookmark.FQ_FIELD_EMAIL, user.getEmail()).
                            eq(Bookmark.FQ_FIELD_TAGS, cpy).criteria().addFacetField(Bookmark.FQ_FIELD_TAGS);
                } else {
                    criteria = Search.query().eq(Bookmark.FQ_FIELD_EMAIL, user.getEmail()).
                            criteria().addFacetField(Bookmark.FQ_FIELD_TAGS);
                }
                PagedResult result = proxy.findAllByCriteria(Bookmark.class, criteria);
                List<Bookmark> bookList = result.getAll();
                List<FacetTopic> topics = result.getTopic(Bookmark.FQ_FIELD_TAGS);
                openSearchActions.setBookmarkList(bookList);
                openSearchActions.setSuggestionList(topics);
                openSearchActions.findSuggestions(searchLine);
                request.setAttribute("openSearchAction", openSearchActions);
            }
        }
    }

    /* @param request               servlet request
     * @param response              servlet response
     * @param session               current session
     * @throws ServletException     if a servlet error occurs
     * @description                 delete the session
     */
    protected void actionLogout(HttpServletRequest request, HttpServletResponse response, HttpSession session)
            throws IOException, ServletException {
        session.invalidate();
        request.getRequestDispatcher("login.jsp").forward(request, response);
    }

    /* @param request               servlet request
     * @param response              servlet response
     * @throws ServletException     if a servlet error occurs
     * @description                 when the user click on the bookmark bowServletUrl, we
     *                              increment the number of click
     */
    protected void actionAddClick(HttpServletRequest request, HttpServletResponse response)
            throws IOException, ServletException {
        String bookmarkId = request.getParameter("bookmarkId");
        if (bookmarkId != null && !bookmarkId.isEmpty()) {
            WikittyProxy proxy = BowProxy.getInstance();
            Bookmark bookmark = proxy.restore(Bookmark.class, bookmarkId);
            if (bookmark != null) {
                int click = bookmark.getClick();
                ++click;
                bookmark.setClick(click);
                proxy.store(bookmark);
                String link = bookmark.getLink();
                response.sendRedirect(link);
            }
        }
    }

    /* @param request               servlet request
     * @param response              servlet response
     * @param user                  the user
     * @throws ServletException     if a servlet error occurs
     * @description                 order the bookmarks by type
     */
    protected void actionOrder(HttpServletRequest request, HttpServletResponse response, User user)
            throws IOException, ServletException {
        String type = request.getParameter("type");
        String searchLine = request.getParameter("searchLine");
        Criteria baseCriteria = getBookmarkListCriteriaByUser(user, searchLine);
        if (type != null && baseCriteria != null && !type.isEmpty()) {
            WikittyProxy proxy = BowProxy.getInstance();
            PagedResult result = null;
            if (type.equals("ascName")) {
                Criteria criteria = baseCriteria.addSortAscending(Bookmark.FQ_FIELD_DESCRIPTION);
                result = proxy.findAllByCriteria(Bookmark.class, criteria);
            } else if (type.equals("ascDate")) {
                Criteria criteria = baseCriteria.addSortAscending(Bookmark.FQ_FIELD_DATE);
                result = proxy.findAllByCriteria(Bookmark.class, criteria);
            } else if (type.equals("ascClick")) {
                Criteria criteria = baseCriteria.addSortAscending(Bookmark.FQ_FIELD_CLICK);
                result = proxy.findAllByCriteria(Bookmark.class, criteria);
            } else if (type.equals("descName")) {
                Criteria criteria = baseCriteria.addSortDescending(Bookmark.FQ_FIELD_DESCRIPTION);
                result = proxy.findAllByCriteria(Bookmark.class, criteria);
            } else if (type.equals("descDate")) {
                Criteria criteria = baseCriteria.addSortDescending(Bookmark.FQ_FIELD_DATE);
                result = proxy.findAllByCriteria(Bookmark.class, criteria);
            } else if (type.equals("descClick")) {
                Criteria criteria = baseCriteria.addSortDescending(Bookmark.FQ_FIELD_CLICK);
                result = proxy.findAllByCriteria(Bookmark.class, criteria);
            }
            BookmarkActions bookmarkActions = createBookmarkActions(request, result, searchLine);
            request.setAttribute("bookmarkActions", bookmarkActions);
        }
    }

    /* @param request               servlet request
     * @param response              servlet response
     * @param user                  the user
     * @throws ServletException     if a servlet error occurs
     * @description                 edit the bookmark
     */
    protected void actionEditBookmark(HttpServletRequest request, HttpServletResponse response, User user)
            throws IOException, ServletException {
        String bookmarkId = request.getParameter("bookmarkId");
        if (bookmarkId != null && !bookmarkId.isEmpty()) {
            WikittyProxy proxy = BowProxy.getInstance();
            Bookmark bookmark = proxy.restore(Bookmark.class, bookmarkId);
            if (bookmark != null) {
                request.setAttribute("link", bookmark.getLink());
                request.setAttribute("name", bookmark.getDescription());
                request.setAttribute("alias", bookmark.getAlias());
                request.setAttribute("tags", BookmarkActions.getBookmarkTagsString(bookmark));
                request.setAttribute("action", "bow?action=modifyBookmark");
                request.setAttribute("bookmarkId", bookmarkId);
            }
        }
        String searchLine = request.getParameter("searchLine");
        if (searchLine == null) {
            actionHome(request, user);
        } else {
            actionSearch(request, user);
        }
        request.getRequestDispatcher("main.jsp").forward(request, response);
    }

    /* @param request               servlet request
     * @param response              servlet response
     * @param user                  the user
     * @throws ServletException     if a servlet error occurs
     * @description                 remove the bookmark
     */
    protected void actionRemoveBookmark(HttpServletRequest request, HttpServletResponse response, User user)
            throws IOException, ServletException {
        String bookmarkId = request.getParameter("bookmarkId");
        if (bookmarkId != null && !bookmarkId.isEmpty()) {
            WikittyProxy proxy = BowProxy.getInstance();
            Bookmark bookmark = proxy.restore(Bookmark.class, bookmarkId);
            if (bookmark != null) {
                proxy.delete(bookmarkId);
            }
        }
        redirectToTheGoodPage(request, response);
    }

    /* @param request               servlet request
     * @param response              servlet response
     * @param user                  the user
     * @throws ServletException     if a servlet error occurs
     * @description                 delete the bookmark tag
     */
    protected void actionDeleteTag(HttpServletRequest request, HttpServletResponse response, User user)
            throws IOException, ServletException {
        String bookmarkId = request.getParameter("bookmarkId");
        String tag = request.getParameter("deleteTag");
        if (tag != null && bookmarkId != null) {
            if (!bookmarkId.isEmpty()) {
                WikittyProxy proxy = BowProxy.getInstance();
                Bookmark bookmark = proxy.restore(Bookmark.class, bookmarkId);
                if (bookmark != null) {
                    bookmark.removeTags(tag);
                    proxy.store(bookmark);
                }
            }
        }
        redirectToTheGoodPage(request, response);
    }


    /* @param request               servlet request
     * @param response              servlet response
     * @param user                  User user
     * @param session               HttpSession session
     * @throws ServletException     if a servlet error occurs
     * @description                 delete the current token and generate an another
     */
    protected void actionGenerateToken(HttpServletRequest request, HttpServletResponse response, User user, HttpSession session)
            throws IOException, ServletException, NoSuchAlgorithmException {

        WikittyProxy proxy = BowProxy.getInstance();
        Criteria criteria = Search.query().eq(Token.FQ_FIELD_EMAIL, user.getEmail()).criteria();
        Token currentToken = proxy.findByCriteria(Token.class, criteria);
        if (currentToken != null) {
            String id = currentToken.getWikittyId();
            proxy.delete(id);
        }
        TokenActions tokenActions = (TokenActions) session.getAttribute("tokenActions");
        if (tokenActions != null) {
            String token = tokenActions.generateToken(); // Generate an encoding MD5 token
            criteria = Search.query().eq(Token.FQ_FIELD_TOKEN, token).criteria();
            Token oldToken = proxy.findByCriteria(Token.class, criteria);
            if (oldToken == null) { // Check if the token already exists
                TokenImpl newToken = new TokenImpl();
                newToken.setToken(token);
                newToken.setEmail(user.getEmail());
                proxy.store((Token) newToken); // If the token doesn't exist, it is stored
                tokenActions.setPermanentToken(token);
            } else {
                tokenActions.setPermanentToken("");
            }
        }
    }

    protected void actionModifyBookmark(HttpServletRequest request, HttpServletResponse response, User user)
            throws IOException {
        String link = request.getParameter("url"); // bowServletUrl of the website
        String name = request.getParameter("name"); // website name
        String alias = request.getParameter("alias");
        String tags = request.getParameter("tags");
        String id = request.getParameter("bookmarkId");
        WikittyProxy proxy = BowProxy.getInstance();
        Bookmark bookmark = proxy.restore(Bookmark.class, id);
        if (bookmark != null) {
            Criteria criteria = Search.query().eq(Bookmark.FQ_FIELD_ALIAS, alias).criteria();
            if (proxy.findByCriteria(Bookmark.class, criteria) != null) {
                alias = bookmark.getAlias();
            }
            BookmarkActions.updateBookmark(bookmark, name, link, tags, alias);
            proxy.store(bookmark);
        }
        redirectToTheGoodPage(request, response);
    }

    /* @param request               servlet request
     * @param response              servlet response
     * @param user                  User user
     * @throws ServletException     if a servlet error occurs
     * @description                 add a new bookmark
     */
    protected void actionAddUrl(HttpServletRequest request, HttpServletResponse response, User user)
            throws IOException, ServletException {
        addUrl(request, user);
        redirectToTheGoodPage(request, response);
    }

    protected void addUrl(HttpServletRequest request, User user) {
        String link = request.getParameter("url"); // bowServletUrl of the website
        String name = request.getParameter("name"); // website name
        String alias = request.getParameter("alias");
        WikittyProxy proxy = BowProxy.getInstance();
        Bookmark bookmark = null;
        if (name != null) {
            String tags = request.getParameter("tags"); // tags
            if (alias != null && !alias.isEmpty()) {
                Criteria criteria = Search.query().eq(Bookmark.FQ_FIELD_ALIAS, alias).criteria();
                if (proxy.findByCriteria(Bookmark.class, criteria) != null) {
                    alias = "";
                }
            }
            bookmark = BookmarkActions.createBookmark(link, name, tags, user, alias, null);
        } else { // this part is for the bookmark addition by script
            String nameAndTags = request.getParameter("nameAndTags");
            bookmark = BookmarkActions.createBookmark(link, nameAndTags, user);
        }
        if (bookmark != null) {
            proxy.store(bookmark); // store the bookmark if all is Ok
            if (log.isDebugEnabled()) {
                log.debug("Adding URL");
            }
        }
    }
    /* @param token                 String token
     * @param session               HttpSession session
     * @return                      User
     * @description                 check if the token is valid and return the
     *                              token owner
     */

    protected User checkToken(String token, HttpSession session) {
        if (checkTemporaryToken(token, session) == true) {
            User user = (User) session.getAttribute("user");
            return user;
        }
        User user = checkPermanentToken(token);
        return user;
    }

    /* @param token         String which contains the MD5 encoding token
     * @return null         the token doesn't exist
     * @return User         the token owner
     */
    protected User checkPermanentToken(String token) {
        if (token != null) {
            WikittyProxy proxy = BowProxy.getInstance();
            Criteria criteria = Search.query().eq(Token.FQ_FIELD_TOKEN, token).criteria();
            Token DbToken = proxy.findByCriteria(Token.class, criteria);
            if (DbToken != null) {  // check if the token exists
                String userEmail = DbToken.getEmail(); // the token owner user name (email)
                criteria = Search.query().eq(User.FQ_FIELD_EMAIL, userEmail).criteria(); // retrieve user by token
                return proxy.findByCriteria(User.class, criteria);
            }
        }
        return null;
    }

    /* @param token         String which contains the MD5 encoding token
     * @return null         the token doesn't exist
     * @return User         the token owner
     */
    protected boolean checkTemporaryToken(String token, HttpSession session) {
        TokenActions tokenActions = (TokenActions) session.getAttribute("tokenActions");
        if (tokenActions != null) {
            String temporaryToken = tokenActions.getTemporaryToken();
            if (temporaryToken != null) {
                if (temporaryToken.equals(token)) {
                    return true;
                }
            }
        }
        return false;
    }

    /* @param request               servlet request
     * @param response              servlet response
     * @param user                  User user
     * @throws ServletException     if a servlet error occurs
     * @description                 set the bookmarks and the tagCloud by search
     *                              or tag addition
     */
    protected void actionSearch(HttpServletRequest request, User user)
            throws IOException, ServletException {
        String searchLine = request.getParameter("searchLine");
        String tag = request.getParameter("addTag");
        if (tag != null && !tag.isEmpty()) {
            if (searchLine == null || searchLine.isEmpty()) {
                searchLine = tag;
            } else {
                searchLine += " " + tag;
            }
        }
        WikittyProxy proxy = BowProxy.getInstance();
        Criteria criteria = getBookmarkListCriteriaByUser(user, searchLine);
        criteria = criteria.addSortDescending(Bookmark.FQ_FIELD_CLICK);
        PagedResult result = proxy.findAllByCriteria(Bookmark.class, criteria); // select all bookmarks by user
        BookmarkActions bookmarkActions = createBookmarkActions(request, result, searchLine);
        request.setAttribute("bookmarkActions", bookmarkActions);
    }

    /* @param request               servlet request
     * @param result                PageResult result
     * @param searchLine            String searchLine
     * @return bookmarkActions      the bookmarkAction
     * @description                 create the tagCLoud by research type
     */
    protected BookmarkActions createBookmarkActions(HttpServletRequest request, PagedResult result, String searchLine) {
        String fullText = request.getParameter("fullTextLine");
        BookmarkActions bookmarkActions = new BookmarkActions();
        List<Bookmark> bookList = result.getAll();
        if (bookList != null) {
            bookmarkActions.setBookmarks(bookList);
        }
        if (fullText == null || fullText.isEmpty()) {
            if (searchLine != null && searchLine.isEmpty()) {
                bookmarkActions.emptySearchline();
            } else {
                bookmarkActions.addTags(searchLine); // add the new tags
            }
        }
        List<FacetTopic> topics = result.getTopic(Bookmark.FQ_FIELD_TAGS);
        bookmarkActions.createTagCloud(topics);
        return bookmarkActions;
    }

    /* @param request               servlet request
     * @param response              servlet response
     * @param session               HttpSession session
     * @throws ServletException     if a servlet error occurs
     * @description                 check if the registration is correct or not
     */
    protected void actionRegister(HttpServletRequest request, HttpServletResponse response, HttpSession session)
            throws IOException, ServletException, NoSuchAlgorithmException, AddressException, MessagingException {
        String email = request.getParameter("email");
        String password = request.getParameter("password");
        String md5 = StringUtil.encodeMD5(password);
        if (this.checkRegister(email, md5)) { // check if all is well
            String error = "Email and password must be correctly filled";
            request.setAttribute("msgError", error);
            request.getRequestDispatcher("register.jsp").forward(request, response);
        } else {
            WikittyProxy proxy = BowProxy.getInstance();
            UserImpl newUser = new UserImpl();
            newUser.setPassword(md5);
            newUser.setEmail(email);
            User login = proxy.store((User) newUser); // store the new user
            if (login == null) {
                request.getRequestDispatcher("error.jsp").forward(request, response);
            } else {
                session.setAttribute("user", login);
                Preference preference = proxy.restore(Preference.class, login.getWikittyId());
                session.setAttribute("preference", preference);
                initializeToken(session, login);
                actionHome(request, login);
                checkAdmin(email, session);
                sendMail(email, password);
                request.getRequestDispatcher("main.jsp").forward(request, response);
            }
        }
    }

    /* @param request               servlet request
     * @param response              servlet response
     * @param session               HttpSession session
     * @throws ServletException     if a servlet error occurs
     * @description                 check if the authentication is correct or not
     */
    protected void actionLogin(HttpServletRequest request, HttpServletResponse response, HttpSession session)
            throws IOException, ServletException, NoSuchAlgorithmException {
        String email = request.getParameter("email");
        String password = request.getParameter("password");
        password = StringUtil.encodeMD5(password);
        User login = this.checkLogin(email, password); // check if the user exists
        if (login != null) {
            WikittyProxy proxy = BowProxy.getInstance();
            session.setAttribute("user", login);
            Preference preference = proxy.restore(Preference.class, login.getWikittyId());
            session.setAttribute("preference", preference);
            initializeToken(session, login);
            checkAdmin(email, session);
            actionHome(request, login);
            request.getRequestDispatcher("main.jsp").forward(request, response);

        } else {
            String error = "Unknow email or incorrect password";
            request.setAttribute("msgError", error);
            request.getRequestDispatcher("login.jsp").forward(request, response);
        }
    }

    protected void checkAdmin(String login, HttpSession session) {
        String[] admins = BowConfig.getInstance().getAdmins();
        if (admins != null) {
            for (int count = 0; count < admins.length; ++count) {
                if (login.equals(admins[count])) {
                    session.setAttribute("admin", true);
                    return;
                }
            }
        }
        session.setAttribute("admin", false);
    }

    /* @param session               HttpSession session
     * @throws ServletException     if a servlet error occurs
     * @description                 retrieve tokens or create there if the user are just
     *                              registered
     */
    protected void initializeToken(HttpSession session, User login)
            throws NoSuchAlgorithmException {
        WikittyProxy proxy = BowProxy.getInstance();
        Criteria criteria = Search.query().eq(Token.FQ_FIELD_EMAIL, login.getEmail()).criteria();
        Token token = proxy.findByCriteria(Token.class, criteria);
        TokenActions tokenActions = new TokenActions();
        if (token == null) {
            token = new TokenImpl();
            String newToken = tokenActions.generateToken();
            token.setToken(newToken);
            token.setEmail(login.getEmail());
            proxy.store(token);
        }
        tokenActions.setPermanentToken(token.getToken());
        String temporaryToken = tokenActions.generateToken();
        tokenActions.setTemporaryToken(temporaryToken);
        session.setAttribute("tokenActions", tokenActions);
    }


    /* @param email         String which contains the user name (email)
     * @param password      String which contains the user password
     * @return null         the user doesn't exist
     * @return User         the user exists
     */
    protected User checkLogin(String email, String password) {
        if (email != null && password != null) {
            if (!email.isEmpty() && !password.isEmpty()) {
                WikittyProxy proxy = BowProxy.getInstance();

                Criteria criteria = Search.query().eq(User.FQ_FIELD_EMAIL, email).
                        eq(User.FQ_FIELD_PASSWORD, password).criteria();
                return proxy.findByCriteria(User.class, criteria); // retrieve user by user name (email) and password
            }
        }
        return null;
    }


    /* @param email         String which contains the user name (email)
     * @param password      String which contains the user password
     * @return bool         false if the user doesn't exists or true
     */
    protected boolean checkRegister(String email, String password) {
        if (email != null && password != null) {
            if (!email.isEmpty() && !password.isEmpty()) {
                WikittyProxy proxy = BowProxy.getInstance();

                Criteria criteria = Search.query().eq(User.FQ_FIELD_EMAIL, email).criteria(); // retrieve user by user name (email)
                if (proxy.findByCriteria(User.class, criteria) == null) {
                    return false;
                }
            }
        }
        return true;
    }

    /* @param request               servlet request
     * @param user                  User user
     * @description                 initialize all for the home page
     */
    protected void actionHome(HttpServletRequest request, User user) {
        WikittyProxy proxy = BowProxy.getInstance();
        Criteria criteria = getBookmarkListCriteriaByUser(user, null);
        if (criteria != null) {
            Criteria sortCriteria = criteria.addSortDescending(Bookmark.FQ_FIELD_CLICK);
            PagedResult result = proxy.findAllByCriteria(Bookmark.class, sortCriteria); // select all bookmarks by user
            sortCriteria = criteria.addSortDescending(Bookmark.FQ_FIELD_DATE).setEndIndex(10);
            List<Bookmark> lastBookmarks = proxy.findAllByCriteria(Bookmark.class, sortCriteria).getAll();
            BookmarkActions bookmarkActions = createBookmarkActions(request, result, null);
            bookmarkActions.setTagSearch(null);
            List<Bookmark> bookList = bookmarkActions.getBookmarks();
            if (bookList.size() > 10) {
                bookList = bookmarkActions.getBookmarks().subList(0, 10);
                bookmarkActions.setBookmarks(bookList);
            }
            if (lastBookmarks != null) {
                bookmarkActions.setLastBookmarks(lastBookmarks);
            }
            request.setAttribute("bookmarkActions", bookmarkActions);
        }
    }

    /* @param request               servlet request
     * @param response              servlet response
     * @param user                  User user
     * @throws ServletException     if a servlet error occurs
     * @description                 import bookmark for an user
     */
    protected void actionImportBookmarks(HttpServletRequest request, HttpServletResponse response, User user)
            throws IOException, FileUploadException, ParserException {
        // Check that we have a file upload request
        boolean isMultipart = ServletFileUpload.isMultipartContent(request);
        if (isMultipart == true) {
            // Create a factory for disk-based file items
            DiskFileItemFactory factory = new DiskFileItemFactory();
            // Create a new file upload handler
            ServletFileUpload upload = new ServletFileUpload(factory);
            // Process the uploaded items
            // Parse the request
            List<FileItem> items = upload.parseRequest(request);
            Iterator iter = items.iterator();
            while (iter.hasNext()) {
                FileItem item = (FileItem) iter.next();
                if (!item.isFormField()) {
                    WikittyProxy proxy = BowProxy.getInstance();
                    String content = item.getString();
                    Parser parser = new Parser(content);
                    NodeList list = parser.parse(null);
                    List<Bookmark> bookmarks = new ArrayList<Bookmark>();
                    parseHtmlToBookmarks(list, user, bookmarks, new ArrayList<String>());
                    proxy.store(bookmarks);
                }
            }
        }
        redirectToTheGoodPage(request, response);
    }

    /* @param list                  NodeList list
     * @param bookmarks             List<Bookmark> bookmarks
     * @param tagList               List<String> tagList
     * @throws ParserException      if a parser error occurs
     * @description                 parse the html by recursion and retrieve bookmarks
     */
    protected void parseHtmlToBookmarks(NodeList list, User user, List<Bookmark> bookmarks, List<String> tagList)
            throws ParserException {
        if (list != null) {
            boolean isFolder = false;
            SimpleNodeIterator it = list.elements();
            while (it.hasMoreNodes()) {
                Node node = it.nextNode();
                String plainText = node.toPlainTextString(); // the text between two heads ==> <toto>plainText</toto>
                String text = node.getText(); // the text in the head ==> <text></toto>
                if (text != null && text.startsWith("H3")) { // H3 = folder
                    if (plainText != null && !plainText.isEmpty()) {
                        tagList.add(plainText); // add the folder name to the tagList
                        isFolder = true;
                    }
                } else if (text != null && text.startsWith("A HREF")) { // HREF = new bookmarks
                    Bookmark bookmark = BookmarkActions.createBookmarkFromHtml(text, plainText, user);
                    BookmarkActions.addTagsToBookmark(tagList, bookmark);
                    if (bookmark != null) {
                        bookmarks.add(bookmark);
                    }
                }
                NodeList children = node.getChildren();
                if (children != null) {
                    parseHtmlToBookmarks(children, user, bookmarks, tagList); // if there is an under node = recursion
                }
            }
            if (isFolder == true) { // if we find a folder, we have to remove it
                int index = tagList.size() - 1;
                if (index > -1) {
                    tagList.remove(index);
                }
            }
        }
    }

    /* @param response              servlet response
     * @param user                  User user
     * @description                 export bookmarks
     */
    protected void actionExportBookmarks(HttpServletResponse response, User user)
            throws IOException {
        WikittyProxy proxy = BowProxy.getInstance();
        Criteria criteria = Search.query().eq(Bookmark.FQ_FIELD_EMAIL, user.getEmail()).criteria();
        List<Bookmark> bookmarks = proxy.findAllByCriteria(Bookmark.class, criteria).getAll();
        String export = BookmarkActions.getExportHtmlBookmark(bookmarks);
        byte[] buff = export.getBytes();
        ServletOutputStream op = response.getOutputStream();
        response.setContentType("application/octet-stream");
        response.setHeader("Content-Disposition", "attachment; filename=\"bookmarks.html\"");
        response.setContentLength(buff.length);
        op.write(buff, 0, buff.length);
        op.flush();
        op.close();
    }

    /* @param request               servlet request
     * @param response              servlet response
     * @throws ServletException     if a servlet error occurs
     * @description                 add an alias for one bookmark
     */
    protected void actionAddAlias(HttpServletRequest request, HttpServletResponse response)
            throws IOException {
        String alias = request.getParameter("alias");
        if (alias != null && !alias.isEmpty()) {
            WikittyProxy proxy = BowProxy.getInstance();
            Criteria criteria = Search.query().eq(Bookmark.FQ_FIELD_ALIAS, alias).criteria();
            List<Bookmark> bookmarks = proxy.findAllByCriteria(Bookmark.class, criteria).getAll();
            if (bookmarks == null || bookmarks.isEmpty()) {
                String id = request.getParameter("bookmarkId");
                if (id != null && !id.isEmpty()) {
                    Bookmark bookmark = proxy.restore(Bookmark.class, id);
                    if (bookmark != null) {
                        bookmark.setAlias(alias);
                        proxy.store(bookmark);
                    }
                }
            }
        }
        redirectToTheGoodPage(request, response);
    }

    /* @param request               servlet request
     * @param user                  User user
     * @throws ServletException     if a servlet error occurs
     * @description                 retrieve bookmark for the full text research
     */
    protected void actionFullText(HttpServletRequest request, User user) {
        String fullText = request.getParameter("fullTextLine");
        WikittyProxy proxy = BowProxy.getInstance();
        Criteria criteria = null;
        if (fullText != null && !fullText.isEmpty()) {
            criteria = Search.query().keyword(fullText).
                    eq(Bookmark.FQ_FIELD_EMAIL, user.getEmail()).criteria().
                    addFacetField(Bookmark.FQ_FIELD_TAGS);

        } else {
            criteria = getBookmarkListCriteriaByUser(user, null);
        }
        PagedResult result = proxy.findAllByCriteria(Bookmark.class, criteria);
        BookmarkActions bookmarkActions = createBookmarkActions(request, result, null);
        request.setAttribute("bookmarkActions", bookmarkActions);
    }

    /* @param request               servlet request
     * @param response              servlet response
     * @throws ServletException     if a servlet error occurs
     * @description                 choose the good page
     */
    protected void redirectToTheGoodPage(HttpServletRequest request, HttpServletResponse response)
            throws IOException {
        String searchLine = request.getParameter("searchLine");
        String fullText = request.getParameter("fullTextLine");
        if (fullText == null) {
            fullText = "";
        }
        if (searchLine != null) {
            response.sendRedirect("bow?action=search&searchLine=" + searchLine + "&fullTextLine=" + fullText);
        } else {
            response.sendRedirect("bow?action=home");
        }
    }

    /* @param user                  User user
     * @param searchLine            String searchLine
     * @description                 return the bookmark criteria by search
     */
    protected Criteria getBookmarkListCriteriaByUser(User user, String searchLine) {
        Criteria criteria = null;
        if (user != null) {
            if (searchLine != null && !searchLine.isEmpty()) {
                String[] words = searchLine.split("\\s+"); // put the tags in an array
                List<String> tags = new ArrayList<String>(Arrays.asList(words));
                criteria = Search.query().eq(Bookmark.FQ_FIELD_EMAIL, user.getEmail()).
                        eq(Bookmark.FQ_FIELD_TAGS, tags).criteria().addFacetField(Bookmark.FQ_FIELD_TAGS);
            } else {
                criteria = Search.query().eq(Bookmark.FQ_FIELD_EMAIL, user.getEmail()).
                        criteria().addFacetField(Bookmark.FQ_FIELD_TAGS);
            }
        }
        return criteria;
    }

    private void actionChangePreferences(HttpServletRequest request, HttpSession session, User user)
            throws NoSuchAlgorithmException, AddressException, MessagingException {

        WikittyProxy proxy = BowProxy.getInstance();

        Preference preference = (Preference) session.getAttribute("preference");
        // Retrieve Preference fields
        String colors = request.getParameter("colors");
        String tags = request.getParameter("tags");
        String bookmarks = request.getParameter("bookmarks");
        String searchEngineSuggestions = request.getParameter("searchEngineUrlSuggestions");
        String searchEngineResults = request.getParameter("searchEngineUrlResults");

        preference.setColors(colors);
        preference.setTags(Integer.valueOf(tags));
        preference.setBookmarks(Integer.valueOf(bookmarks));
        preference.setSearchEngineUrlSuggestions(searchEngineSuggestions);
        preference.setSearchEngineUrlResults(searchEngineResults);

        proxy.store(preference);

        User newUser = proxy.restore(User.class, user.getWikittyId());
        // Retrieve User fields
        String email = request.getParameter("email");
        String newPassword = request.getParameter("newPassword");
        String currentPassword = request.getParameter("currentPassword");
        String confirmNewPassword = request.getParameter("confirmNewPassword");

        if (email != null && !email.isEmpty()) {
            newUser.setEmail(email);
        }
        if (newPassword != null && confirmNewPassword != null && currentPassword != null) {
            if (!newPassword.isEmpty() && !confirmNewPassword.isEmpty() && !currentPassword.isEmpty()) {
                if (newPassword.equals(confirmNewPassword)) {
                    currentPassword = StringUtil.encodeMD5(currentPassword);
                    if (currentPassword.equals(newUser.getPassword())) {
                        String md5 = StringUtil.encodeMD5(newPassword);
                        newUser.setPassword(md5);
                    }
                }
            }
        }
        newUser = proxy.store(newUser);
        session.setAttribute("user", newUser);

        preference = proxy.restore(Preference.class, newUser.getWikittyId());
        session.setAttribute("preference", preference);
    }

    protected void sendMail(String email, String password)
            throws AddressException, MessagingException {

        BowConfig config = BowConfig.getInstance();
        String smtpServer = config.getSmtpServer();
        Properties properties = new Properties();
        properties.put("mail.smtp.host", smtpServer);

        Session session = Session.getDefaultInstance(properties, null);
        session.setDebug(true);

        Message msg = new MimeMessage(session);

        InternetAddress toAddress = new InternetAddress(email);
        msg.addRecipient(Message.RecipientType.TO, toAddress);

        String addressFrom = config.getAddressFrom();
        InternetAddress fromAddress = new InternetAddress(addressFrom);
        msg.setFrom(fromAddress);

        String messageSubject = "[Bow] New Password";
        msg.setSubject(messageSubject);

        String messageBody = "Hi,\n\n";
        messageBody += "Your new password: " + password + "\n";
        messageBody += "Your email: " + email + "\n\n";
        msg.setContent(messageBody, "text/plain");

        Transport.send(msg);
    }

    protected boolean passwordExists(String password) {
        WikittyProxy proxy = BowProxy.getInstance();
        Criteria criteria = Search.query().eq(User.FQ_FIELD_PASSWORD, password).criteria();
        List<User> users = proxy.findAllByCriteria(User.class, criteria).getAll();
        if (users != null && users.size() > 0) {
            return true;
        }
        return false;
    }

    protected void actionSendPassword(HttpServletRequest request, HttpServletResponse response)
            throws NoSuchAlgorithmException, AddressException, MessagingException, ServletException, IOException {
        String email = request.getParameter("email");
        if (email != null && !email.isEmpty()) {
            WikittyProxy proxy = BowProxy.getInstance();
            Criteria criteria = Search.query().eq(User.FQ_FIELD_EMAIL, email).criteria();
            User user = proxy.findByCriteria(User.class, criteria);
            if (user != null) {
                boolean bool = true;
                String password = "";
                String md5 = "";
                while (bool) {
                    password = RandomStringUtils.randomAlphanumeric(20);
                    md5 = StringUtil.encodeMD5(password);
                    bool = passwordExists(md5);
                }
                sendMail(email, password);
                user.setPassword(md5);
                proxy.store(user);
                request.getRequestDispatcher("login.jsp").forward(request, response);
            } else {
                request.setAttribute("msgError", "This email doesn't exist...");
                request.getRequestDispatcher("forgotPassword.jsp").forward(request, response);
            }
        } else {
            request.setAttribute("msgError", "Please enter an email");
            request.getRequestDispatcher("forgotPassword.jsp").forward(request, response);
        }
    }

    private void actionReIndexation(HttpServletRequest request, HttpSession session, User user) {
        WikittyProxy proxy = BowProxy.getInstance();
        proxy.getWikittyService().syncEngin(proxy.getSecurityToken());
    }
}
